<?php

/**
 * Ceon URI Mappings Manager Import Old Category/Product URIs Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMImportOldURIsCategoriesProducts.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the admin class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonURIMappingAdmin.php');


// {{{ CeonUMMImportOldURIsCategoriesProducts

/**
 * Handles the import of the store's existing URIs for categories (and optionally the products
 * within), including via AJAX calls.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMImportOldURIsCategoriesProducts extends CeonURIMappingAdmin
{
	// {{{ properties
	
	/**
	 * The output imported for this action.
	 *
	 * @var     string
	 * @access  protected
	 */
	var $_output;
	
	/**
	 * Maintains a list of any errors encountered.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_error_messages = array();
	
	/**
	 * Should AJAX be used for progress indicators?
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_use_ajax_progress_indicators = null;
	
	/**
	 * How often (in milliseconds) should AJAX progress indicators update their status? 
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_ajax_progress_refresh_sec = null;
	
	/**
	 * How often (in seconds) should standard progress indicators "update their status" (Resubmit)? 
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_standard_progress_refresh_sec = null;
	
	/**
	 * Variable tracks when the import process was started. 
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_import_start_timestamp = null;
	
	/**
	 * Whether or not debug output should be sent to a file in the cache folder, for tracking the progress of AJAX
	 * import of category/product URIs.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_debug_ajax_to_file = false;
	
	/**
	 * The name of the AJAX debug output file for the current import session.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_ajax_debug_file = null;
	
	/**
	 * The IDs of the categories which are to have their URIs imported.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_categories_to_import = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		$this->_loadConfig();
		
		// Simple processing of posted variables, carried across from an earlier version of the software. Simply
		// sets the variables as properties of this instance
		foreach ($_POST as $key => $value) {
			if (isset($value) && is_array($value)) continue;
			
			$this->_posted_data[$key] = $value;
		}
		
		// Is the software running in script/AJAX/IFrame Mode? If so, let the appropriate method take over 
		$this->_handleAutomatedProcess();
		
		// Run the actual functionality and build the output
		$this->importOldURIsForCategorySelection();
	}
	
	// }}}
	
	
	// {{{ _handleAutomatedProcess()
	
	/**
	 * Checks if this software is being run in script mode, or if an AJAX call has been made to the software.
	 * Carries out any operation requried for the script mode or handles any AJAX request accordingly and returns
	 * the appropriate XML. If running in automated mode, the runtime process will be ended once the appropriate
	 * method has output its data. If not in an automated process mode, this will simply return.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _handleAutomatedProcess()
	{
		if (isset($_GET['ajax']) || strpos($_SERVER['REQUEST_URI'], 'ajax=1') !== false) {
			$this->_handleCategoriesProductsAJAXRequest();
		}
		
		return;
	}
	
	// }}}
	
	
	// {{{ importOldURIsForCategorySelection()
	
	/**
	 * Handles the import of URIs for the subcategories and/or products in the selected category/categories.
	 *
	 * @access  public
	 * @return  string    The HTML/JavaScript source for the output of the current stage of the import procedure.
	 */
	public function importOldURIsForCategorySelection()
	{
		global $db, $languages, $num_languages;
		
		// Start the control timer
		$this->_import_start_timestamp = time();
		
		if ($this->_debug_ajax_to_file) {
			$_SESSION['ceon_umm_ajax_debug_file'] = (defined('DIR_FS_LOGS') ? DIR_FS_LOGS : DIR_FS_SQL_CACHE) . '/ceon_umm_debug-' .
				date('Ymd his', time()) . '.txt';
			
			$this->_ajax_debug_file = fopen($_SESSION['ceon_umm_ajax_debug_file'], 'a');
			
			fwrite($this->_ajax_debug_file, "Starting at " . date('Ymd His', time()) . "\n\n");
		}
		
		if (!isset($this->_posted_data['cats-to-import'])) {
			// Just starting import process. Initialise environment
			$content = $this->_initialiseCategoriesProductsImportEnvironment();
			
			if (is_string($content)) {
				// Couldn't initialise environment. Output response built
				$this->_output = $content;
				
				return;
			}
		} else {
			// Executing an import stage for output to the browser. Determine the stage and configure the
			// environment appropriately
			$this->_configureCategoriesProductsImportEnvironment();
		}
		
		$this->_importCategoriesProductsURIs();
		
		$content = $this->_buildCategoriesProductsImportPanel();
		
		if ($this->_use_ajax_progress_indicators == true) {
			// Only run the AJAX controller if there is work still to be done
			if ($this->_num_categories_inc > $this->_num_categories_processed ||
					($this->_import_product_uris && $this->_num_products_inc > $this->_num_products_processed)) {
				$content .= $this->_buildCategoriesProductsImportAJAXController();
			}
		}
		
		if ($this->_debug_ajax_to_file) {
			fwrite($this->_ajax_debug_file, "\n\nStopping at " . date('Ymd His', time()) . "\n\n");
			
			fclose($this->_ajax_debug_file);
		}
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _initialiseCategoriesProductsImportEnvironment()
	
	/**
	 * Sets up the import process' initial environment.
	 *
	 * @access  protected
	 * @return  true|string    True if the environment was initialised, HTML to be displayed within the Import Old
	 *                         URIs panel's IFrame if a problem occurred.
	 */
	protected function _initialiseCategoriesProductsImportEnvironment()
	{
		global $db, $languages, $num_languages;
		
		// Set up the environment for the import process
		$this->_categories_to_import = array();
		
		// Get the list of categories selected
		$selected_categories = array();
		
		/*foreach ($this->_posted_data['cat'] as $category_id) {
			$selected_categories[] = (int) $category_id;
		}*/
		
		// BEGIN TEMP Defaults for this script //////
		// Import all categories
		$categories_result = $db->Execute("
			SELECT
				c.categories_id
            FROM
				" . TABLE_CATEGORIES . " c
			LEFT JOIN
				" . TABLE_CATEGORIES_DESCRIPTION . " cd
            ON
				c.categories_id = cd.categories_id
            AND
				cd.language_id = " . (int) $_SESSION['languages_id'] . "
            WHERE
				c.parent_id = " . 0 . "
            ORDER BY
				c.sort_order, cd.categories_name");
		
		while (!$categories_result->EOF) {
			$selected_categories[] = (int) $categories_result->fields['categories_id'];
			
			$categories_result->MoveNext();
		}
		//$selected_categories = array(163, 164);
		$this->_posted_data['import-category-uris'] = 'inc-products';
		
		//$this->_posted_data['inc-disabled'] = true;
		
		// END TEMP //////
		
		if (count($selected_categories) == 0) {
			// No categories selected!
			$content = '<h1>' . TEXT_CATEGORY_PRODUCT_URIS_TITLE . '</h1>' . "\n";
			
			$content = '<p>' . TEXT_ERROR_NO_CATEGORIES_SELECTED_FOR_IMPORT . '</p>' . "\n";
			
			$params = zen_get_all_get_params();
			
			if ((int) $_GET['cat-id'] != 0) {
				$category_name_result = $db->Execute("
					SELECT
						cd.categories_name
					FROM
						" . TABLE_CATEGORIES_DESCRIPTION . " cd
					WHERE
						cd.categories_id = " . (int) $_GET['cat-id'] . "
					AND
						cd.language_id = " . (int) $_SESSION['languages_id'] . ";");
				
				if (!$category_name_result->EOF) {
					$category_name = $category_name_result->fields['categories_name'];
				}
			} else {
				$category_name = TEXT_ROOT_CATEGORY;
			}
			
			$back_up_link = zen_href_link(FILENAME_CEON_UMM_IMPORT_OLD_URIS, $params, 'NONSSL');
			
			$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/' .
				'ceon-umm-back-to.png') . ' <a href="' . $back_up_link . '">' .
				sprintf(TEXT_BACK_TO, htmlentities($category_name, ENT_COMPAT, CHARSET)) .
				'</a></p>' . "\n";
			
			return $content;
		}
		
		// Are product mappings to be imported or only category/subcategory mappings?
		$this->_import_product_uris = false;
		
		if ($this->_posted_data['import-category-uris'] == 'inc-products') {
			$this->_import_product_uris = true;
		}
		
		$this->_include_disabled_cats_prods = false;
		
		if (isset($this->_posted_data['inc-disabled'])) {
			$this->_include_disabled_cats_prods = true;
		}
		
		if (!$this->_include_disabled_cats_prods) {
			// Make sure only enabled categories are handled
			$disabled_categories = array();
			
			$selected_category_ids = implode(',', $selected_categories);
			
			$categories_disabled_result = $db->Execute("
				SELECT
					categories_id
				FROM
					" . TABLE_CATEGORIES . " c
				WHERE
					c.categories_id IN (" . $selected_category_ids . ")
				AND
					c.categories_status = '0';");
			
			while (!$categories_disabled_result->EOF) {
				$disabled_categories[] = $categories_disabled_result->fields['categories_id'];
				
				$categories_disabled_result->MoveNext();
			}
		}
		
		// Count the number of categories (and products if applicable) that are included in the import procedure to
		// be run
		$this->_num_categories_inc = 0;
		$this->_num_products_inc = 0;
		
		foreach ($selected_categories as $category_id) {
			
			if (!$this->_include_disabled_cats_prods && in_array($category_id, $disabled_categories)) {
				// Don't include this disabled category
				continue;
			}
			
			$this->_categories_to_import[] = $category_id;
			
			$this->_num_categories_inc++;
			
			$category_counts = $this->_getCategorySubcategoriesProductsCount($category_id,
				$this->_import_product_uris, $this->_include_disabled_cats_prods);
			
			$this->_num_categories_inc += $category_counts['num_subcategories'];
			
			if ($this->_import_product_uris) {
				// Get the number of products in the current category
				if ($this->_include_disabled_cats_prods) {
					$products_count_result = $db->Execute("
						SELECT
							count(*) as num_products
						FROM
							" . TABLE_PRODUCTS . " p
						WHERE
							p.master_categories_id = " . (int) $category_id . ";");
				} else {
					$products_count_result = $db->Execute("
						SELECT
							count(*) as num_products
						FROM
							" . TABLE_PRODUCTS . " p
						WHERE
							p.master_categories_id = " . (int) $category_id . "
						AND
							p.products_status = '1';");
				}
				
				if (!$products_count_result->EOF) {
					$this->_num_products_inc += $products_count_result->fields['num_products'];
				}
				
				// Add the number of products in the current category's subcategories (already counted above)
				$this->_num_products_inc += $category_counts['num_products'];
			}
		}
		
		// Only starting
		$this->_cur_category_pos = array();
		$this->_num_categories_processed = 0;
		$this->_num_category_uris_imported = 0;
		$this->_num_category_uris_already_imported = 0;
		$this->_num_category_uris_using_zc_uris = 0;
		
		if ($this->_import_product_uris) {
			$this->_cur_product_pos = 0;
			$this->_processing_cat_mixed_with_products = 0;
			$this->_num_products_processed = 0;
			$this->_num_product_uris_imported = 0;
			$this->_num_product_uris_already_imported = 0;
			$this->_num_product_uris_using_zc_uris = 0;
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _importCategoriesProductsURIs()
	
	/**
	 * Imports the existing URIs for the subcategories and/or products in the selected category/categories.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _importCategoriesProductsURIs()
	{
		$this->_category_uris_imported_messages = array();
		$this->_category_uris_using_zc_uris_messages = array();
		$this->_product_uris_imported_messages = array();
		$this->_product_uris_using_zc_uris_messages = array();
		
		// Establish the point in the process the current stage should begin
		$this->_importCategoriesProductsURIsForCategories($this->_categories_to_import, 0);
	}
	
	// }}}
	
	
	// {{{ _importCategoriesProductsURIsForCategories()
	
	/**
	 * Imports the existing URIs for the subcategories and/or products in the specified categories.
	 *
	 * @access  protected
	 * @param   array      $categories   An array of the IDs of the categories.
	 * @param   integer    $current_category_level   The current level, with respect to the current category
	 *                                               position array class property, that the array of categories is
	 *                                              for.
	 * @return  boolean   Whether or not the method completed before a timeout was encountered. Necessary to
	 *                    prevent resetting of position variable in the middle of the import process.
	 */
	protected function _importCategoriesProductsURIsForCategories($categories, $current_category_level)
	{
		global $db;
		
		$start_category_i = 0;
		
		// Identify the category to start import at
		if (isset($this->_cur_category_pos[$current_category_level])) {
			foreach ($categories as $category_id) {
				if ($category_id == $this->_cur_category_pos[$current_category_level]) {
					// Have found the index for the category in which to start/resume import
					break;
				}
				
				// Haven't yet found the category in which to start/resume import
				$start_category_i++;
			}
		}
		
		if ($this->_debug_ajax_to_file) {
			fwrite($this->_ajax_debug_file, 'Cur Cat Pos: ' . implode (', ', $this->_cur_category_pos) . "\n");
			
			fwrite($this->_ajax_debug_file, 'Cur Cat Level: ' . $current_category_level . "\n");
			fwrite($this->_ajax_debug_file, 'Start Cat I: ' . $start_category_i . "\n");
		}
		
		$num_categories = sizeof($categories);
		
		for ($cat_i = $start_category_i; $cat_i < $num_categories; $cat_i++) {
			// Don't carry out import if the time limit has passed
			if ($this->_importTimeHasExpired()) {
				return false;
			}
			
			if (!isset($this->_cur_category_pos[$current_category_level])) {
				// This is the first time a category has been processed at this level for the parent category (if
				// any)
				if ($this->_processing_cat_mixed_with_products == 0) {
					$this->_createHistoricalURIForCategory($categories[$cat_i]);
				}
				
				$this->_cur_category_pos[$current_category_level] = $categories[$cat_i];
				
				if ($this->_debug_ajax_to_file) {
					fwrite($this->_ajax_debug_file, 'Cur Cat Pos Moved to: ' . $categories[$cat_i] . "\n");
					
					fwrite($this->_ajax_debug_file, 'Cur Cat Pos Updated is: ' .
						implode (', ', $this->_cur_category_pos) . "\n");
				}	
			} else if ($cat_i > 0 &&
					$this->_cur_category_pos[$current_category_level] == $categories[$cat_i - 1]) {
				// Have moved on to next category, all mappings will have been imported for previous category.
				// Reset position
				if ($this->_processing_cat_mixed_with_products == 0) {
					$this->_createHistoricalURIForCategory($categories[$cat_i]);
				}
				
				$temp = array();
				
				for ($temp_i = 0; $temp_i < $current_category_level; $temp_i++) {
					$temp[] = $this->_cur_category_pos[$temp_i];
				}
				
				$this->_cur_category_pos = $temp;
				
				$this->_cur_category_pos[$current_category_level] = $categories[$cat_i];
				
				if ($this->_debug_ajax_to_file) {
					fwrite($this->_ajax_debug_file, 'Cur Cat Pos Moved to: ' . $categories[$cat_i] . "\n");
					
					fwrite($this->_ajax_debug_file, 'Cur Cat Pos Updated is: ' .
						implode (', ', $this->_cur_category_pos) . "\n");
				}
				
			} else {
				// Must be resuming the processing of a subcategory of this category, so mapping has already been
				// processed for this category
				
				if ($this->_debug_ajax_to_file) {
					fwrite($this->_ajax_debug_file, 'Resuming processing of subcategory ' . $categories[$cat_i] .
						"\n");
				}
			}
			
			if ($this->_include_disabled_cats_prods) {
				$subcategories_result = $db->Execute("
					SELECT
						c.categories_id
					FROM
						" . TABLE_CATEGORIES . " c
					WHERE
						c.parent_id = " . (int) $categories[$cat_i] . "
					ORDER BY
						c.categories_id;");
			} else {
				$subcategories_result = $db->Execute("
					SELECT
						c.categories_id
					FROM
						" . TABLE_CATEGORIES . " c
					WHERE
						c.parent_id = " . (int) $categories[$cat_i] . "
					AND
						c.categories_status = '1'
					ORDER BY
						c.categories_id;");
			}
			
			$subcategories = array();
			
			while (!$subcategories_result->EOF) {
				$subcategories[] = $subcategories_result->fields['categories_id'];
				
				$subcategories_result->MoveNext();
			}
			
			if ($this->_debug_ajax_to_file) {
				fwrite($this->_ajax_debug_file, 'Subcategories for this category: ' .
					implode(', ', $subcategories) . "\n");
			}
			
			if (sizeof($subcategories) == 0 && $this->_processing_cat_mixed_with_products == 0) {
				// Import URIs for any products which are in this category
				$completed_without_timeout = $this->_importProductsURIsForCategory($categories[$cat_i]);
				
				if (!$completed_without_timeout) {
					return false;
				}
			} else if (sizeof($subcategories) > 0) {
				$completed_without_timeout = $this->_importCategoriesProductsURIsForCategories($subcategories,
					$current_category_level + 1);
				
				if (!$completed_without_timeout) {
					return false;
				}
			}
		}
		
		// Getting to this stage means that all subcategories for this category have been processed. Set pointer
		// appropriately
		$temp = array();
		
		for ($temp_i = 0; $temp_i < $current_category_level; $temp_i++) {
			$temp[] = $this->_cur_category_pos[$temp_i];
		}
		
		$this->_cur_category_pos = $temp;
		
		if ($this->_debug_ajax_to_file) {
			fwrite($this->_ajax_debug_file, 'Cur Cat Pos Updated At end of Category Run is: ' .
				implode (', ', $this->_cur_category_pos) . "\n");
		}
		
		if ($current_category_level > 0) {
			// Import URIs for any products which are in this category
			$this->_processing_cat_mixed_with_products = 1;
			
			$completed_without_timeout = $this->_importProductsURIsForCategory(
				$this->_cur_category_pos[$current_category_level - 1]);
			
			if (!$completed_without_timeout) {
				return false;
			}
			
			$this->_processing_cat_mixed_with_products = 0;
		}
		
		// Didn't time out during processing
		return true;
	}
	
	// }}}
	
	
	// {{{ _importProductsURIsForCategory()
	
	/**
	 * Imports the existing URIs for the products in the specified category.
	 *
	 * @access  protected
	 * @param   integer    $category_id   The ID of the category which contains the products.
	 * @return  boolean   Whether or not the method completed before a timeout was encountered. Necessary to
	 *                    prevent resetting of position variable in the middle of the import process.
	 */
	protected function _importProductsURIsForCategory($category_id)
	{
		global $db;
		
		if ($this->_import_product_uris) {
			if ($this->_include_disabled_cats_prods) {
				$products_result = $db->Execute("
					SELECT
						p.products_id,
						p.products_type
					FROM
						" . TABLE_PRODUCTS . " p
					WHERE
						p.master_categories_id = " . (int) $category_id . "
					ORDER BY
						p.products_id;");
			} else {
				$products_result = $db->Execute("
					SELECT
						p.products_id,
						p.products_type
					FROM
						" . TABLE_PRODUCTS . " p
					WHERE
						p.master_categories_id = " . (int) $category_id . "
					AND
						p.products_status = '1'
					ORDER BY
						p.products_id;");
			}
			
			if ($products_result->EOF) {
				// Category has no products! Obviously product position pointer can't refer to this category so
				// don't reset it
				return true;
			}
			
			// Skip past any products which have already been processed in a process which is being resumed
			$skipped_imported_products = true;
			
			if ($this->_debug_ajax_to_file) {
				fwrite($this->_ajax_debug_file, 'Cur Prod Pos Is: ' .
					$this->_cur_product_pos . "\n");
			}
			
			if ($this->_cur_product_pos != 0) {
				$skipped_imported_products = false;
			}
			
			while (!$products_result->EOF) {
				$product_id = $products_result->fields['products_id'];
				
				if (!$skipped_imported_products) {
					// If the current product ID hasn't been matched yet, then given the ordering, this product
					// must already have been processed
					if ($product_id == $this->_cur_product_pos) {
						// This was the last product to be processed
						$skipped_imported_products = true;
					}
					
					$products_result->MoveNext();
					
					continue;
				}
				
				$this->_createHistoricalURIForProduct($product_id, $products_result->fields['products_type'],
					(int) $category_id);
				
				$this->_cur_product_pos = $products_result->fields['products_id'];
				
				if ($this->_debug_ajax_to_file) {
					fwrite($this->_ajax_debug_file, 'Cur Prod Pos Moved to: ' . $this->_cur_product_pos . "\n");
				}
				
				if ($this->_importTimeHasExpired()) {
					return false;
				}
				
				$products_result->MoveNext();
			}
			
			// Reaching this point without a timeout means that all products for this category have been processed.
			// Reset pointer position
			$this->_cur_product_pos = 0;
			
			return true;
		}
	}
	
	// }}}
	
	
	// {{{ _importTimeHasExpired()
	
	/**
	 * Simply checks if the current stage of the import process has still got time to run.
	 *
	 * @access  protected
	 * @return  boolean   True if the current stage has run out of time, false otherwise.
	 */
	protected function _importTimeHasExpired()
	{
		if ($this->_use_ajax_progress_indicators) {
			return (time() > ($this->_import_start_timestamp + $this->_ajax_progress_refresh_sec));
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _createHistoricalURIForCategory()
	
	/**
	 * Imports an existing, "old" URI for a category by creating a historical URI for it. Information about the
	 * status of the import is stored in the appropriate reporting array and the respective counter variable is
	 * updated.
	 *
	 * @access  protected
	 * @param   integer   $id   The ID of the category to have its URI imported.
	 * @return  none
	 */
	protected function _createHistoricalURIForCategory($id)
	{
		global $db, $ceon_uri_mapping_demo;
		
		$this->_num_categories_processed++;
		
		$main_page = FILENAME_DEFAULT;
		
		// Get the current URI for this category
		$uri = zen_href_link($main_page, 'cPath=' . $id);
		
		// Check if this is a standard Zen Cart dynamic URI (assumed to be if main_page is present)
		if (strpos(strtolower($uri), 'main_page=' . strtolower($main_page)) !== false) {
			// Build a link to the category's edit page so the user can change the dynamic URI immediately
			$category_info_result = $db->Execute("
				SELECT
					c.parent_id,
					cd.categories_name
				FROM
					" . TABLE_CATEGORIES . " c 
				LEFT JOIN
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				ON
					c.categories_id = cd.categories_id
				AND
					cd.language_id = " . (int) $_SESSION['languages_id'] . "
				WHERE
					c.categories_id = " . $id . "
				;");
			
			$parent_category_id = $category_info_result->fields['parent_id'];
			
			$edit_link = $_GET['admin-dir-path'] . 'categories.php';
			
			$edit_link .= '?cPath=' . $parent_category_id . '&cID=' . $id . '&action=edit_category';
			
			$edit_link_title = TEXT_EDIT_CATEGORY;
			
			$this->_category_uris_using_zc_uris_messages[] = array(
				'message' => TEXT_CATEGORY_URI_USES_ZC_URI,
				'link' => HTTP_SERVER . $uri,
				'link_title' => $category_info_result->fields['categories_name'],
				'edit_link' => $edit_link,
				'edit_link_title' => $edit_link_title
				);
			
			$this->_num_category_uris_using_zc_uris++;
			
			return;
		}
		
		// Remove the protocol and domain name, getting the actual URI
		$uri = preg_replace('|^http[^/]+//[^/]+|i', '', $uri);
		
		// Remove any session ID
		if (strpos($uri, '?' . zen_session_name() . '=') !== false) {
			$uri = preg_replace('|\?' . zen_session_name() . '=[^&]+|', '', $uri);
		} else if (strpos($uri, '&' . zen_session_name() . '=') !== false) {
			$uri = preg_replace('|&' . zen_session_name() . '=[^&]+|', '', $uri);
		}
		
		
		// @TODO Let user choose language for import in admin
		$language_id = (int) $_SESSION['languages_id'];
		
		//$language_name = ucwords($languages[$i]['name']);
		
		// Check if this category has already been imported
		$columns_to_retrieve = array(
			'uri'
			);
		
		$selections = array(
			'uri' => zen_db_prepare_input($uri),
			'main_page' => zen_db_prepare_input($main_page),
			'associated_db_id' => (int) $id,
			'query_string_parameters' => null,
			'language_id' => $language_id
			);
		
		$existing_uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
		
		if (!$existing_uri_mapping_result->EOF) {
			if ($this->_debug_ajax_to_file) {
				fwrite($this->_ajax_debug_file, "Cat Already Imported: " . $id . ' Mapping: ' .
					$uri_mapping . "\n");
			}
			
			$this->_num_category_uris_already_imported++;
			
			return;
		}
		
		// URI not yet imported, create historical URI mapping for it now
		$uri_quoted = "'" . zen_db_input(zen_db_prepare_input($uri)) . "'";
		
		$language_id_quoted = "'" . (int) $language_id . "'";
		
		$current_uri_quoted = "'0'";
		
		$main_page_quoted = "'" . zen_db_input(zen_db_prepare_input($main_page)) . "'";
		
		$query_string_parameters_quoted = 'NULL';
		
		$associated_db_id_quoted = "'" . (int) $id . "'";
		
		$date_added_quoted = "'" . date('Y-m-d H:i:s') . "'";
		
		$sql = "
			INSERT INTO
				" . TABLE_CEON_URI_MAPPINGS . "
				(
				uri,
				language_id,
				current_uri,
				main_page,
				query_string_parameters,
				associated_db_id,
				date_added
				)
			VALUES
				(
				" . $uri_quoted . ",
				" . $language_id_quoted . ",
				" . $current_uri_quoted . ",
				" . $main_page_quoted . ",
				" . $query_string_parameters_quoted . ",
				" . $associated_db_id_quoted . ",
				" . $date_added_quoted . "
				);";
		
		if (!isset($ceon_uri_mapping_demo) || $ceon_uri_mapping_demo == false) {
			$db->Execute($sql);
		}
		
		$success_message = TEXT_CATEGORY_URI_IMPORTED;
		
		// Build a link to the category's edit page so the user can change the imported URI immediately
		$category_info_result = $db->Execute("
			SELECT
				c.parent_id,
				cd.categories_name
			FROM
				" . TABLE_CATEGORIES . " c 
			LEFT JOIN
				" . TABLE_CATEGORIES_DESCRIPTION . " cd
			ON
				c.categories_id = cd.categories_id
			AND
				cd.language_id = " . (int) $_SESSION['languages_id'] . "
			WHERE
				c.categories_id = " . $id . "
			;");
		
		$parent_category_id = $category_info_result->fields['parent_id'];
		
		$edit_link = $_GET['admin-dir-path'] . 'categories.php';
		
		$edit_link .= '?cPath=' . $parent_category_id . '&cID=' . $id . '&action=edit_category';
		
		$edit_link_title = TEXT_EDIT_CATEGORY . ' ' . $id . ': ' .
			$category_info_result->fields['categories_name'];
		
		$this->_category_uris_imported_messages[] = array(
			'message' => $success_message,
			'link' => HTTP_SERVER . $uri,
			'link_title' => $uri,
			'edit_link' => $edit_link,
			'edit_link_title' => $edit_link_title
			);
		
		$this->_num_category_uris_imported++;
	}
	
	// }}}
	
	
	// {{{ _createHistoricalURIForProduct()
	
	/**
	 * Imports an existing, "old" URI for a product by creating a historical URI for it. Information about the
	 * status of the import is stored in the appropriate reporting array and the respective counter variable is
	 * updated.
	 *
	 * @access  protected
	 * @param   integer   $id   The ID of the product to have its URI imported.
	 * @param   integer   $product_type_id    The product's type ID.
	 * @param   integer   $master_categories_id   The product's master category's ID.
	 * @return  none
	 */
	protected function _createHistoricalURIForProduct($id, $product_type_id, $master_categories_id)
	{
		global $db, $ceon_uri_mapping_demo;
		
		$this->_num_products_processed++;
		
		$sql = "
			SELECT
				type_handler
			FROM
				" . TABLE_PRODUCT_TYPES . "
			WHERE
				type_id = '" . (int) $product_type_id . "'";
		
		$product_type_handler_result = $db->Execute($sql);
		
		$product_type_handler = $product_type_handler_result->fields['type_handler'];
		
		$main_page = $product_type_handler . '_info';
		
		// Get the current URI for this product
		$uri = zen_href_link($main_page, 'products_id=' . $id);
		
		// Check if this is a standard Zen Cart dynamic URI (assumed to be if main_page is present)
		if (strpos(strtolower($uri), 'main_page=' . strtolower($main_page)) !== false) {
			$product_info_result = $db->Execute("
				SELECT
					pd.products_name
				FROM
					" . TABLE_PRODUCTS_DESCRIPTION . " pd
				WHERE
					pd.language_id = '" . (int) $_SESSION['languages_id'] . "'
				AND
					pd.products_id = '" . (int)$id . "'
				;");
			
//			$parent_category_id = $category_info_result->fields['parent_id'];
			
			$edit_link = $_GET['admin-dir-path'] . 'product.php';
			
			$edit_link .= '?product_type=' . $product_type_id . '&pID=' . $id . '&cPath=' .
				$master_categories_id . '&action=new_product';
			
			$edit_link_title = TEXT_EDIT_PRODUCT . ' ' . $id . ': ' .
				$product_info_result->fields['products_name'];
			
			$this->_product_uris_using_zc_uris_messages[] = array(
				'message' => TEXT_PRODUCT_URI_USES_ZC_URI,
				'link' => HTTP_SERVER . $uri,
				'link_title' => $uri,
				'edit_link' => $edit_link,
				'edit_link_title' => $edit_link_title
				);
			
			$this->_num_product_uris_using_zc_uris++;
			
			return;
		}
		
		// Remove the protocol and domain name, getting the actual URI
		$uri = preg_replace('|^http[^/]+//[^/]+|i', '', $uri);
		
		// Remove any session ID
		if (strpos($uri, '?' . zen_session_name() . '=') !== false) {
			$uri = preg_replace('|\?' . zen_session_name() . '=[^&]+|', '', $uri);
		} else if (strpos($uri, '&' . zen_session_name() . '=') !== false) {
			$uri = preg_replace('|&' . zen_session_name() . '=[^&]+|', '', $uri);
		}
		
		// @TODO Let user choose language for import in admin
		$language_id = (int) $_SESSION['languages_id'];
		
		//$language_name = ucwords($languages[$i]['name']);
		
		
		// Check if this product has already been imported
		$columns_to_retrieve = array(
			'uri'
			);
		
		$selections = array(
			'uri' => zen_db_prepare_input($uri),
			'main_page' => zen_db_prepare_input($main_page),
			'associated_db_id' => (int) $id,
			'query_string_parameters' => null,
			'language_id' => $language_id
			);
		
		$existing_uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
		
		if (!$existing_uri_mapping_result->EOF) {
			if ($this->_debug_ajax_to_file) {
				fwrite($this->_ajax_debug_file, "Product Already Imported: " . $id . ' Mapping: ' . $uri_mapping .
					"\n");
			}
			
			$this->_num_product_uris_already_imported++;
			
			return;
		}
		
		// URI not yet imported, create historical URI mapping for it now
		$uri_quoted = "'" . zen_db_input(zen_db_prepare_input($uri)) . "'";
		
		$language_id_quoted = "'" . (int) $language_id . "'";
		
		$current_uri_quoted = "'0'";
		
		$main_page_quoted = "'" . zen_db_input(zen_db_prepare_input($main_page)) . "'";
		
		$query_string_parameters_quoted = 'NULL';
		
		$associated_db_id_quoted = "'" . (int) $id . "'";
		
		$date_added_quoted = "'" . date('Y-m-d H:i:s') . "'";
		
		$sql = "
			INSERT INTO
				" . TABLE_CEON_URI_MAPPINGS . "
				(
				uri,
				language_id,
				current_uri,
				main_page,
				query_string_parameters,
				associated_db_id,
				date_added
				)
			VALUES
				(
				" . $uri_quoted . ",
				" . $language_id_quoted . ",
				" . $current_uri_quoted . ",
				" . $main_page_quoted . ",
				" . $query_string_parameters_quoted . ",
				" . $associated_db_id_quoted . ",
				" . $date_added_quoted . "
				);";
		
		if (!isset($ceon_uri_mapping_demo) || $ceon_uri_mapping_demo == false) {
			$db->Execute($sql);
		}
		
		$success_message = TEXT_PRODUCT_URI_IMPORTED;
		
		// Build a link to the product's edit page so the user can change the imported URI immediately
		$product_info_result = $db->Execute("
			SELECT
				pd.products_name
			FROM
				" . TABLE_PRODUCTS_DESCRIPTION . " pd
			WHERE
				pd.language_id = " . (int) $_SESSION['languages_id'] . "
			AND
				pd.products_id = " . $id . "
			;");
		
		$parent_category_id = $category_info_result->fields['parent_id'];
		
		$edit_link = $_GET['admin-dir-path'] . 'product.php';
		
		$edit_link .= '?product_type=' . $product_type_id . '&pID=' . $id . '&cPath=' . $master_categories_id .
			'&action=new_product';
		
		$edit_link_title = TEXT_EDIT_PRODUCT . ' ' . $id . ': ' . $product_info_result->fields['products_name'];
		
		$this->_product_uris_imported_messages[] = array(
			'message' => $success_message,
			'link' => HTTP_SERVER . $uri,
			'link_title' => $uri,
			'edit_link' => $edit_link,
			'edit_link_title' => $edit_link_title
			);
		
		$this->_num_product_uris_imported++;
	}
	
	// }}}
	
	
	// {{{ _buildCategoriesProductsImportPanel()
	
	/**
	 * Outputs the HTML/JavaScript source for the output of either the initial stage of the import process, or, if
	 * AJAX is not being used, the output current for the current stage of the process. If AJAX is being used, the
	 * output of subsequent stages is handled via the appropriate AJAX method.
	 *
	 * @access  protected
	 * @return  string    The HTML/JavaScript source for the output of the initial/current stage of the import
	 *                    procedure.
	 */
	protected function _buildCategoriesProductsImportPanel()
	{
		global $db;
		
		$content = '<h1 id="import-title">' . TEXT_CATEGORY_PRODUCT_URIS_TITLE . '</h1>' . "\n";
		
		$content .= '<p>' . TEXT_NUM_CATEGORIES_PROCESSED . '<span id="num-cats-processed">' .
			$this->_num_categories_processed . '</span> / <span id="num-cats">' . $this->_num_categories_inc .
			'</span></p>' . "\n";
		
		if ($this->_import_product_uris) {
			$content .= '<p>' . TEXT_NUM_PRODUCTS_PROCESSED . '<span id="num-prods-processed">' .
				$this->_num_products_processed . '</span> / <span id="num-prods">' . $this->_num_products_inc .
				'</span></p>' . "\n";
		}
		
		if ($this->_num_categories_inc == 0) {
			$percentage_completed_style_width = 96;
			
			$percentage_completed = 100;
			
		} else if ($this->_num_categories_processed == 0) {
			$percentage_completed_style_width = 0;
			
			$percentage_completed = 0;
			
		} else {	
			$percentage_completed_style_width =
				(96 / ($this->_num_categories_inc / $this->_num_categories_processed));
			
			$percentage_completed =
				round(100 / ($this->_num_categories_inc / $this->_num_categories_processed), 2);
		}
		
		$progress_bar = '<div class="ProgressBar">' .
			'<div class="ProgressBarIndicator" id="categories-progress" style="width: ' .
			$percentage_completed_style_width . '%">' . $percentage_completed . '%</div>' .
			'</div>' . "\n";
		
		$content .= $progress_bar;
		
		if ($this->_import_product_uris) {
			if ($this->_num_products_inc == 0) {
				$percentage_completed_style_width = 96;
				
				$percentage_completed = 100;
				
			} else if ($this->_num_products_processed == 0) {
				$percentage_completed_style_width = 0;
				
				$percentage_completed = 0;
				
			} else {	
				$percentage_completed_style_width =
					(96 / ($this->_num_products_inc / $this->_num_products_processed));
				
				$percentage_completed =
					round(100 / ($this->_num_products_inc / $this->_num_products_processed), 2);
			}
			
			$progress_bar = '<div class="ProgressBar">' .
				'<div class="ProgressBarIndicator" id="products-progress" style="width: ' .
				$percentage_completed_style_width . '%">' . $percentage_completed . '%</div>' .
				'</div>' . "\n";
			
			$content .= $progress_bar;
		}
		
		$content .= '<h3 class="ImportSectionTitle">' . TEXT_IMPORTED_CATEGORY_URIS . '</h3>' . "\n";
		
		$content .= '<p id="num-cats-imported">' . TEXT_NUM_CATEGORY_URIS_IMPORTED .
			' <span id="num-cats-imported-count">' . $this->_num_category_uris_imported . '</span></p>' . "\n";
		
		$content .= '<p id="num-cats-already-imported">' . TEXT_NUM_CATEGORY_URIS_ALREADY_IMPORTED .
			' <span id="num-cats-already-imported-count">' . $this->_num_category_uris_already_imported .
			'</span></p>' . "\n";
		
		if (sizeof($this->_category_uris_imported_messages) > 0) {
			// Output information about the mappings imported
			// @TODO currently this only outputs information for the current stage.. handling for carrying over
			// from previous stage is not yet implemented
			$content .= '<div id="categories-imported-listing">';
			
			foreach ($this->_category_uris_imported_messages as $message) {
				$content .= '<p>' . htmlspecialchars($message['message']) . ' <a href="' .
					htmlspecialchars($message['link']) . '" target="_blank">' .
					htmlspecialchars($message['link_title']) . '</a>';
				
				if (isset($message['edit_link'])) {
					$content .= ' - ' . ' <a href="' . htmlspecialchars($message['edit_link']) .
						'" target="_blank" class="EditLink">' . htmlspecialchars($message['edit_link_title']) .
						'</a>';
				}
				
				$content .= '<p>' . "\n";
			}
			
			$content .= '</div>' . "\n";
		}
		
		if (sizeof($this->_category_uris_using_zc_uris_messages) > 0) {
			$content .= '<p id="num-cats-using-zc-uris">';
		} else {
			$content .= '<p id="num-cats-using-zc-uris" style="display: none;">';
		}
		
		$content .= TEXT_NUM_CATEGORY_URIS_USING_ZC_URIS . ' <span id="num-cats-using-zc-uris-count">' .
			$this->_num_category_uris_using_zc_uris . '</span></p>' . "\n";
		
		if (sizeof($this->_category_uris_using_zc_uris_messages) > 0) {
			// Output information about the errors encountered
			// @TODO currently this only outputs information for the current stage.. handling for carrying over
			// from previous stage is not yet implemented
			$content .= '<div id="category-uris-using-zc-uris-listing">';
			
			foreach ($this->_category_uris_using_zc_uris_messages as $message) {
				$content .= '<p>' . htmlspecialchars($message['message']) . ' <a href="' .
					htmlspecialchars($message['link']) . '" target="_blank">' .
					htmlspecialchars($message['link_title']) . '</a>';
				
				if (isset($message['edit_link'])) {
					$content .= ' - ' . ' <a href="' . htmlspecialchars($message['edit_link']) .
						'" target="_blank" class="EditLink">' . htmlspecialchars($message['edit_link_title']) .
						'</a>';
				}
				
				$content .= '<p>' . "\n";
			}
			
			$content .= '</div>' . "\n";
		}
		
		if ($this->_import_product_uris) {
			$content .= '<h3 class="ImportSectionTitle">' . TEXT_IMPORTED_PRODUCT_URIS . '</h3>' . "\n";
			
			$content .= '<p id="num-prods-imported">' . TEXT_NUM_PRODUCT_URIS_IMPORTED .
				' <span id="num-prods-imported-count">' . $this->_num_product_uris_imported . '</span></p>' . "\n";
			
			$content .= '<p id="num-prods-already-imported">' . TEXT_NUM_PRODUCT_URIS_ALREADY_IMPORTED .
				' <span id="num-prods-already-imported-count">' . $this->_num_product_uris_already_imported .
				'</span></p>' . "\n";
			
			if (sizeof($this->_product_uris_imported_messages) > 0) {
				// Output information about the mappings imported
				// @TODO currently this only outputs information for the current stage.. handling for carrying over
				// from previous stage is not yet implemented
				$content .= '<div id="products-imported-listing">';
				
				foreach ($this->_product_uris_imported_messages as $message) {
					$content .= '<p>' . htmlspecialchars($message['message']) . ' <a href="' .
						htmlspecialchars($message['link']) . '" target="_blank">' .
						htmlspecialchars($message['link_title']) . '</a>';
					
					if (isset($message['edit_link'])) {
						$content .= ' - ' . ' <a href="' . htmlspecialchars($message['edit_link']) .
							'" target="_blank" class="EditLink">' . htmlspecialchars($message['edit_link_title']) .
							'</a>';
					}
					
					$content .= '<p>' . "\n";
				}
				
				$content .= '</div>' . "\n";
			}
			
			if (sizeof($this->_product_uris_using_zc_uris_messages) > 0) {
				$content .= '<p id="num-prods-using-zc-uris">';
			} else {
				$content .= '<p id="num-prods-using-zc-uris" style="display: none;">';
			}
			
			$content .= TEXT_NUM_PRODUCT_URIS_USING_ZC_URIS . ' <span id="num-prods-using-zc-uris-count">' .
				$this->_num_product_uris_using_zc_uris . '</span></p>' . "\n";
			
			if (sizeof($this->_product_uris_using_zc_uris_messages) > 0) {
				// Output information about the errors encountered
				// @TODO currently this only outputs information for the current stage.. handling for carrying over
				// from previous stage is not yet implemented
				$content .= '<div id="product-uris-using-zc-uris-listing">';
				
				foreach ($this->_product_uris_using_zc_uris_messages as $message) {
					$content .= '<p>' . htmlspecialchars($message['message']) . ' <a href="' .
						htmlspecialchars($message['link']) . '" target="_blank">' .
						htmlspecialchars($message['link_title']) . '</a>';
					
					if (isset($message['edit_link'])) {
						$content .= ' - ' . ' <a href="' . htmlspecialchars($message['edit_link']) .
							'" target="_blank" class="EditLink">' . htmlspecialchars($message['edit_link_title']) .
							'</a>';
					}
					
					$content .= '<p>' . "\n";
				}
				
				$content .= '</div>' . "\n";
			}
		}
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM_IMPORT_OLD_URIS, zen_get_all_get_params(), 'NONSSL', false,
			false, true);
		
		$content .= '<form action="' . $form_action . '" method="post" name="import-form" id="import-form">' .
			"\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Build form parameters to pass on to next stage
		$content .= zen_draw_hidden_field('import-product-uris', ($this->_import_product_uris ? 1 : 0));
		
		$content .= zen_draw_hidden_field('include-disabled', ($this->_include_disabled_cats_prods ? 1 : 0));
		
		$content .= zen_draw_hidden_field('cats-to-import', implode(',', $this->_categories_to_import));
		
		$content .= zen_draw_hidden_field('num-cats', $this->_num_categories_inc);
		
		$content .= zen_draw_hidden_field('num-cats-proc', $this->_num_categories_processed);
		
		$content .= zen_draw_hidden_field('num-cats-imp', $this->_num_category_uris_imported);
		
		$content .= zen_draw_hidden_field('num-cats-ia', $this->_num_category_uris_already_imported);
		
		$content .= zen_draw_hidden_field('num-cats-zc', $this->_num_category_uris_using_zc_uris);
		
		$content .= zen_draw_hidden_field('cur-cat-pos', implode(',', $this->_cur_category_pos));
		
		if ($this->_import_product_uris) {
			$content .= zen_draw_hidden_field('num-prods', $this->_num_products_inc);
			
			$content .= zen_draw_hidden_field('num-prods-proc', $this->_num_products_processed);
			
			$content .= zen_draw_hidden_field('num-prods-gen', $this->_num_product_uris_imported);
			
			$content .= zen_draw_hidden_field('num-prods-ia', $this->_num_product_uris_already_imported);
			
			$content .= zen_draw_hidden_field('num-prods-zc', $this->_num_product_uris_using_zc_uris);
			
			$content .= zen_draw_hidden_field('cur-prod-pos', $this->_cur_product_pos);
		}
		
		// Add button for browsers with JavaScript disabled
		$content .= "<noscript>\n";
		$content .= '<input type="submit">' . "\n";
		$content .= "</noscript>\n";
		
		if (!$this->_use_ajax_progress_indicators) {
			// Build autosubmitter
			$content .= '<script type="text/javascript">' . "\n";
			$content .= 'window.setTimeout(\'location.href="' . $_SERVER["PHP_SELF"] . "?start=$linenumber&fn=" .
				urlencode($curfilename) . "&foffset=$foffset&totalqueries=$totalqueries\";', 500 + " .
				$delaypersession . ");";
			$content .= '</script>' . "\n";
		}
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _buildCategoriesProductsImportAJAXController()
	
	/**
	 * Builds the HTML/JavaScript needed to run the import process as an AJAX process.
	 *
	 * @access  protected
	 * @return  string    The HTML/JavaScript source for the AJAX controller.
	 */
	protected function _buildCategoriesProductsImportAJAXController()
	{
		// Output the code necessary to have the script run using AJAX
		ob_start();
		
		$params = zen_get_all_get_params();
		
		$params .= 'ajax=1';
		
		$link = zen_href_link(FILENAME_CEON_UMM_IMPORT_OLD_URIS, $params, 'NONSSL', false, false,
			true);
		
	?>
<script type="text/javascript">			

// Build the parameters to be sent
<?php if ($this->_import_product_uris) { ?>
function buildParams(import_product_uris, include_disabled_cats_prods, cats_to_gen, num_cats, num_cats_proc,
	num_cats_imp, num_cats_ai, num_cats_zc, cur_cat_pos, num_prods, num_prods_proc, num_prods_imp, num_prods_ia,
	num_prods_zc, cur_prod_pos, proc_mixed_cat_prods)
{
	var params = buildMainParams(import_product_uris, include_disabled_cats_prods, cats_to_gen, num_cats,
		num_cats_proc, num_cats_imp, num_cats_ai, num_cats_zc, cur_cat_pos);
	
	params += buildProductParams(num_prods, num_prods_proc, num_prods_imp, num_prods_ia, num_prods_zc,
		cur_prod_pos, proc_mixed_cat_prods);
	
	return params;
}
<?php } else { ?>
function buildParams(import_product_uris, include_disabled_cats_prods, cats_to_gen, num_cats, num_cats_proc,
	num_cats_imp, num_cats_ai, num_cats_zc, cur_cat_pos)
{
	var params = buildMainParams(import_product_uris, include_disabled_cats_prods, cats_to_gen, num_cats,
		num_cats_proc, num_cats_imp, num_cats_ai, num_cats_zc, cur_cat_pos);
	
	return params;
}
<?php } ?>

function buildMainParams(import_product_uris, include_disabled_cats_prods, cats_to_gen, num_cats, num_cats_proc,
	num_cats_imp, num_cats_ai, num_cats_zc, cur_cat_pos)
{
	return 'import-product-uris=' + import_product_uris + '&include-disabled-cats-prods=' +
		include_disabled_cats_prods + '&cats-to-import=' + cats_to_gen + '&num-cats=' + num_cats +
		'&num-cats-proc=' + num_cats_proc + '&num-cats-imp=' + num_cats_imp + '&num-cats-ia=' + num_cats_ai +
		'&num-cats-zc=' + num_cats_zc + '&cur-cat-pos=' + cur_cat_pos;
}

function buildProductParams(num_prods, num_prods_proc, num_prods_imp, num_prods_ia, num_prods_zc, cur_prod_pos,
	proc_mixed_cat_prods)
{
	return '&num-prods=' + num_prods + '&num-prods-proc=' + num_prods_proc + '&num-prods-gen=' + num_prods_imp +
		'&num-prods-ia=' + num_prods_ia + '&num-prods-zc=' + num_prods_zc + '&cur-prod-pos=' + cur_prod_pos +
		'&proc-mixed-cat-prods=' + proc_mixed_cat_prods;
}

// Extracts text from XML element (itemname must be unique)
function getValueFromXML(xml, id) {
	if (xml.getElementsByTagName(id)[0] == undefined || xml.getElementsByTagName(id)[0].firstChild == undefined) {
		return null;
	}
	
	return xml.getElementsByTagName(id)[0].firstChild.data;
}

var http_request = false;

var uri = '<?php echo $link; ?>';

// Build parameters for first stage
var first_stage = true;

<?php if ($this->_import_product_uris) { ?>
var params_to_send = buildParams(<?php echo ($this->_import_product_uris ? 1 : 0); ?>,
	<?php echo ($this->_include_disabled_cats_prods ? 1 : 0); ?>,
	<?php echo "'" . implode(',', $this->_categories_to_import) . "'"; ?>,
	<?php echo $this->_num_categories_inc; ?>, <?php echo $this->_num_categories_processed; ?>,
	<?php echo $this->_num_category_uris_imported; ?>,
	<?php echo $this->_num_category_uris_already_imported; ?>, 
	<?php echo $this->_num_category_uris_using_zc_uris; ?>, 
	<?php echo "'" . implode(',', $this->_cur_category_pos) . "'"; ?>,
	<?php echo $this->_num_products_inc; ?>,
	<?php echo $this->_num_products_processed; ?>,
	<?php echo $this->_num_product_uris_imported; ?>,
	<?php echo $this->_num_product_uris_already_imported; ?>,
	<?php echo $this->_num_product_uris_using_zc_uris; ?>,
	<?php echo $this->_cur_product_pos; ?>,
	<?php echo $this->_processing_cat_mixed_with_products; ?>);
<?php } else { ?>
var params_to_send = buildParams(<?php echo ($this->_import_product_uris ? 1 : 0); ?>,
	<?php echo ($this->_include_disabled_cats_prods ? 1 : 0); ?>,
	<?php echo "'" . implode(',', $this->_categories_to_import) . "'"; ?>,
	<?php echo $this->_num_categories_inc; ?>, <?php echo $this->_num_categories_processed; ?>,
	<?php echo $this->_num_category_uris_imported; ?>,
	<?php echo $this->_num_category_uris_already_imported; ?>, 
	<?php echo $this->_num_category_uris_using_zc_uris; ?>, 
	<?php echo "'" . implode(',', $this->_cur_category_pos . "'"); ?>);
<?php } ?>

function makeRequest(uri, params_to_send)
{
	http_request = false;
	
	if (window.XMLHttpRequest) { 
		// Mozilla,...
		http_request = new XMLHttpRequest();
		
		if (http_request.overrideMimeType) {
			http_request.overrideMimeType("text/xml");
		}
	} else if (window.ActiveXObject) { 
		// IE
		try {
			http_request = new ActiveXObject("Msxml2.XMLHTTP");
		} catch(e) {
			try {
				http_request = new ActiveXObject("Microsoft.XMLHTTP");
			} catch(e) {}
		}
	}
	
	if (!http_request) {
		alert("Cannot create an XMLHTTP instance");
		return false;
	}
	
	// Indicate that an AJAX request is taking place
	var page_title_el = document.getElementById('import-title');
	
	if (page_title_el != undefined) {
		ajax_request_in_progress_el = document.getElementById('ajax-request-in-progress');
		
		if (ajax_request_in_progress_el == undefined) {
			ajax_request_in_progress_el = document.createElement('div');
			ajax_request_in_progress_el.setAttribute('id', 'ajax-request-in-progress');
			
			image_el = document.createElement('img');
			image_el.setAttribute('src', '<?php echo DIR_WS_IMAGES . 'ceon-star-processing.gif'; ?>');
			
			ajax_request_in_progress_el.appendChild(image_el);
			
			var text_node = document.createTextNode('<?php echo TEXT_AJAX_REQUEST_IN_PROGRESS; ?>');
			
			ajax_request_in_progress_el.appendChild(text_node);
			
			page_title_el.appendChild(ajax_request_in_progress_el);
		}
		
		ajax_request_in_progress_el.style.display = 'block';
	}
	
	http_request.onreadystatechange = handleServerResponse;
	http_request.open("POST", uri, true);
	
	http_request.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
	http_request.setRequestHeader("Content-length", params_to_send.length);
	http_request.setRequestHeader("Connection", "close");
	
	http_request.send(params_to_send);
}

function handleServerResponse() 
{
	// Waiting for correct response
	if (http_request.readyState != 4) {
		return;
	}
	
	if (http_request.status != 200) {
		alert("Page unavailable, or wrong url!")
		return;
	}
	
	var xml = http_request.responseXML;
	
	// Check that expected XML was received
	if (xml.getElementsByTagName('root').length == 0) {
		var text = http_request.responseText;
		
		document.open();
		document.write(text);
		document.close();
		
		return;
	}
	
	// Build parameters for next stage
	var import_product_uris = parseInt(getValueFromXML(xml, 'import_product_uris'));
	var include_disabled_cats_prods = parseInt(getValueFromXML(xml, 'include_disabled_cats_prods'));
	var categories_to_import = getValueFromXML(xml, 'categories_to_import');
	var num_categories_inc = parseInt(getValueFromXML(xml, 'num_categories_inc'));
	var num_categories_processed = parseInt(getValueFromXML(xml, 'num_categories_processed'));
	var num_category_uris_imported = parseInt(getValueFromXML(xml, 'num_category_uris_imported'));
	var num_category_uris_already_imported = parseInt(getValueFromXML(xml, 'num_category_uris_already_imported'));
	var num_category_uris_using_zc_uris = parseInt(getValueFromXML(xml, 'num_category_uris_using_zc_uris'));
	var cur_category_pos = getValueFromXML(xml, 'cur_category_pos');
	
	<?php if ($this->_import_product_uris) { ?>
	var num_products_inc = parseInt(getValueFromXML(xml, 'num_products_inc'));
	var num_products_processed = parseInt(getValueFromXML(xml, 'num_products_processed'));
	var num_product_uris_imported = parseInt(getValueFromXML(xml, 'num_product_uris_imported'));
	var num_product_uris_already_imported = parseInt(getValueFromXML(xml, 'num_product_uris_already_imported'));
	var num_product_uris_using_zc_uris = parseInt(getValueFromXML(xml, 'num_product_uris_using_zc_uris'));
	var cur_product_pos = parseInt(getValueFromXML(xml, 'cur_product_pos'));
	var proc_mixed_cat_prods = parseInt(getValueFromXML(xml, 'proc_mixed_cat_prods'));
	<?php } ?>
	
	var num_categories_processed_el = document.getElementById('num-cats-processed');
	
	if (num_categories_processed_el != undefined) {
		var text_node = document.createTextNode(num_categories_processed);
		
		num_categories_processed_el.removeChild(num_categories_processed_el.firstChild);
		
		num_categories_processed_el.appendChild(text_node);
	}
	
	// Update categories progress bar
	var progress_bar_el = document.getElementById('categories-progress');
	
	if (progress_bar_el != undefined) {
		// Adjust for padding width of 2% each side
		if (num_categories_inc == 0) {
			percentage_completed = 96;
		} else if (num_categories_processed == 0) {
			percentage_completed = 0;
		} else {
			percentage_completed = (96 / (num_categories_inc / num_categories_processed));
		}
		
		progress_bar_el.style.width = percentage_completed.toFixed(2) + '%';
		
		if (num_categories_inc == 0) {
			percentage_completed = 96;
		} else if (num_categories_processed == 0) {
			percentage_completed = 0;
		} else {
			percentage_completed = (100 / (num_categories_inc / num_categories_processed));
		}
		
		var text_node = document.createTextNode(percentage_completed.toFixed(2) + '%');
		
		progress_bar_el.removeChild(progress_bar_el.firstChild);
		
		progress_bar_el.appendChild(text_node);
	}
	
	
	var num_category_uris_imported_count_el =
		document.getElementById('num-cats-imported-count');
	
	if (num_category_uris_imported_count_el != undefined) {
		var text_node = document.createTextNode(num_category_uris_imported);
		
		num_category_uris_imported_count_el.removeChild(
			num_category_uris_imported_count_el.firstChild);
		
		num_category_uris_imported_count_el.appendChild(text_node);
	}
	
	
	var num_category_uris_already_imported_el = document.getElementById('num-cats-already-imported');
	
	var num_category_uris_already_imported_count_el = document.getElementById('num-cats-already-imported-count');
	
	if (num_category_uris_already_imported > 0 && num_category_uris_already_imported_el != undefined &&
			num_category_uris_already_imported_count_el != undefined) {
		
		num_category_uris_already_imported_el.style.display = 'block';
		
		var text_node = document.createTextNode(num_category_uris_already_imported);
		
		num_category_uris_already_imported_count_el.removeChild(
			num_category_uris_already_imported_count_el.firstChild);
		
		num_category_uris_already_imported_count_el.appendChild(text_node);
	}
	
	// Output information about any mappings that were imported
	if (num_category_uris_imported > 0) {
		// Create the listing div if it doesn't exist yet
		var categories_imported_listing_el = document.getElementById('categories-imported-listing');
		
		if (categories_imported_listing_el == undefined) {
			categories_imported_listing_el = document.createElement('div');
			categories_imported_listing_el.setAttribute('id', 'categories-imported-listing');
			
			num_category_uris_already_imported_el.appendChild(categories_imported_listing_el);
		}
		
		category_uris_imported = xml.getElementsByTagName('categories_imported');
		
		if (category_uris_imported != undefined && category_uris_imported[0] != undefined &&
				categories_imported_listing_el != undefined) {
			
			category_uris_imported = category_uris_imported[0].getElementsByTagName('cat');
			
			if (category_uris_imported != undefined) {
				// Build info needed for auto-scrolling
				var start_height = 0;
				
				if (categories_imported_listing_el.scrollHeight > 0) {
					start_height = categories_imported_listing_el.scrollHeight;
				} else if (categories_imported_listing_el.offsetHeight > 0) {
					start_height = categories_imported_listing_el.offsetHeight;
				}
				
				for (var cat_i = 0; cat_i < category_uris_imported.length; cat_i++) {
					message = getValueFromXML(category_uris_imported[cat_i], 'message');
					
					link = getValueFromXML(category_uris_imported[cat_i], 'link');
					
					link_title = getValueFromXML(category_uris_imported[cat_i], 'link_title');
					
					edit_link = getValueFromXML(category_uris_imported[cat_i], 'edit_link');
					
					edit_link_title = getValueFromXML(category_uris_imported[cat_i], 'edit_link_title');
					
					if (message != undefined) {
						var p_el = document.createElement('p');
						
						p_el.appendChild(document.createTextNode(message + ' '));
						
						if (link != undefined && link_title != undefined) {
							var link_el = document.createElement('a');
							link_el.setAttribute('href', link);
							link_el.setAttribute('target', '_blank');
							link_el.appendChild(document.createTextNode(link_title));
							
							p_el.appendChild(link_el);
						}
						
						if (edit_link != undefined && edit_link_title != undefined) {
							var edit_link_el = document.createElement('a');
							edit_link_el.setAttribute('href', edit_link);
							edit_link_el.setAttribute('target', '_blank');
							edit_link_el.appendChild(document.createTextNode(edit_link_title));
							
							p_el.appendChild(document.createTextNode(' - '));
							
							p_el.appendChild(edit_link_el);
						}
						
						categories_imported_listing_el.appendChild(p_el);
					}
				}
				
				// Scroll to the bottom if possible
				var current_height = 0;
				
				if (categories_imported_listing_el.scrollHeight > 0) {
					current_height = categories_imported_listing_el.scrollHeight;
				} else if (categories_imported_listing_el.offsetHeight > 0) {
					current_height = categories_imported_listing_el.offsetHeight;
				}
				
				// Height of updated div must allowed for in test to see if user has scrolled the div
				var added_height = current_height - start_height;
				
				if (first_stage || (current_height - categories_imported_listing_el.scrollTop -
						((categories_imported_listing_el.style.pixelHeight) ?
						categories_imported_listing_el.style.pixelHeight :
						categories_imported_listing_el.offsetHeight) < (added_height + 25))) {
					categories_imported_listing_el.scrollTop = current_height;
				}
			}
		}
	}
	
	// Output information about any errors that were encountered
	if (num_category_uris_using_zc_uris > 0) {
		var num_category_uris_using_zc_uris_el = document.getElementById('num-cats-using-zc-uris');
		
		var num_category_uris_using_zc_uris_count_el = document.getElementById('num-cats-using-zc-uris-count');
		
		if (num_category_uris_using_zc_uris_el != undefined &&
				num_category_uris_using_zc_uris_count_el != undefined) {
			// Make sure info is visible
			num_category_uris_using_zc_uris_el.style.display = 'block';
			
			var text_node = document.createTextNode(num_category_uris_using_zc_uris);
			
			if (num_category_uris_using_zc_uris_count_el.firstChild != undefined) {
				num_category_uris_using_zc_uris_count_el.removeChild(
					num_category_uris_using_zc_uris_count_el.firstChild);
			}
			
			num_category_uris_using_zc_uris_count_el.appendChild(text_node);
			
			// Create the listing div if it doesn't exist yet
			var category_uris_using_zc_uris_listing_el =
				document.getElementById('category-uris-using-zc-uris-listing');
			
			if (category_uris_using_zc_uris_listing_el == undefined) {
				category_uris_using_zc_uris_listing_el = document.createElement('div');
				category_uris_using_zc_uris_listing_el.setAttribute('id', 'category-uris-using-zc-uris-listing');
				
				num_category_uris_using_zc_uris_el.appendChild(category_uris_using_zc_uris_listing_el);
			}
			
			category_uris_using_zc_uris = xml.getElementsByTagName('category_uris_using_zc_uris');
			
			if (category_uris_using_zc_uris != undefined && category_uris_using_zc_uris[0] != undefined) {
				
				category_uris_using_zc_uris = category_uris_using_zc_uris[0].getElementsByTagName('cat');
				
				if (category_uris_using_zc_uris != undefined) {
					// Build info needed for auto-scrolling
					var start_height = 0;
					
					if (category_uris_using_zc_uris_listing_el.scrollHeight > 0) {
						start_height = category_uris_using_zc_uris_listing_el.scrollHeight;
					} else if (category_uris_using_zc_uris_listing_el.offsetHeight > 0) {
						start_height = category_uris_using_zc_uris_listing_el.offsetHeight;
					}
					
					for (var cat_i = 0; cat_i < category_uris_using_zc_uris.length; cat_i++) {
						message = getValueFromXML(category_uris_using_zc_uris[cat_i], 'message');
						
						link = getValueFromXML(category_uris_using_zc_uris[cat_i], 'link');
						
						link_title = getValueFromXML(category_uris_using_zc_uris[cat_i], 'link_title');
						
						edit_link = getValueFromXML(category_uris_using_zc_uris[cat_i], 'edit_link');
						
						edit_link_title = getValueFromXML(category_uris_using_zc_uris[cat_i], 'edit_link_title');
						
						if (message != undefined) {
							var p_el = document.createElement('p');
							
							p_el.appendChild(document.createTextNode(message + ' '));
							
							if (link != undefined && link_title != undefined) {
								var link_el = document.createElement('a');
								link_el.setAttribute('href', link);
								link_el.setAttribute('target', '_blank');
								link_el.appendChild(document.createTextNode(link_title));
								
								p_el.appendChild(link_el);
							}
							
							if (edit_link != undefined && edit_link_title != undefined) {
								var edit_link_el = document.createElement('a');
								edit_link_el.setAttribute('href', edit_link);
								edit_link_el.setAttribute('target', '_blank');
								edit_link_el.appendChild(document.createTextNode(edit_link_title));
								
								p_el.appendChild(document.createTextNode(' - '));
								
								p_el.appendChild(edit_link_el);
							}
							
							category_uris_using_zc_uris_listing_el.appendChild(p_el);
						}
					}
					
					// Scroll to the bottom if possible
					var current_height = 0;
					
					if (category_uris_using_zc_uris_listing_el.scrollHeight > 0) {
						current_height = category_uris_using_zc_uris_listing_el.scrollHeight;
					} else if (category_uris_using_zc_uris_listing_el.offsetHeight > 0) {
						current_height = category_uris_using_zc_uris_listing_el.offsetHeight;
					}
					
					// Height of updated div must allowed for in test to see if user has scrolled the div
					var added_height = current_height - start_height;
					
					if (first_stage ||
							(current_height - category_uris_using_zc_uris_listing_el.scrollTop -
							((category_uris_using_zc_uris_listing_el.style.pixelHeight) ?
							category_uris_using_zc_uris_listing_el.style.pixelHeight :
							category_uris_using_zc_uris_listing_el.offsetHeight) < (added_height + 25))) {
						category_uris_using_zc_uris_listing_el.scrollTop = current_height;
					}
				}
			}
		}
	}
	
	
	<?php if ($this->_import_product_uris) { ?>
	var num_products_processed_el = document.getElementById('num-prods-processed');
	
	if (num_products_processed_el != undefined) {
		var text_node = document.createTextNode(num_products_processed);
		
		num_products_processed_el.removeChild(num_products_processed_el.firstChild);
		
		num_products_processed_el.appendChild(text_node);
	}
	
	// Update products progress bar
	var progress_bar_el = document.getElementById('products-progress');
	
	if (progress_bar_el != undefined) {
		// Adjust for padding width of 2% each side
		if (num_products_inc == 0) {
			percentage_completed = 96;
		} else if (num_products_processed == 0) {
			percentage_completed = 0;
		} else {	
			percentage_completed = (96 / (num_products_inc / num_products_processed));
		}
		
		progress_bar_el.style.width = percentage_completed.toFixed(2) + '%';
		
		if (num_products_inc == 0) {
			percentage_completed = 100;
		} else if (num_products_processed == 0) {
			percentage_completed = 0;
		} else {
			percentage_completed = (100 / (num_products_inc / num_products_processed));
		}
		
		var text_node = document.createTextNode(percentage_completed.toFixed(2) + '%');
		
		progress_bar_el.removeChild(progress_bar_el.firstChild);
		
		progress_bar_el.appendChild(text_node);
	}
	
	
	var num_product_uris_imported_count_el = document.getElementById('num-prods-imported-count');
	
	if (num_product_uris_imported_count_el != undefined) {
		var text_node = document.createTextNode(num_product_uris_imported);
		
		num_product_uris_imported_count_el.removeChild(num_product_uris_imported_count_el.firstChild);
		
		num_product_uris_imported_count_el.appendChild(text_node);
	}
	
	
	var num_product_uris_already_imported_el = document.getElementById('num-prods-already-imported');
	
	var num_product_uris_already_imported_count_el = document.getElementById('num-prods-already-imported-count');
	
	if (num_product_uris_already_imported > 0 && num_product_uris_already_imported_el != undefined &&
			num_product_uris_already_imported_count_el != undefined) {
		
		num_product_uris_already_imported_el.style.display = 'block';
		
		var text_node = document.createTextNode(num_product_uris_already_imported);
		
		num_product_uris_already_imported_count_el.removeChild(
			num_product_uris_already_imported_count_el.firstChild);
		
		num_product_uris_already_imported_count_el.appendChild(text_node);
	}
	
	// Output information about any mappings that were imported
	if (num_product_uris_imported > 0) {
		// Create the listing div if it doesn't exist yet
		var products_imported_listing_el = document.getElementById('products-imported-listing');
		
		if (products_imported_listing_el == undefined) {
			products_imported_listing_el = document.createElement('div');
			products_imported_listing_el.setAttribute('id', 'products-imported-listing');
			
			num_product_uris_already_imported_el.appendChild(products_imported_listing_el);
		}
		
		product_uris_imported = xml.getElementsByTagName('products_imported');
		
		if (product_uris_imported != undefined && product_uris_imported[0] != undefined &&
				products_imported_listing_el != undefined) {
			
			product_uris_imported = product_uris_imported[0].getElementsByTagName('product');
			
			if (product_uris_imported != undefined) {
				// Build info needed for auto-scrolling
				var start_height = 0;
				
				if (products_imported_listing_el.scrollHeight > 0) {
					start_height = products_imported_listing_el.scrollHeight;
				} else if (products_imported_listing_el.offsetHeight > 0) {
					start_height = products_imported_listing_el.offsetHeight;
				}
				
				for (var prod_i = 0; prod_i < product_uris_imported.length; prod_i++) {
					message = getValueFromXML(product_uris_imported[prod_i], 'message');
					
					link = getValueFromXML(product_uris_imported[prod_i], 'link');
					
					link_title = getValueFromXML(product_uris_imported[prod_i], 'link_title');
					
					edit_link = getValueFromXML(product_uris_imported[prod_i], 'edit_link');
					
					edit_link_title = getValueFromXML(product_uris_imported[prod_i], 'edit_link_title');
					
					if (message != undefined) {
						var p_el = document.createElement('p');
						
						p_el.appendChild(document.createTextNode(message + ' '));
						
						if (link != undefined && link_title != undefined) {
							var link_el = document.createElement('a');
							link_el.setAttribute('href', link);
							link_el.setAttribute('target', '_blank');
							link_el.appendChild(document.createTextNode(link_title));
							
							p_el.appendChild(link_el);
						}
						
						if (edit_link != undefined && edit_link_title != undefined) {
							var edit_link_el = document.createElement('a');
							edit_link_el.setAttribute('href', edit_link);
							edit_link_el.setAttribute('target', '_blank');
							edit_link_el.appendChild(document.createTextNode(edit_link_title));
							
							p_el.appendChild(document.createTextNode(' - '));
							
							p_el.appendChild(edit_link_el);
						}
						
						products_imported_listing_el.appendChild(p_el);
					}
				}
				
				// Scroll to the bottom if possible
				var current_height = 0;
				
				if (products_imported_listing_el.scrollHeight > 0) {
					current_height = products_imported_listing_el.scrollHeight;
				} else if (products_imported_listing_el.offsetHeight > 0) {
					current_height = products_imported_listing_el.offsetHeight;
				}
				
				// Height of updated div must allowed for in test to see if user has scrolled the div
				var added_height = current_height - start_height;
				
				if (first_stage || (current_height - products_imported_listing_el.scrollTop -
						((products_imported_listing_el.style.pixelHeight) ?
						products_imported_listing_el.style.pixelHeight :
						products_imported_listing_el.offsetHeight) < (added_height + 25))) {
					
					products_imported_listing_el.scrollTop = current_height;
				}
			}
		}
	}
	
	// Output information about any errors that were encountered
	if (num_product_uris_using_zc_uris > 0) {
		var num_product_uris_using_zc_uris_el = document.getElementById('num-prods-using-zc-uris');
		
		var num_product_uris_using_zc_uris_count_el = document.getElementById('num-prods-using-zc-uris-count');
		
		if (num_product_uris_using_zc_uris_el != undefined &&
				num_product_uris_using_zc_uris_count_el != undefined) {
			// Make sure info is visible
			num_product_uris_using_zc_uris_el.style.display = 'block';
			
			var text_node = document.createTextNode(num_product_uris_using_zc_uris);
			
			if (num_product_uris_using_zc_uris_count_el.firstChild != undefined) {
				num_product_uris_using_zc_uris_count_el.removeChild(
					num_product_uris_using_zc_uris_count_el.firstChild);
			}
			
			num_product_uris_using_zc_uris_count_el.appendChild(text_node);
			
			// Create the listing div if it doesn't exist yet
			var product_uris_using_zc_uris_listing_el =
				document.getElementById('product-uris-using-zc-uris-listing');
			
			if (product_uris_using_zc_uris_listing_el == undefined) {
				product_uris_using_zc_uris_listing_el = document.createElement('div');
				product_uris_using_zc_uris_listing_el.setAttribute('id', 'product-uris-using-zc-uris-listing');
				
				num_product_uris_using_zc_uris_el.appendChild(product_uris_using_zc_uris_listing_el);
			}
			
			product_uris_using_zc_uris = xml.getElementsByTagName('product_uris_using_zc_uris');
			
			if (product_uris_using_zc_uris != undefined && product_uris_using_zc_uris[0] != undefined) {
				product_uris_using_zc_uris = product_uris_using_zc_uris[0].getElementsByTagName('product');
				
				if (product_uris_using_zc_uris != undefined) {
					// Build info needed for auto-scrolling
					var start_height = 0;
					
					if (product_uris_using_zc_uris_listing_el.scrollHeight > 0) {
						start_height = product_uris_using_zc_uris_listing_el.scrollHeight;
					} else if (product_uris_using_zc_uris_listing_el.offsetHeight > 0) {
						start_height = product_uris_using_zc_uris_listing_el.offsetHeight;
					}
					
					for (var prod_i = 0; prod_i < product_uris_using_zc_uris.length; prod_i++) {
						message = getValueFromXML(product_uris_using_zc_uris[prod_i], 'message');
						
						link = getValueFromXML(product_uris_using_zc_uris[prod_i], 'link');
						
						link_title = getValueFromXML(product_uris_using_zc_uris[prod_i], 'link_title');
						
						edit_link = getValueFromXML(product_uris_using_zc_uris[prod_i], 'edit_link');
						
						edit_link_title = getValueFromXML(product_uris_using_zc_uris[prod_i], 'edit_link_title');
						
						if (message != undefined) {
							var p_el = document.createElement('p');
							
							p_el.appendChild(document.createTextNode(message + ' '));
							
							if (link != undefined && link_title != undefined) {
								var link_el = document.createElement('a');
								link_el.setAttribute('href', link);
								link_el.setAttribute('target', '_blank');
								link_el.appendChild(document.createTextNode(link_title));
								
								p_el.appendChild(link_el);
							}
							
							if (edit_link != undefined && edit_link_title != undefined) {
								var edit_link_el = document.createElement('a');
								edit_link_el.setAttribute('href', edit_link);
								edit_link_el.setAttribute('target', '_blank');
								edit_link_el.appendChild(document.createTextNode(edit_link_title));
								
								p_el.appendChild(document.createTextNode(' - '));
								
								p_el.appendChild(edit_link_el);
							}
							
							product_uris_using_zc_uris_listing_el.appendChild(p_el);
						}
					}
					
					// Scroll to the bottom if possible
					var current_height = 0;
					
					if (product_uris_using_zc_uris_listing_el.scrollHeight > 0) {
						current_height = product_uris_using_zc_uris_listing_el.scrollHeight;
					} else if (product_uris_using_zc_uris_listing_el.offsetHeight > 0) {
						current_height = product_uris_using_zc_uris_listing_el.offsetHeight;
					}
					
					// Height of updated div must allowed for in test to see if user has scrolled the div
					var added_height = current_height - start_height;
					
					if (first_stage || (current_height - product_uris_using_zc_uris_listing_el.scrollTop -
							((product_uris_using_zc_uris_listing_el.style.pixelHeight) ?
							product_uris_using_zc_uris_listing_el.style.pixelHeight :
							product_uris_using_zc_uris_listing_el.offsetHeight) < (added_height + 25))) {
						
						product_uris_using_zc_uris_listing_el.scrollTop = current_height;
					}
				}
			}
		}
	}
	<?php } ?>
	
	if (first_stage) {
		first_stage = false;
	}
	
	<?php if ($this->_import_product_uris) { ?>
	// Has the process completed?
	if (num_categories_inc <= num_categories_processed && num_products_inc <= num_products_processed) {
		
		// Hide processing indicator as process is complete!
		ajax_request_in_progress_el = document.getElementById('ajax-request-in-progress');
		
		if (ajax_request_in_progress_el != undefined) {
			ajax_request_in_progress_el.style.display = 'none';
		}
		
		return;
	}
	
	params_to_send = buildParams(import_product_uris, include_disabled_cats_prods, categories_to_import,
		num_categories_inc, num_categories_processed, num_category_uris_imported,
		num_category_uris_already_imported, num_category_uris_using_zc_uris, cur_category_pos, num_products_inc,
		num_products_processed, num_product_uris_imported, num_product_uris_already_imported,
		num_product_uris_using_zc_uris, cur_product_pos, proc_mixed_cat_prods);
	<?php } else { ?>
	
	// Has the process completed?
	if (num_categories_inc <= num_categories_processed) {
		// Hide processing indicator as process is complete!
		ajax_request_in_progress_el = document.getElementById('ajax-request-in-progress');
		
		if (ajax_request_in_progress_el != undefined) {
			ajax_request_in_progress_el.style.display = 'none';
		}
		
		return;
	}
	
	params_to_send = buildParams(import_product_uris, include_disabled_cats_prods, categories_to_import,
		num_categories_inc, num_categories_processed, num_category_uris_imported,
		num_category_uris_already_imported, num_category_uris_using_zc_uris, cur_category_pos);
	<?php } ?>
	
	// Begin next stage of process
	makeRequest(uri, params_to_send);
}

// Begin process!
window.setTimeout("makeRequest(uri, params_to_send)", 50);

</script>
	<?php
		$content = ob_get_clean();
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _handleCategoriesProductsAJAXRequest()
	
	/**
	 * Handles an AJAX request for the categories/products import section.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _handleCategoriesProductsAJAXRequest()
	{
		// Start the control timer
		$this->_import_start_timestamp = time();
		
		if ($this->_debug_ajax_to_file) {
			$this->_ajax_debug_file = fopen($_SESSION['ceon_umm_ajax_debug_file'], 'a');
			
			fwrite($this->_ajax_debug_file, "Starting AJAX REQUEST at " . date('Ymd His', time()) . "\n\n");
		}
		
		// Get the data posted for this request
		$this->_import_product_uris = ($this->_posted_data['import-product-uris'] == 1 ? true : false);
		
		$this->_include_disabled_cats_prods =
			($this->_posted_data['include-disabled-cats-prods'] == 1 ? true : false);
		
		$this->_categories_to_import = explode(',', $this->_posted_data['cats-to-import']);
		
		$this->_num_categories_inc = $this->_posted_data['num-cats'];
		
		$this->_num_categories_processed = $this->_posted_data['num-cats-proc'];
		$this->_num_category_uris_imported = $this->_posted_data['num-cats-imp'];
		$this->_num_category_uris_already_imported = $this->_posted_data['num-cats-ia'];
		$this->_num_category_uris_using_zc_uris = $this->_posted_data['num-cats-zc'];
		$this->_cur_category_pos = (strlen($this->_posted_data['cur-cat-pos']) > 0 ?
			explode(',', $this->_posted_data['cur-cat-pos']) : array());
		
		if ($this->_import_product_uris) {
			$this->_num_products_inc = $this->_posted_data['num-prods'];
			$this->_num_products_processed = $this->_posted_data['num-prods-proc'];
			$this->_num_product_uris_imported = $this->_posted_data['num-prods-gen'];
			$this->_num_product_uris_already_imported = $this->_posted_data['num-prods-ia'];
			$this->_num_product_uris_using_zc_uris = $this->_posted_data['num-prods-zc'];
			$this->_cur_product_pos = $this->_posted_data['cur-prod-pos'];
			$this->_processing_cat_mixed_with_products = $this->_posted_data['proc-mixed-cat-prods'];
		}
		
		// Carry out next stage of process, if there is one
		if ($this->_num_categories_inc > $this->_num_categories_processed ||
				($this->_import_product_uris && $this->_num_products_inc > $this->_num_products_processed)) {
			$this->_importCategoriesProductsURIs();
		}
		
		// Return XML for request
		header('Content-Type: application/xml');
		header('Cache-Control: no-cache');
		
		ob_start();
		
		echo '<?xml version="1.0" encoding="ISO-8859-1"?>';
		echo '<root>';
		echo '<import_product_uris>';
		echo ($this->_import_product_uris ? 1 : 0);
		echo '</import_product_uris>';
		echo '<include_disabled_cats_prods>';
		echo ($this->_include_disabled_cats_prods ? 1 : 0);
		echo '</include_disabled_cats_prods>';
		echo '<categories_to_import>';
		echo implode(',', $this->_categories_to_import);
		echo '</categories_to_import>';
		echo '<num_categories_inc>';
		echo $this->_num_categories_inc;
		echo '</num_categories_inc>';
		echo '<num_categories_processed>';
		echo $this->_num_categories_processed;
		echo '</num_categories_processed>';
		echo '<num_category_uris_imported>';
		echo $this->_num_category_uris_imported;
		echo '</num_category_uris_imported>';
		echo '<num_category_uris_already_imported>';
		echo $this->_num_category_uris_already_imported;
		echo '</num_category_uris_already_imported>';
		echo '<num_category_uris_using_zc_uris>';
		echo $this->_num_category_uris_using_zc_uris;
		echo '</num_category_uris_using_zc_uris>';
		echo '<cur_category_pos>';
		echo implode(',', $this->_cur_category_pos);
		echo '</cur_category_pos>';
		
		if (sizeof($this->_category_uris_imported_messages) > 0) {
			echo '<categories_imported>';
			
			foreach ($this->_category_uris_imported_messages as $message) {
				echo '<cat><message>';
				echo htmlspecialchars($message['message']);
				echo '</message><link>';
				echo htmlspecialchars($message['link']);
				echo '</link><link_title>';
				echo htmlspecialchars($message['link_title']);
				echo '</link_title>';
				
				if (isset($message['edit_link'])) {
					echo '<edit_link>';
					echo htmlspecialchars($message['edit_link']);
					echo '</edit_link><edit_link_title>';
					echo htmlspecialchars($message['edit_link_title']);
					echo '</edit_link_title>';
				}
				
				echo '</cat>';
			}
			
			echo '</categories_imported>';
		}
		
		if (sizeof($this->_category_uris_using_zc_uris_messages) > 0) {
			echo '<category_uris_using_zc_uris>';
			
			foreach ($this->_category_uris_using_zc_uris_messages as $message) {
				echo '<cat><message>';
				echo htmlspecialchars($message['message']);
				echo '</message><link>';
				echo htmlspecialchars($message['link']);
				echo '</link><link_title>';
				echo htmlspecialchars($message['link_title']);
				echo '</link_title>';
				
				if (isset($message['edit_link'])) {
					echo '<edit_link>';
					echo htmlspecialchars($message['edit_link']);
					echo '</edit_link><edit_link_title>';
					echo htmlspecialchars($message['edit_link_title']);
					echo '</edit_link_title>';
				}
				
				echo '</cat>';
			}
			
			echo '</category_uris_using_zc_uris>';
		}
		
		if ($this->_import_product_uris) {
			echo '<num_products_inc>';
			echo $this->_num_products_inc;
			echo '</num_products_inc>';
			echo '<num_products_processed>';
			echo $this->_num_products_processed;
			echo '</num_products_processed>';
			echo '<num_product_uris_imported>';
			echo $this->_num_product_uris_imported;
			echo '</num_product_uris_imported>';
			echo '<num_product_uris_already_imported>';
			echo $this->_num_product_uris_already_imported;
			echo '</num_product_uris_already_imported>';
			echo '<num_product_uris_using_zc_uris>';
			echo $this->_num_product_uris_using_zc_uris;
			echo '</num_product_uris_using_zc_uris>';
			echo '<cur_product_pos>';
			echo $this->_cur_product_pos;
			echo '</cur_product_pos>';
			echo '<proc_mixed_cat_prods>';
			echo $this->_processing_cat_mixed_with_products;
			echo '</proc_mixed_cat_prods>';
			
			if (sizeof($this->_product_uris_imported_messages) > 0) {
				echo '<products_imported>';
				
				foreach ($this->_product_uris_imported_messages as $message) {
					echo '<product><message>';
					echo htmlspecialchars($message['message']);
					echo '</message><link>';
					echo htmlspecialchars($message['link']);
					echo '</link><link_title>';
					echo htmlspecialchars($message['link_title']);
					echo '</link_title>';
					
					if (isset($message['edit_link'])) {
						echo '<edit_link>';
						echo htmlspecialchars($message['edit_link']);
						echo '</edit_link><edit_link_title>';
						echo htmlspecialchars($message['edit_link_title']);
						echo '</edit_link_title>';
					}
					
					echo '</product>';
				}
				
				echo '</products_imported>';
			}
			
			if (sizeof($this->_product_uris_using_zc_uris_messages) > 0) {
				echo '<product_uris_using_zc_uris>';
				
				foreach ($this->_product_uris_using_zc_uris_messages as $message) {
					echo '<product><message>';
					echo htmlspecialchars($message['message']);
					echo '</message><link>';
					echo htmlspecialchars($message['link']);
					echo '</link><link_title>';
					echo htmlspecialchars($message['link_title']);
					echo '</link_title>';
					
					if (isset($message['edit_link'])) {
						echo '<edit_link>';
						echo htmlspecialchars($message['edit_link']);
						echo '</edit_link><edit_link_title>';
						echo htmlspecialchars($message['edit_link_title']);
						echo '</edit_link_title>';
					}
					
					echo '</product>';
				}
				
				echo '</product_uris_using_zc_uris>';
			}
		}
		
		echo "</root>";	
		
		$xml = ob_get_clean();
		
		echo $xml;
		
		if ($this->_debug_ajax_to_file) {
			fwrite($this->_ajax_debug_file, "\n\nEnding AJAX Request at " .
				date('Ymd His', time()) . "\n\n");
			
			fclose($this->_ajax_debug_file);
		}
		
		include('includes/application_bottom.php');
		
		exit();
	}
	
	// }}}
	
	
	// {{{ _getCategorySubcategoriesProductsCount()
	
	/**
	 * Simply builds a count of the number of subcategories the specified category contains, as well as the number
	 * of products contained within those subcategories.
	 *
	 * @access  protected
	 * @param   integer   $category_id    The ID of the category.
	 * @param   boolean   $inc_products   Whether or not products should be also counted.
	 * @param   boolean   $inc_disabled   Whether or not disabled subcategories/products should be included in the
	 *                                    count.
	 * @return  array     An associative array with the number of subcategories the specified category contains,
	 *                    and optinally the number of products contained within all those subcategories.
	 */
	protected function _getCategorySubcategoriesProductsCount($category_id, $inc_products, $inc_disabled)
	{
		global $db;
		
		$category_counts = array(
			'num_subcategories' => 0,
			'num_products' => 0
			);
		
		if ($inc_disabled) {
			$subcategories_result = $db->Execute("
				SELECT
					c.categories_id
				FROM
					" . TABLE_CATEGORIES . " c
				WHERE
					c.parent_id = " . (int) $category_id . ";");
		} else {
			$subcategories_result = $db->Execute("
				SELECT
					c.categories_id
				FROM
					" . TABLE_CATEGORIES . " c
				WHERE
					c.parent_id = " . (int) $category_id . "
				AND
					c.categories_status = '1';");
		}
		
		while (!$subcategories_result->EOF) {
			$category_counts['num_subcategories']++;
			
			$subcategory_id = $subcategories_result->fields['categories_id'];
			
			$sub_category_info =
				$this->_getCategorySubcategoriesProductsCount($subcategory_id, $inc_products, $inc_disabled);
			
			$category_counts['num_subcategories'] += $sub_category_info['num_subcategories'];
			
			if ($inc_products) {
				if ($inc_disabled) {
					$products_count_result = $db->Execute("
						SELECT
							count(*) as num_products
                        FROM
							" . TABLE_PRODUCTS . " p
                        WHERE
							p.master_categories_id = " . (int) $subcategory_id . ";");
				} else {
					$products_count_result = $db->Execute("
						SELECT
							count(*) as num_products
                        FROM
							" . TABLE_PRODUCTS . " p
                        WHERE
							p.master_categories_id = " . (int) $subcategory_id . "
						AND
							p.products_status = '1';");
				}
				
				if (!$products_count_result->EOF) {
					$category_counts['num_products'] +=
						$products_count_result->fields['num_products'];
				}
				
				$category_counts['num_products'] += $sub_category_info['num_products'];
			}
			
			$subcategories_result->MoveNext();
		}
		
		return $category_counts;
	}
	
	// }}}
	
	
	// {{{ getOutput()
	
	/**
	 * Returns the output imported by this action.
	 *
	 * @access  public
	 * @return  string     The HTML output for this action.
	 */
	public function getOutput()
	{
		return $this->_output;
	}
	
	// }}}
	
	
	// {{{ getErrorMessages()
	
	/**
	 * Returns any error messages for this action.
	 *
	 * @access  protected
	 * @return  array      The error messages for this action.
	 */
	protected function getErrorMessages()
	{
		return $this->_error_messages;
	}
	
	// }}}
	
	
	// {{{ _loadConfig()
	
	/**
	 * Loads the configuration for the software from the database.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _loadConfig()
	{
		global $db;
		
		// Only one config currently supported so its ID is hard-coded in the following SQL
		$load_config_sql = "
			SELECT
				use_ajax_progress_indicators,
				ajax_progress_refresh_sec,
				standard_progress_refresh_sec
			FROM
				" . TABLE_CEON_UMM_CONFIGS . " 
			WHERE
				id = '1';";
		
		$load_config_result = $db->Execute($load_config_sql);
		
		if ($load_config_result->EOF) {
			
		} else {
			$this->_use_ajax_progress_indicators =
				($load_config_result->fields['use_ajax_progress_indicators'] == 1 ? true : false);
			
			$this->_ajax_progress_refresh_sec = $load_config_result->fields['ajax_progress_refresh_sec'];
			
			$this->_standard_progress_refresh_sec = $load_config_result->fields['standard_progress_refresh_sec'];
		}
	}
	
	// }}}
}

// }}}
