<?php

/**
 * Ceon URI Mapping - UMM Edition - URI Admin Base Class.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingAdmin.php 1027 2012-07-17 20:31:10Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingDBLookup.php');


// {{{ constants

define('CEON_URI_MAPPING_SINGLE_UNDERSCORE', 1);
define('CEON_URI_MAPPING_SINGLE_DASH', 2);
define('CEON_URI_MAPPING_SINGLE_FULL_STOP', 3);
define('CEON_URI_MAPPING_REMOVE', 4);

define('CEON_URI_MAPPING_CAPITALISATION_LOWERCASE', 1);
define('CEON_URI_MAPPING_CAPITALISATION_AS_IS', 2);
define('CEON_URI_MAPPING_CAPITALISATION_UCFIRST', 3);

define('CEON_URI_MAPPING_ADD_MAPPING_SUCCESS', 1);
define('CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS', -1);
define('CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR', -2);
define('CEON_URI_MAPPING_ADD_MAPPING_ERROR_DB_ERROR', -3);

define('CEON_URI_MAPPING_MAKE_MAPPING_HISTORICAL_SUCCESS', 1);
define('CEON_URI_MAPPING_MAKE_MAPPING_HISTORICAL_ERROR_DATA_ERROR', -1);

define('CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED', 1);
define('CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED', 2);
define('CEON_URI_MAPPING_MAPPING_TEMPLATE_DELETED', 3);
define('CEON_URI_MAPPING_MAPPING_TEMPLATE_UNCHANGED', -1);
define('CEON_URI_MAPPING_MAPPING_TEMPLATE_DIDNT_EXIST', -2);

define('CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME', 1);
define('CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME', 2);
define('CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME', 3);

// }}}


// {{{ CeonURIMappingAdmin

/**
 * Provides shared functionality for the Ceon URI Mapping admin functionality.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingAdmin extends CeonURIMappingDBLookup
{
	// {{{ properties
	
	/**
	 * Whether autogeneration of URI mappings is enabled or not.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_autogen_new = null;
	
	/**
	 * The site's setting for the whitesapce replacement character.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_whitespace_replacement = null;
	
	/**
	 * The site's setting for the capitalisation of words in URI mappings.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_capitalisation = null;
	
	/**
	 * The list of words which should be removed from URI mappings being autogenerated. This property is not parsed
	 * from its encoded string into a array until used for the first time, to save processing time.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_remove_words = null;
	
	/**
	 * The list of character to string replacements which should be applied to URI mappings being autogenerated.
	 * This is not parsed from its encoded string into a usable array until used for the first time, to save
	 * processing time.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_char_str_replacements = null;
	
	/** 
	 * The add language code identifier to URI setting for the store. 
	 * 
	 * @var     integer 
	 * @access  protected 
	 */ 
	protected $_language_code_add = null; 


	/**
	 * The site's setting for the handling of a URI mapping which clashes with an existing mapping.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_mapping_clash_action = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingAdmin class. Loads the autogeneration settings for the store and
	 * sets the class properties' values.
	 * 
	 * @param   boolean   Whether or not the autogeneration configuration should be loaded when instantiating the
	 *                    class.
	 * @access  public
	 */
	public function __construct($load_config = true)
	{
		parent::__construct();
		
		if ($load_config) {
			$this->_loadAutogenerationSettings();
		}
	}
	
	// }}}
	
	
	// {{{ _loadAutogenerationSettings()
	
	/**
	 * Loads the autogeneration settings for the store and sets the class properties' values.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _loadAutogenerationSettings()
	{
		global $db;
		
		// Only one config currently supported so the ID is hard-coded in the following SQL
		$load_autogen_settings_sql = "
			SELECT
				autogen_new,
				whitespace_replacement,
				capitalisation,
				remove_words,
				char_str_replacements,
				language_code_add,
				mapping_clash_action
			FROM
				" . TABLE_CEON_URI_MAPPING_CONFIGS . "
			WHERE
				id ='1';";
		
		$load_autogen_settings_result = $db->Execute($load_autogen_settings_sql);
		
		if (!$load_autogen_settings_result->EOF) {
			$this->_autogen_new = ($load_autogen_settings_result->fields['autogen_new'] == 1 ? true : false);
			
			$this->_whitespace_replacement = $load_autogen_settings_result->fields['whitespace_replacement'];
			
			$this->_capitalisation = $load_autogen_settings_result->fields['capitalisation'];
			
			$this->_remove_words = $load_autogen_settings_result->fields['remove_words'];
			
			$this->_char_str_replacements = $load_autogen_settings_result->fields['char_str_replacements'];
			
			$this->_language_code_add = $load_autogen_settings_result->fields['language_code_add'];
			
			$this->_mapping_clash_action = $load_autogen_settings_result->fields['mapping_clash_action'];
		}
	}
	
	// }}}
	
	
	// {{{ addURIMapping()
	
	/**
	 * Adds a URI mapping to the database. Use of this method abstracts the calling code from the actual
	 * implementation of the database's structure.
	 *
	 * @access  public
	 * @param   string    $uri   The URI for the mapping.
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping.
	 * @param   string    $main_page   The Zen Cart page type the mapping is for.
	 * @param   string    $query_string_parameters   The query string parameters which are to be passed on to Zen
	 *                                               Cart as if they were in the original URI. ZC often uses this
	 *                                               information to "define" a page type or a particular instance
	 *                                               of a page type.
	 * @param   integer   $associated_db_id   The ID of a database record to be passed on to ZC. ZC uses this to
	 *                                        identify a particular instance of a page type.
	 * @param   string    $alternate_uri   An alternative URI to redirect to, instead of mapping to a Zen Cart
	 *                                     page.
	 * @param   integer   $redirection_type_code   The redirection type code to be used when redirecting to an
	 *                                             alternate URI.
	 * @param   boolean   $avoid_duplicates   Whether or not to prevent the creation of a duplicate mapping. Should
	 *                                        only ever be disabled if checks have already been made and should
	 *                                        therefore be skipped here for efficiency.
	 * @return  integer   A positive value if the mapping was added, a negative integer (constant value) if a
	 *                    mapping already exists for the specified URI or an error was encountered.
	 */
	public function addURIMapping($uri, $language_id = null, $main_page = null, $query_string_parameters = null,
		$associated_db_id = null, $alternate_uri = null, $redirection_type_code = null, $avoid_duplicates = true)
	{
		global $db;
		
		if (is_null($uri) || strlen($uri) == 0) {
			// Cannot be null!
			return CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR;
		} else {
			$uri_quoted = "'" . zen_db_input(zen_db_prepare_input($uri)) . "'";
		}
		
		if (is_null($language_id) || (int) $language_id <= 0) {
			$language_id_quoted = 'NULL';
		} else {
			$language_id_quoted = "'" . (int) $language_id . "'";
		}
		
		$current_uri_quoted = "'1'";
		
		if (is_null($main_page) || strlen($main_page) == 0) {
			$main_page_quoted = 'NULL';
		} else {
			$main_page_quoted = "'" . zen_db_input(zen_db_prepare_input($main_page)) . "'";
		}
		
		if (is_null($query_string_parameters) || strlen($query_string_parameters) == 0) {
			$query_string_parameters_quoted = 'NULL';
		} else {
			$query_string_parameters_quoted =
				"'" . zen_db_input(zen_db_prepare_input($query_string_parameters)) . "'";
		}
		
		if (is_null($associated_db_id) || (int) $associated_db_id <= 0) {
			$associated_db_id_quoted = 'NULL';
		} else {
			$associated_db_id_quoted = "'" . (int) $associated_db_id . "'";
		}
		
		if (is_null($alternate_uri) || strlen($alternate_uri) == 0) {
			$alternate_uri_quoted = 'NULL';
		} else {
			$alternate_uri_quoted = "'" . zen_db_input(zen_db_prepare_input($alternate_uri)) . "'";
		}
		
		if (is_null($redirection_type_code) || (int) $redirection_type_code <= 0) {
			$redirection_type_code_quoted = 'NULL';
		} else {
			$redirection_type_code_quoted = "'" . (int) $redirection_type_code . "'";
		}
		
		// Language ID can only be null when the mapping is a redirect to an alternate URI
		if ($language_id_quoted == 'NULL' && $alternate_uri_quoted == 'NULL') {
			return CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR;
		}
		
		if ($avoid_duplicates && $alternate_uri_quoted == 'NULL') {
			// Make sure no current mapping already exists for the specified URI (don't bother with checks for
			// mappings to alternate URIs)
			$check_exists_sql = "
				SELECT
					language_id
				FROM
					" . TABLE_CEON_URI_MAPPINGS . "
				WHERE
					uri = " . $uri_quoted . "
				AND
					language_id = " . $language_id_quoted . "
				AND
					current_uri = '1';";
			
			$check_exists_result = $db->Execute($check_exists_sql);
			
			if (!$check_exists_result->EOF) {
				return CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS;
			}
		}
		
		$date_added_quoted = "'" . date('Y-m-d H:i:s') . "'";
		
		$sql = "
			INSERT INTO
				" . TABLE_CEON_URI_MAPPINGS . "
				(
				uri,
				language_id,
				current_uri,
				main_page,
				query_string_parameters,
				associated_db_id,
				alternate_uri,
				redirection_type_code,
				date_added
				)
			VALUES
				(
				" . $uri_quoted . ",
				" . $language_id_quoted . ",
				" . $current_uri_quoted . ",
				" . $main_page_quoted . ",
				" . $query_string_parameters_quoted . ",
				" . $associated_db_id_quoted . ",
				" . $alternate_uri_quoted . ",
				" . $redirection_type_code_quoted . ",
				" . $date_added_quoted . "
				);";
		
		$db->Execute($sql);
		
		return CEON_URI_MAPPING_ADD_MAPPING_SUCCESS;
	}
	
	// }}}
	
	
	// {{{ makeURIMappingHistorical()
	
	/**
	 * Makes the URI mapping a historical mapping. Use of this method abstracts the calling code from the actual
	 * implementation of the database's structure.
	 *
	 * @access  public
	 * @param   string    $uri   The URI of the mapping.
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping.
	 * @param   boolean   $avoid_duplicates   Whether or not to prevent the creation of duplicate historical
	 *                                        mappings. Should only ever be disabled if checks have already been
	 *                                        made and should therefore be skipped here for efficiency.
	 * @return  integer   A positive value if the mapping was made historical, a negative integer (constant value)
	 *                    if an error was encountered.
	 */
	public function makeURIMappingHistorical($uri, $language_id, $avoid_duplicates = true)
	{
		global $db;
		
		if (is_null($uri) || strlen($uri) == 0) {
			// Cannot be null!
			return CEON_URI_MAPPING_MAKE_MAPPING_HISTORICAL_ERROR_DATA_ERROR;
		} else {
			$uri_quoted = "'" . zen_db_input(zen_db_prepare_input($uri)) . "'";
		}
		
		if (is_null($language_id) || (int) $language_id <= 0) {
			// Cannot be null!
			return CEON_URI_MAPPING_MAKE_MAPPING_HISTORICAL_ERROR_DATA_ERROR;
		} else {
			$language_id_quoted = "'" . (int) $language_id . "'";
		}
		
		if ($avoid_duplicates) {
			// Make sure no historical mapping already exists for the specified URI
			$check_exists_sql = "
				SELECT
					language_id
				FROM
					" . TABLE_CEON_URI_MAPPINGS . "
				WHERE
					uri = " . $uri_quoted . "
				AND
					language_id = " . $language_id_quoted . "
				AND
					current_uri = '0';";
			
			$check_exists_result = $db->Execute($check_exists_sql);
			
			if (!$check_exists_result->EOF) {
				$selections = array(
					'uri' => $uri,
					'language_id' => $language_id,
					'current_uri' => 0
					);
				
				$this->deleteURIMappings($selections);
			}
		}
		
		$sql = "
			UPDATE
				" . TABLE_CEON_URI_MAPPINGS . "
			SET
				current_uri = '0'
			WHERE
				uri = " . $uri_quoted . "
			AND
				language_id = " . $language_id_quoted . ";";
		
		$db->Execute($sql);
		
		return CEON_URI_MAPPING_MAKE_MAPPING_HISTORICAL_SUCCESS;
	}
	
	// }}}
	
	
	// {{{ deleteURIMappings()
	
	/**
	 * Deletes any URI mappings matching the specified criteria. Use of this method abstracts the calling code from
	 * the actual implementation of the database's structure.
	 *
	 * @access  public                  Made public because of previous uses in functions type files.  Can stay
	 *                                    protected when call is in an observer provided the observer class
	 *                                    ultimately extends this class.
	 * @param   array     $selections   An associative array of column names and values to match for these columns.
	 *                                  A set of values can be grouped with OR by specifying an array of values for
	 *                                  the value.
	 * @return  none
	 */
	public function deleteURIMappings($selections)
	{
		global $db;
		
		$selection_string = '';
		
		$num_selection_columns = count($selections);
		
		$column_name_i = 0;
		
		foreach ($selections as $column_name => $column_value) {
			if (is_array($column_value)) {
				// The value is an array of values so create an ORed group
				$num_column_values = count($column_value);
				
				$selection_string .= '(' . "\n";
				
				for ($column_value_i = 0; $column_value_i < $num_column_values; $column_value_i++) {
					$selection_string .= "\t" . $column_name;
					
					$value = $column_value[$column_value_i];
					
					if (is_null($value) || strtolower($value) == 'null') {
						$selection_string .= " IS NULL\n";
					} else if (strtolower($value) == 'not null') {
						$selection_string .= " IS NOT NULL\n";
					} else {
						if (substr($value, -1) == '%') {
							$selection_string .= ' LIKE ';
						} else {
							$selection_string .= ' = ';
						}
						
						$selection_string .= "'" . zen_db_input($value) . "'\n";
					}
					
					if ($column_value_i < ($num_column_values - 1)) {
						$selection_string .= "OR\n";
					}
				}
				
				$selection_string .= ')' . "\n";
			} else {
				$selection_string .= "\t" . $column_name;
				
				if (is_null($column_value) || strtolower($column_value) == 'null') {
					$selection_string .= " IS NULL\n";
				} else if (strtolower($column_value) == 'not null') {
					$selection_string .= " IS NOT NULL\n";
				} else {
					if (substr($column_value, -1) == '%') {
						$selection_string .= ' LIKE ';
					} else {
						$selection_string .= ' = ';
					}
					
					$selection_string .= "'" . zen_db_input($column_value) . "'\n";
				}
			}
			
			if ($column_name_i < ($num_selection_columns - 1)) {
				$selection_string .= "AND\n";
			}
			
			$column_name_i++;
		}
		
		$sql = "
			DELETE FROM
				" . TABLE_CEON_URI_MAPPINGS . "
			WHERE
				" . $selection_string . ";";
		
		$db->Execute($sql);
	}
	
	// }}}
	
	
	// {{{ _autogenEnabled()
	
	/**
	 * Checks whether auto-generation of URIs is enabled or disabled.
	 *
	 * @access  protected
	 * @return  boolean   Whether auto-generation of URIs is enabled or disabled.
	 */
	protected function _autogenEnabled()
	{
		return $this->_autogen_new;
	}
	
	// }}}
	
	
	// {{{ _autoLanguageCodeAdd()
	
	/**
	 * Checks whether or not the site's settings dictate to prepend the language code to the URI.
	 *
	 * @access  protected
	 * @return  boolean   Whether or not a unique URI Mapping should be autogenerated by appending an integer to a
	 *                    clashing mapping.
	 */
	protected function _autoLanguageCodeAdd()
	{
		return $this->_language_code_add;
	}
	
	// }}}
	
	
	// {{{ _mappingClashAutoAppendInteger()
	
	/**
	 * Checks whether or not the site's settings dictate that when a mapping clashes, an attempt should be made to
	 * autogenerate a new, unique URI mapping by appending an integer to the clashing mapping.
	 *
	 * @access  protected
	 * @return  boolean   Whether or not a unique URI Mapping should be autogenerated by appending an integer to a
	 *                    clashing mapping.
	 */
	protected function _mappingClashAutoAppendInteger()
	{
		if ($this->_mapping_clash_action == 'auto-append') {
			return true;
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _convertStringForURI()
	
	/**
	 * Converts a string to a URI format, capitalising the words within as specified in the configuration, removing
	 * any unwanted words and replacing any specific character strings, transliterating it from a foreign language
	 * if necessary, then applying the whitespace replacement preference for the site.
	 *
	 * @access  protected
	 * @param   string    $text            The string to be converted to a URI part.
	 * @param   string    $language_code   The language code of the string to be converted.
	 * @param   boolean   $strip_slashes   Whether or not to strip foward slashes from the string.
	 * @return  string    The string as converted for use in a URI.
	 */
	protected function _convertStringForURI($text, $language_code, $strip_slashes = true)
	{
		//global $db;
		
		/**
		 * Load in the CeonString class to handle multibyte characters and transliteration.
		 */
		require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonString.php');
		
		
		// If the remove words and character string replacement settings have not yet been parsed,
		// parse them now
		if (!is_array($this->_remove_words)) {
			// Remove any spaces from between the words
			$this->_remove_words = CeonString::regexpReplace($this->_remove_words, '/\s/', '');
			
			// Escape any special characters in the selections of words
			$this->_remove_words = CeonString::regexpReplace($this->_remove_words,
				'/([\/\-\\\!\?\$\^\[\]\|\*\.\(\)\{\}\=\<\>\:\+])/U', '\\\$1');
			
			// Get the list of escaped words - @TODO is this multibyte safe?
			$this->_remove_words = explode(',', $this->_remove_words);
			
			if (count($this->_remove_words) == 1 && (is_null($this->_remove_words[0]) ||
					strlen($this->_remove_words[0]) == 0)) {
				$this->_remove_words = array();
			}
			
			$char_str_replacement_pairs = explode(',', $this->_char_str_replacements);
			
			$this->_char_str_replacements = array();
			
			for ($i = 0, $n = count($char_str_replacement_pairs); $i < $n; $i++) {
				$current_char_str_replacement = explode('=>', $char_str_replacement_pairs[$i]);
				
				if (count($current_char_str_replacement) == 2) {
					// Escape any special characters in the string to be replaced
					$current_char_str_replacement[0] = CeonString::regexpReplace($current_char_str_replacement[0],
						'/([\/\-\\\!\?\$\^\[\]\|\*\.\(\)\{\}\=\<\>\:\+])/U', '\\\$1');
					
					// Remove any spaces surrounding the string to be replaced
					$current_char_str_replacement[0] = trim($current_char_str_replacement[0]);
					
					$this->_char_str_replacements[] = array(
						'char_str' => $current_char_str_replacement[0],
						'replacement' => $current_char_str_replacement[1]
						);
				}
			}
		}
		
		// Convert the case of the string according to the configuration setting
		switch ($this->_capitalisation) {
			case CEON_URI_MAPPING_CAPITALISATION_LOWERCASE:
				$text = CeonString::toLowercase($text);
				break;
			case CEON_URI_MAPPING_CAPITALISATION_UCFIRST:
				$text = CeonString::toUCWords($text);
				break;
		}
		
		// Remove unwanted words
		if (count($this->_remove_words) > 0) {
			// Build a pattern to remove words surrounded by spaces
			$remove_words_pattern = '';
			
			foreach ($this->_remove_words as $remove_word) {
				// Set word to be removed in the pattern by wrapping it with spaces
				$remove_words_pattern .= '\s' . $remove_word . '\s|';
			}
			
			$remove_words_pattern =
				CeonString::substr($remove_words_pattern, 0, CeonString::length($remove_words_pattern) - 1);
			
			// Remove the words surrounded by spaces, replacing them with a single space
			$text = CeonString::regexpReplace($text, '/' . $remove_words_pattern . '/i', ' ');
			
			
			// Build a pattern to remove words at the start of the string
			$remove_words_pattern = '';
			
			foreach ($this->_remove_words as $remove_word) {
				// Set word to be removed in the pattern by wrapping it with spaces
				$remove_words_pattern .= '^' . $remove_word . '\s|';
			}
			
			$remove_words_pattern =
				CeonString::substr($remove_words_pattern, 0, CeonString::length($remove_words_pattern) - 1);
			
			// Remove any word which is at the start of the string
			$text = CeonString::regexpReplace($text, '/' . $remove_words_pattern . '/i', '');
			
			
			// Build a pattern to remove any word which at the end of the string
			$remove_words_pattern = '';
			
			foreach ($this->_remove_words as $remove_word) {
				// Set word to be removed in the pattern by wrapping it with spaces
				$remove_words_pattern .= '\s' . $remove_word . '$|';
			}
			
			$remove_words_pattern =
				CeonString::substr($remove_words_pattern, 0, CeonString::length($remove_words_pattern) - 1);
			
			// Remove any word which is at the end of the string
			$text = CeonString::regexpReplace($text, '/' . $remove_words_pattern . '/i', '');
		}
		
		// Replace specified characters/strings
		if (count($this->_char_str_replacements) > 0) {
			foreach ($this->_char_str_replacements as $char_str_replacement) {
				// Remove the words surrounded by spaces, replacing them with a single space
				$text = CeonString::regexpReplace($text, '/' . $char_str_replacement['char_str'] . '/i',
					$char_str_replacement['replacement']);
			}
		}
		
		// Convert certain characters in the name to spaces, rather than having the words separated by these
		// characters being joined together when they are removed later
		if ($strip_slashes) {
			$pattern = '/[\|\/\+_:;\(\)\[\]\<\>,]/';
		} else {
			$pattern = '/[\|\+_:;\(\)\[\]\<\>,]/';
		}
		
		$text = preg_replace($pattern, ' ', $text);
		
		// Convert the string to English ASCII as that's all that's permitted in URIs
		$text = CeonString::transliterate($text, CHARSET, $language_code);
		
		// Remove illegal characters
		if ($strip_slashes) {
			$pattern = '/[^a-zA-Z0-9\.\-_\s]/';
		} else {
			$pattern = '/[^a-zA-Z0-9\.\-_\s\/]/';
		}
		
		$text = preg_replace($pattern, '', $text);
		
		// Convert double whitespace/tabs etc. to single space
		$text = preg_replace('/\s+/', ' ', $text);
		
		// Remove any starting or trailing whitespace
		$text = trim($text);
		
		// Once again convert the case of the string according to the configuration setting. 
		// This must be repeated as the transliteration could have left some letters in place which may need to be
		// converted. As the string is now ASCII, a simple conversion can be used.
		switch ($this->_capitalisation) {
			case CEON_URI_MAPPING_CAPITALISATION_LOWERCASE:
				$text = strtolower($text);
				break;
			case CEON_URI_MAPPING_CAPITALISATION_UCFIRST:
				$text = ucwords($text);
				break;
		}
		
		// Convert whitespace to configured character (or remove it)
		switch ($this->_whitespace_replacement) {
			case CEON_URI_MAPPING_SINGLE_UNDERSCORE:
				$whitespace_replacement_char = '_';
				break;
			case CEON_URI_MAPPING_SINGLE_DASH:
				$whitespace_replacement_char = '-';
				break;
			case CEON_URI_MAPPING_SINGLE_FULL_STOP:
				$whitespace_replacement_char = '.';
				break;
			case CEON_URI_MAPPING_REMOVE:
				$whitespace_replacement_char = '';
				break;
		}
		
		$text = preg_replace('/\s/', $whitespace_replacement_char, $text);
		
		return $text;
	}
	
	// }}}
	
	
	// {{{ _cleanUpURIMapping()
	
	/**
	 * Ensures that a URI matches the format used by the URI mapping functionality.
	 *
	 * @access  protected
	 * @param   string    $uri   The URI to be checked/cleaned.
	 * @return  string    The checked/cleaned URI.
	 */
	protected function _cleanUpURIMapping($uri)
	{
		// Remove any starting or trailing whitespace
		$uri = trim($uri);
		
		// Replace any backslashes with forward slashes
		$uri = str_replace('\\', '/', $uri);
		
		// Remove illegal characters
		$uri = preg_replace('|[^a-zA-Z0-9\.\-_\/]|', '', $uri);
		
		// Remove any domain specification as all URIs must be relative to the site's root
		$uri = preg_replace('|^http:\/\/[^\/]+|iU', '', $uri);
		
		// Get rid of any double slashes
		while (strpos($uri, '//') !== false) {
			$uri = str_replace('//', '/', $uri);
		}
		
		if (strlen($uri) == 0) {
			return '';
		}
		
		// Prepend the URI with a root slash
		while (substr($uri, 0, 1) == '/') {
			$uri = substr($uri, 1, strlen($uri) - 1);
		}
		
		$uri = '/' . $uri;
		
		// Remove any trailing slashes
		while (substr($uri, -1) == '/') {
			$uri = substr($uri, 0, strlen($uri) - 1);
		}
		
		return $uri;
	}
	
	// }}}
	
	
	// {{{ _validateURIMappingTemplate()
	
	/**
	 * Checks the supplied text to see if it is formatted correctly for use as a mapping template.
	 *
	 * @access  protected
	 * @param   string    $object_type   The type of object the mapping template is intended for.
	 * @param   string    $template      The template string to be checked.
	 * @return  boolean|array   True if the template is valid, an array of error messages if not.
	 */
	protected function _validateURIMappingTemplate($object_type, $template)
	{
		$errors = array();
		
		$this->_loadLanguageDefinitions();
		
		// Get all the placement tags in the template
		$placement_tags = array();
		
		if (!preg_match_all('/\{([^\}]+)\}/iU', $template, $matches, PREG_SET_ORDER)) {
			// Template must have at least one placement tag!
			$errors[] = TEXT_ERROR_NO_PLACEMENT_TAGS_FOUND_IN_TEMPLATE;
			
			return $errors;
		}
		
		// Check for corrupted placement tags
		$num_opening_brackets = substr_count($template, '{');
		$num_closing_brackets = substr_count($template, '}');
		
		$num_unclosed_brackets = $num_opening_brackets - $num_closing_brackets;
		$num_unopened_brackets = $num_closing_brackets - $num_opening_brackets;
		
		if ($num_unclosed_brackets < 0) {
			$num_unclosed_brackets = 0;
		}
		
		if ($num_unopened_brackets < 0) {
			$num_unopened_brackets = 0;
		}
		
		if ($num_unclosed_brackets == 1) {
			$errors[] = TEXT_ERROR_AN_OPENING_BRACKET_WITHOUT_CLOSING;
		} else if ($num_unclosed_brackets > 0) {
			$errors[] = sprintf(TEXT_ERROR_X_OPENING_BRACKETS_WITHOUT_CLOSING,
				$num_unclosed_brackets);
		} else if ($num_unopened_brackets == 1) {
			$errors[] = TEXT_ERROR_A_CLOSING_BRACKET_WITHOUT_OPENING;
		} else if ($num_unopened_brackets > 0) {
			$errors[] = sprintf(TEXT_ERROR_X_CLOSING_BRACKETS_WITHOUT_OPENING,
				$num_unopened_brackets);
		}
		
		// Now check if any of the placement tags are invalid
		switch ($object_type) {
			case 'category':
				$valid_placement_tags = array(
					'dir-ws-catalog',
					'category-path',
					'category-name',
					'category-id',
					'language-name',
					'language-code',
					'language-directory'
					);
				break;
			case 'product':
				$valid_placement_tags = array(
					'dir-ws-catalog',
					'category-path',
					'master-category-name',
					'manufacturer-name',
					'product-name',
					'product-model',
					'product-id',
					'language-name',
					'language-code',
					'language-directory'
					);
				break;
			case 'manufacturer':
				$valid_placement_tags = array(
					'dir-ws-catalog',
					'manufacturer-name',
					'manufacturer-id'
					);
				break;
			case 'ez-page':
				$valid_placement_tags = array(
					'dir-ws-catalog',
					'ez-page-name',
					'ez-page-id',
					'language-name',
					'language-code',
					'language-directory'
					);
				break;
			case 'other-page':
				$valid_placement_tags = array(
					'dir-ws-catalog',
					'page-define-text',
					'language-name',
					'language-code',
					'language-directory'
					);
				break;
		}
		
		// Track if any invalid placement tags are used so that helpful information about the usable placement tags
		// can be displayed to the user
		$invalid_placement_tag_found = false;
		
		foreach ($matches as $match) {
			$placement_tag = strtolower($match[1]);
			
			// Handle placement tags with dynamic info
			if (substr($placement_tag, 0, 13) == 'category-path') {
				$placement_tag = 'category-path';
			}
			
			if (!in_array($placement_tag, $valid_placement_tags)) {
				$invalid_placement_tag_found = true;
				
				$errors[] = sprintf(TEXT_ERROR_INVALID_PLACEMENT_TAG, $placement_tag);
			} else {
				$placement_tags[] = $placement_tag;
			}
		}
		
		if ($invalid_placement_tag_found) {
			// Display information about the valid tags that can be used for the specified mapping
			// template type
			switch ($object_type) {
				case 'category':
					$errors[] = $this->_addTooltipDescriptionsOfPlacementTags('category',
						TEXT_VALID_PLACEMENT_TAGS_FOR_CATEGORY_MAPPING_TEMPLATE);
					break;
				case 'product':
					$errors[] = $this->_addTooltipDescriptionsOfPlacementTags('product',
						TEXT_VALID_PLACEMENT_TAGS_FOR_PRODUCT_MAPPING_TEMPLATE);
					break;
				case 'manufacturer':
					$errors[] = $this->_addTooltipDescriptionsOfPlacementTags('manufacturer',
						TEXT_VALID_PLACEMENT_TAGS_FOR_MANUFACTURER_MAPPING_TEMPLATE);
					break;
				case 'ez-page':
					$errors[] = $this->_addTooltipDescriptionsOfPlacementTags('ez-page',
						TEXT_VALID_PLACEMENT_TAGS_FOR_EZ_PAGE_MAPPING_TEMPLATE);
					break;
				case 'other-page':
					$errors[] = $this->_addTooltipDescriptionsOfPlacementTags('other-page',
						TEXT_VALID_PLACEMENT_TAGS_FOR_OTHER_PAGE_MAPPING_TEMPLATE);
					break;
			}
		}
			
		if (sizeof($errors) == 0) {
			// Make sure that a distinguishing template is being used, otherwise the template is a bit dim
			if (sizeof($placement_tags) == 0) {
				$errors[] = TEXT_ERROR_MAPPING_TEMPLATE_MUST_HAVE_PLACEMENT_TAGS;
			} else {
				$unique_uri_generating_placement_tag_found = false;
				
				foreach ($placement_tags as $placement_tags) {
					if ($placement_tags != 'dir-ws-catalog') {
						$unique_uri_generating_placement_tag_found = true;
						break;
					}
				}
					
				if (!$unique_uri_generating_placement_tag_found) {
					$errors[] = TEXT_ERROR_MAPPING_TEMPLATE_MUST_NOT_JUST_USE_DIR_WS_CATALOG_TAG;
				}
			}
		}
		
		return (sizeof($errors) > 0 ? $errors : true);
	}
	
	// }}}
	
	
	// {{{ _loadLanguageDefinitions()
	
	/**
	 * Loads the language definitions used by this class. Called as a method so that they are only loaded when
	 * necessary, for efficiency.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _loadLanguageDefinitions()
	{
		// Only attempt to load the file once
		if (!isset($language_definitions_load_attempt)) {
			static $language_definitions_load_attempt = true;
		} else {
			return;
		}
		
		// Load the language definition file for the current language
		@include_once(DIR_FS_CATALOG . DIR_WS_LANGUAGES . $_SESSION['language'] . '/' .
			'ceon_uri_mapping_admin.php');
		
		if (!defined('TEXT_ERROR_NO_PLACEMENT_TAGS_FOUND_IN_TEMPLATE') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_FS_CATALOG . DIR_WS_LANGUAGES . 'english/' . 'ceon_uri_mapping_admin.php');
		}
	}
	
	// }}}
	
	
	// {{{ _addTooltipDescriptionsOfPlacementTags()
	
	/**
	 * Convenience method which adds HTML tags to the given string so that the user can see a description of the
	 * various placement tags within. Using this one method means that the  descriptive text can be changed
	 * centrally at any time.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _addTooltipDescriptionsOfPlacementTags($object_type, $text)
	{
		$this->_loadLanguageDefinitions();
		
		$text = str_replace('{dir-ws-catalog}', TEXT_PLACEMENT_TAG_TOOLTIP_DIR_WS_CATALOG, $text);
		
		switch ($object_type) {
			case 'category':
				$text = str_replace('{category-path}', TEXT_PLACEMENT_TAG_TOOLTIP_CATEGORY_CATEGORY_PATH, $text);
				
				$text = str_replace('{category-name}', TEXT_PLACEMENT_TAG_TOOLTIP_CATEGORY_NAME, $text);
				
				$text = str_replace('{category-id}', TEXT_PLACEMENT_TAG_TOOLTIP_CATEGORY_ID, $text);
				
				$text = str_replace('{language-name}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_NAME, $text);
				
				$text = str_replace('{language-code}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_CODE, $text);
				
				$text = str_replace('{language-directory}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_DIRECTORY, $text);
				
				break;
			
			case 'product':
				$text = str_replace('{category-path}', TEXT_PLACEMENT_TAG_TOOLTIP_PRODUCT_CATEGORY_PATH, $text);
				
				$text = str_replace('{manufacturer-name}', TEXT_PLACEMENT_TAG_TOOLTIP_PRODUCT_MANUFACTURER_NAME,
					$text);
				
				$text = str_replace('{product-name}', TEXT_PLACEMENT_TAG_TOOLTIP_PRODUCT_NAME, $text);
				
				$text = str_replace('{product-model}', TEXT_PLACEMENT_TAG_TOOLTIP_PRODUCT_MODEL, $text);
				
				$text = str_replace('{product-id}', TEXT_PLACEMENT_TAG_TOOLTIP_PRODUCT_ID, $text);
				
				$text = str_replace('{language-name}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_NAME, $text);
				
				$text = str_replace('{language-code}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_CODE, $text);
				
				$text = str_replace('{language-directory}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_DIRECTORY, $text);
				
				break;
			
			case 'manufacturer':
				$text = str_replace('{manufacturer-name}', TEXT_PLACEMENT_TAG_TOOLTIP_MANUFACTURER_NAME, $text);
				
				$text = str_replace('{manufacturer-id}', TEXT_PLACEMENT_TAG_TOOLTIP_MANUFACTURER_ID, $text);
				
				break;
			
			case 'ez-page':
				$text = str_replace('{ez-page-name}', TEXT_PLACEMENT_TAG_TOOLTIP_EZ_PAGE_NAME, $text);
				
				$text = str_replace('{ez-page-id}', TEXT_PLACEMENT_TAG_TOOLTIP_EZ_PAGE_ID, $text);
				
				$text = str_replace('{language-name}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_NAME, $text);
				
				$text = str_replace('{language-code}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_CODE, $text);
				
				$text = str_replace('{language-directory}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_DIRECTORY, $text);
				
				break;
			
			case 'other-page':
				$text =
					str_replace('{page-define-text}', TEXT_PLACEMENT_TAG_TOOLTIP_OTHER_PAGE_DEFINE_TEXT, $text);
				
				$text = str_replace('{language-name}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_NAME, $text);
				
				$text = str_replace('{language-code}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_CODE, $text);
				
				$text = str_replace('{language-directory}', TEXT_PLACEMENT_TAG_TOOLTIP_LANGUAGE_DIRECTORY, $text);
				
				break;
		}
		
		return $text;
	}
	
	// }}}
	
	
	// {{{ _cleanUpURIMappingTemplate()
	
	/**
	 * Ensures that a template matches the format used by the URI mappings manager functionality.
	 *
	 * @access  protected
	 * @param   string    $template   The template to be cleaned.
	 * @return  string    The cleaned template.
	 */
	protected function _cleanUpURIMappingTemplate($template)
	{
		// Remove any starting or trailing whitespace
		$template = trim($template);
		
		// Replace any backslashes with forward slashes
		$template = str_replace('\\', '/', $template);
		
		// Remove illegal characters
		$template = preg_replace('|[^a-zA-Z0-9\.\-_\/\{\}]|', '', $template);
		
		// Get rid of any double slashes
		while (strpos($template, '//') !== false) {
			$template = str_replace('//', '/', $template);
		}
		
		// Remove any trailing slashes
		while (substr($template, -1) == '/') {
			$template = substr($template, 0, strlen($template) - 1);
		}
		
		if (strlen($template) == 0) {
			return '';
		}
		
		// Prepend the URI with a root slash
		while (substr($template, 0, 1) == '/') {
			$template = substr($template, 1, strlen($template) - 1);
		}
		
		$template = '/' . $template;
		
		return $template;
	}
	
	// }}}
	
	
	// {{{ getURIMappingTemplatesResultset()
	
	/**
	 * Performs a query against the URI mapping templates database. Use of this method abstracts the calling code
	 * from the actual implementation of the database's structure.
	 *
	 * @access  public
	 * @param   string    $object_type   The type of object the mapping template is for.
	 * @param   array|string   $columns_to_retrieve   The column(s) to retrieve. Either an array of column names
	 *                                                or a single column name.
	 * @param   array     $selections   An associative array of column names and values to match for these columns.
	 *                                  A set of values can be grouped with OR by specifying an array of values for
	 *                                  the value.
	 * @param   string    $order_by   An SQL string to be used to order the resultset.
	 * @param   string    $limit      An SQL string to be used to limit the resultset.
	 * @param   string    $group_by   An SQL string to be used to group the resultset.
	 * @return  resultset   A Zen Cart database resultset.
	 */
	public function getURIMappingTemplatesResultset($object_type, $columns_to_retrieve, $selections, $order_by = null,
		$limit = null, $group_by = null)
	{
		global $db;
		
		if (is_array($columns_to_retrieve)) {
			$columns_to_retrieve = implode(', ', $columns_to_retrieve);
		}
		
		$selection_string = '';
		
		$num_selection_columns = sizeof($selections);
		
		$column_name_i = 0;
		
		foreach ($selections as $column_name => $column_value) {
			if (is_array($column_value)) {
				// The value is an array of values so create an ORed group
				$num_column_values = sizeof($column_value);
				
				$selection_string .= '(' . "\n";
				
				for ($column_value_i = 0; $column_value_i < $num_column_values; $column_value_i++) {
					$selection_string .= "\t" . $column_name;
					
					$value = $column_value[$column_value_i];
					
					if (is_null($value) || strtolower($value) == 'null') {
						$selection_string .= " IS NULL\n";
					} else if (strtolower($value) == 'not null') {
						$selection_string .= " IS NOT NULL\n";
					} else {
						if (substr($value, -1) == '%') {
							$selection_string .= ' LIKE ';
						} else {
							$selection_string .= ' = ';
						}
						
						$selection_string .= "'" . zen_db_input($value) . "'\n";
					}
					
					if ($column_value_i < ($num_column_values - 1)) {
						$selection_string .= "OR\n";
					}
				}
				
				$selection_string .= ')' . "\n";
				
			} else {
				$selection_string .= "\t" . $column_name;
				
				if (is_null($column_value) || strtolower($column_value) == 'null') {
					$selection_string .= " IS NULL\n";
				} else if (strtolower($column_value) == 'not null') {
					$selection_string .= " IS NOT NULL\n";
				} else {
					if (substr($column_value, -1) == '%') {
						$selection_string .= ' LIKE ';
					} else {
						$selection_string .= ' = ';
					}
					
					$selection_string .= "'" . zen_db_input($column_value) . "'\n";
				}
			}
			
			if ($column_name_i < ($num_selection_columns - 1)) {
				$selection_string .= "AND\n";
			}
			
			$column_name_i++;
		}
		
		switch ($object_type) {
			case 'category':
				$table = TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES;
				break;
			case 'product':
				$table = TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES;
				break;
			case 'manufacturer':
			case 'ez-page':
			case 'other-page':
				$table = TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES;
				break;
		}
		
		$sql = "
			SELECT
				" . $columns_to_retrieve . "
			FROM
				" . $table . "
			WHERE
				" . $selection_string;
		
		if (!is_null($order_by)) {
			$sql .= "\n" . 'ORDER BY ' . $order_by;
		}
		
		if (!is_null($limit)) {
			$sql .= "\n" . 'LIMIT ' . $limit;
		}
		
		if (!is_null($group_by)) {
			$sql .= "\n" . 'GROUP_BY ' . $group_by;
		}
		
		$sql .= ';';
		
		return $db->Execute($sql);
	}
	
	// }}}
	
	
	// {{{ getObjectURIMappingTemplate()
	
	/**
	 * Looks up the mapping template for the specified object type and language. Use of this method abstracts the
	 * calling code from the actual implementation of the database's structure.
	 *
	 * @access  public
	 * @param   string    $object_type   The type of object the mapping template is for.
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping template.
	 * @param   boolean   $use_cache     Whether or not the cache should be used.
	 * @return  string    The mapping template.
	 */
	public function getObjectURIMappingTemplate($object_type, $language_id, $use_cache = true)
	{
		if ($use_cache) {
			if (!isset($GLOBALS['ceon-uri-mapping-templates-' . $object_type])) {
				// Cache not initialised yet, initialise it now
				$GLOBALS['ceon-uri-mapping-templates-' . $object_type] = array();
			} else if (isset($GLOBALS['ceon-uri-mapping-templates-' . $object_type]
					[$language_id])) {
				// Mapping is already in cache, return it!
				return $GLOBALS['ceon-uri-mapping-templates-' . $object_type][$language_id];
			}
		}
		
		$columns_to_retrieve = array(
			'mapping_template'
			);
		
		$selections = array(
			'page_type' => $object_type,
			'language_id' => $language_id
			);
		
		$mapping_template_result = $this->getURIMappingTemplatesResultset($object_type, $columns_to_retrieve,
			$selections, null, 1);
		
		if (!$mapping_template_result->EOF) {
			if ($use_cache) {
				// Add this template to the cache
				$GLOBALS['ceon-uri-mapping-templates-' . $object_type][$language_id] =
					$mapping_template_result->fields['mapping_template'];
			}
			
			return $mapping_template_result->fields['mapping_template'];
		}
	}
	
	// }}}
	
	
	// {{{ addUpdateURIMappingTemplate()
	
	/**
	 * Adds or updates a mapping template to/in the database. Use of this method abstracts the calling code from
	 * the actual implementation of the database's structure.
	 *
	 * @access  public
	 * 
	 * @param   string    $template      The mapping template.
	 * @param   string    $object_type   The type of object the mapping template is for.
	 * @param   integer   $object_id     The ID of the object the mapping template is for.
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping template.
	 * @return  integer|false   A number representing the template having been inserted or updated or left
	 *                          unchanged, or false if a problem occurred.
	 */
	public function addUpdateURIMappingTemplate($template, $object_type, $object_id, $language_id)
	{
		global $db;
		
		if (is_null($template) || strlen($template) == 0) {
			// Cannot be null!
			return false;
		} else {
			$template_quoted = "'" . zen_db_input(zen_db_prepare_input($template)) . "'";
		}
		
		if (is_null($object_type) || ($object_type != 'category' && $object_type != 'product' &&
				$object_type != 'manufacturer' && $object_type != 'ez-page' && $object_type != 'other-page')) {
			return false;
		}
		
		if ($object_type == 'category' || $object_type == 'product') {
			if (is_null($object_id) || $object_id < 0) {
				return false;
			} else {
				$object_id_quoted = "'" . (int) $object_id . "'";
			}
		} else {
			if (is_null($object_id) || strlen($object_id) == 0) {
				return false;
			} else {
				$object_id_quoted = "'" . $object_id . "'";
			}
		}
		
		if (is_null($language_id) || (int) $language_id <= 0) {
			return false;
		} else {
			$language_id_quoted = "'" . (int) $language_id . "'";
		}
		
		switch ($object_type) {
			case 'category':
				$table = TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES;
				$object_id_column = 'category_id';
				break;
			case 'product':
				$table = TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES;
				$object_id_column = 'category_id';
				break;
			case 'manufacturer':
			case 'ez-page':
			case 'other-page':
				$table = TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES;
				$object_id_column = 'page_type';
				break;
		}
		
		$date_added_modified_quoted = "'" . date('Y-m-d H:i:s') . "'";
		
		// Find out if this is an insert or update
		$insert = true;
		
		$check_exists_sql = "
			SELECT
				mapping_template
			FROM
				" . $table . "
			WHERE
				" . $object_id_column . " = " . $object_id_quoted . "
			AND
				language_id = " . $language_id_quoted . ";";
		
		$check_exists_result = $db->Execute($check_exists_sql);
		
		if (!$check_exists_result->EOF) {
			$insert = false;
			
			if ($check_exists_result->fields['mapping_template'] == $template) {
				return CEON_URI_MAPPING_MAPPING_TEMPLATE_UNCHANGED;
			}
		}
		
		if ($insert) {
			$sql = "
				INSERT INTO
					" . $table . "
					(
					" . $object_id_column . ",
					language_id,
					mapping_template,
					date_added_modified
					)
				VALUES
					(
					" . $object_id_quoted . ",
					" . $language_id_quoted . ",
					" . $template_quoted . ",
					" . $date_added_modified_quoted . "
					);";
		} else {
			$sql = "
				UPDATE
					" . $table . "
				SET
					mapping_template = " . $template_quoted . ",
					date_added_modified = " . $date_added_modified_quoted . "
				WHERE
					" . $object_id_column . " = " . $object_id_quoted . "
				AND
					language_id = " . $language_id_quoted . ";";
		}
		
		$db->Execute($sql);
		
		return ($insert ? CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED : CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED);
	}
	
	// }}}
	
	
	// {{{ deleteURIMappingTemplate()
	
	/**
	 * Deletes a mapping template from the database. Use of this method abstracts the calling code from the actual
	 * implementation of the database's structure.
	 *
	 * @access  public
	 * @param   string    $object_type   The type of object the mapping template is for.
	 * @param   integer   $object_id     The ID of the object the mapping template is for.
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping template.
	 * @return  integer|false   A number representing the template having deleted or false if a problem occurred.
	 */
	public function deleteURIMappingTemplate($object_type, $object_id, $language_id)
	{
		global $db;
		
		if (is_null($object_type) || ($object_type != 'category' && $object_type != 'product')) {
			return false;
		}
		
		if (is_null($object_id) || $object_id < 0) {
			return false;
		} else {
			$object_id_quoted = "'" . (int) $object_id . "'";
		}
		
		if (is_null($language_id) || (int) $language_id <= 0) {
			return false;
		} else {
			$language_id_quoted = "'" . (int) $language_id . "'";
		}
		
		switch ($object_type) {
			case 'category':
				$table = TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES;
				$object_id_column = 'category_id';
				break;
			case 'product':
				$table = TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES;
				$object_id_column = 'category_id';
				break;
			case 'manufacturer':
			case 'ez-page':
			case 'other-page':
				$table = TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES;
				$object_id_column = 'page_type';
				break;
		}
		
		// If the template doesn't exist it doesn't need deleting!
		$check_exists_sql = "
			SELECT
				mapping_template
			FROM
				" . $table . "
			WHERE
				" . $object_id_column . " = " . $object_id . "
			AND
				language_id = " . $language_id . ";";
		
		$check_exists_result = $db->Execute($check_exists_sql);
		
		if ($check_exists_result->EOF) {
			return CEON_URI_MAPPING_MAPPING_TEMPLATE_DIDNT_EXIST;
		}
		
		$sql = "
			DELETE FROM
				" . $table . "
			WHERE
				" . $object_id_column . " = " . $object_id_quoted . "
			AND
				language_id = " . $language_id_quoted . ";";
		
		$db->Execute($sql);
		
		return CEON_URI_MAPPING_MAPPING_TEMPLATE_DELETED;
	}
	
	// }}}
}

// }}}
