<?php

/**
 * Ceon URI Mapping - UMM Edition - URI HREF Link Builder Class.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingHREFLinkBuilder.php 1054 2012-09-22 15:45:15Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdmin.php');


// {{{ CeonURIMappingHREFLinkBuilder

/**
 * Provides functionality to build a static URI for a Zen Cart page, for use in a HTML link's href
 * parameter.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingHREFLinkBuilder extends CeonURIMappingAdmin
{
	// {{{ properties
	
	/**
	 * The link built by this instance.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected  $_href_link = '';
	
	/**
	 * A mapping generated by this instance.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_autogenerated_mapping = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingHREFLinkBuilder class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ buildHREFLink()
	
	/**
	 * Builds a static URI-based link for a Zen Cart page, if a mapping exists for the page. The link is built in a
	 * format that can be directly included in a HTML link's href attribute.
	 *
	 * @access  public
	 * @param   string    $base_link    The base string for the link, as built by the standard Zen Cart 
	 *                                  zen_href_link() function.
	 * @param   string    $main_page    The Zen Cart page type the link is to.
	 * @param   string    $parameters   The query string parameters which are to be used to identify the specific
	 *                                  Zen Cart page and to build any additional parameters for that page.
	 * @param   string    $connection   The connection type the link is to use.
	 * @param   boolean   $add_session_id   Whether or not to add the session ID to the link.
	 * @return  boolean   Whether or not a static URI was successfully built for the Zen Cart page.
	 */
	public function buildHREFLink($base_link, $main_page, $parameters, $connection, $add_session_id)
	{
		global $request_type, $session_started, $http_domain, $https_domain, $ceon_uri_mapping_product_pages,
			$ceon_uri_mapping_product_related_pages;
		
		// Reset the link for this instance
		$this->_href_link = '';
		
		// Variable holds the main part of the link being built
		$link = '';
		
		if (preg_match('/.*index.php\?main_page=([^&]+)/i', $main_page, $matches)) {
			// Link isn't to a page's filename but to a specific URI (e.g. an EZ-Page alt url)
			// Extract the page ID and any parameters from the string
			$main_page = $matches[1];
			$page = ''; // @TODO determine if $page should equal $main_page
			$temp = str_replace($matches[0], '', $page);
			
			$parameters = $parameters . $temp;
		}
		
		if ($main_page == FILENAME_DEFAULT && $parameters != '' &&
				strpos(strtolower($parameters), 'manufacturers_id=') !== false) {
			// This link is to a manufacturer's page
			
			// Get the manufacturer ID
			$pattern = '/[&\?]?(manufacturers_id=([0-9]+))/i';
			
			if (preg_match($pattern, $parameters, $matches)) {
				$manufacturer_query_pair = $matches[1];
				$manufacturer_id = $matches[2];
				
				$manufacturer_parameter = 'manufacturers_id=' . $manufacturer_id;
				
				$columns_to_retrieve = array(
					'uri'
					);
				
				$selections = array(
					'main_page' => $main_page,
					'query_string_parameters' => $manufacturer_parameter,
					'language_id' => (int) $_SESSION['languages_id'],
					'current_uri' => 1
					);
				
				$uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
				
				if (!$uri_mapping_result->EOF) {
					$link = $uri_mapping_result->fields['uri'];
				} else if (defined('CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES') &&
						CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES == 1) {
					// No URI mapping exists for this manufacturer. Attempt to generate one now
					$link = $this->_autogenManufacturerURIMapping($manufacturer_id);
				}
				
				if ($link != '') {
					// Remove the manufacturer ID from the URI as it will be regenerated when the static URI is
					// loaded
					$parameters = str_replace($manufacturer_query_pair . '&', '', $parameters);
					$parameters = str_replace($manufacturer_query_pair, '', $parameters);
				}
			}
		} else if ($main_page == FILENAME_DEFAULT && $parameters != '' &&
				strpos(strtolower($parameters), 'typefilter=') !== false) {
			// This link is to a category page
			
			// Get the filter information
			$pattern = '/[&\?]?(typefilter=([^&]+))/i';
			
			if (preg_match($pattern, $parameters, $matches)) {
				$typefilter_name_query_pair = $matches[1];
				$typefilter_name = $matches[2];
				
				// Attempt to get ID for filter's property
				$pattern = '/[&\?]?(' . $typefilter_name . '_id=([^&]+))/i';
				
				if (preg_match($pattern, $parameters, $matches)) {
					$typefilter_id_query_pair = $matches[1];
					$typefilter_id = $matches[2];
					
					$typefilter_parameter = 'typefilter=' . $typefilter_name . '&' . $typefilter_name . '_id=' .
						$typefilter_id;
					
					$columns_to_retrieve = array(
						'uri'
						);
					
					$selections = array(
						'main_page' => $main_page,
						'query_string_parameters' => $typefilter_parameter,
						'language_id' => (int) $_SESSION['languages_id'],
						'current_uri' => 1
						);
					
					$uri_mapping_result =
						$this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
					
					if (!$uri_mapping_result->EOF) {
						$link = $uri_mapping_result->fields['uri'];
						
						// Remove the typefilter info from the URI as it will be regenerated when the static URI is
						// loaded
						$parameters = str_replace($typefilter_id_query_pair . '&', '', $parameters);
						$parameters = str_replace($typefilter_id_query_pair, '', $parameters);
						
						$parameters = str_replace($typefilter_name_query_pair . '&', '', $parameters);
						$parameters = str_replace($typefilter_name_query_pair, '', $parameters);
					}
				}
			}
		} else if ($main_page == FILENAME_DEFAULT && $parameters != '' &&
				strpos(strtolower($parameters), 'cpath=') !== false) {
			// This link is to a category page
			
			// Get the category ID
			$pattern = '/[&\?]?(cPath=([0-9\_]+))/i';
			
			if (preg_match($pattern, $parameters, $matches)) {
				$category_query_pair = $matches[1];
				$category_parts = explode('_', $matches[2]);
				
				// Category ID to be linked to is the last part of the specification
				$category_id = $category_parts[count($category_parts) - 1];
				
				$columns_to_retrieve = array(
					'uri'
					);
				
				$selections = array(
					'main_page' => $main_page,
					'associated_db_id' => (int) $category_id,
					'language_id' => (int) $_SESSION['languages_id'],
					'current_uri' => 1
					);
				
				$uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
				
				if (!$uri_mapping_result->EOF) {
					$link = $uri_mapping_result->fields['uri'];
				} else if (defined('CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES') &&
						CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES == 1) {
					// No URI mapping exists for this category. Attempt to generate one now
					$link = $this->_autogenCategoryURIMapping($category_id);
				}
				
				if ($link != '') {
					// Remove the category ID from the URI as it will be regenerated when the static URI is loaded
					$parameters = str_replace($category_query_pair . '&', '', $parameters);
					$parameters = str_replace($category_query_pair, '', $parameters);
				}
			}
		} else if ($main_page == FILENAME_DEFAULT && strpos(strtolower($parameters), 'typefilter=') === false) {
			// Link is to the index page.. no need to look up mapping in database, simply use the default index
			// page static URI
			$link = DIR_WS_CATALOG;
			
		} else if (in_array($main_page, $ceon_uri_mapping_product_pages) ||
				in_array($main_page, $ceon_uri_mapping_product_related_pages)) {
			// This link is to a product info page or one of the other pages related to the product
			
			// Get the product ID
			$pattern = '/[&\?]?(products_id=([0-9]+)[^&]*)/i';
			if ($main_page == FILENAME_ASK_A_QUESTION) {
				$pattern = '/[&\?]?(pid=([0-9]+)[^&]*)/i';
			}
			
			if (preg_match($pattern, $parameters, $matches)) {
				$product_query_pair = $matches[1];
				$product_id = $matches[2];
				
				$columns_to_retrieve = array(
					'uri'
					);
				
				$selections = array(
					'main_page' => $main_page,
					'associated_db_id' => (int) $product_id,
					'language_id' => (int) $_SESSION['languages_id'],
					'current_uri' => 1
					);
				
				$uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
				
				if (!$uri_mapping_result->EOF) {
					$link = $uri_mapping_result->fields['uri'];
				} else if (defined('CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES') &&
						CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES == 1) {
					if (in_array($main_page, $ceon_uri_mapping_product_pages)) {
						// No URI mapping exists for this product. Attempt to generate one now for it and any
						// auto-managed product related pages
						$link = $this->_autogenProductAndPRPURIMappings($product_id, $main_page);
					}
				}
				
				if ($link != '') {
					if (strpos($product_query_pair, ':') === false) {
						// Remove product ID from query string unless it includes information about selected
						// attributes (which is added when linking from shopping cart to product info page)
						$parameters = str_replace($product_query_pair . '&', '', $parameters);
						$parameters = str_replace($product_query_pair, '', $parameters);
					}
					
					// Don't add the cPath to the URI if it matches the product's full cPath. All  other variations
					// must still be added to retain context on the site
					$cpath_being_linked_to = null;
					
					$pattern = '/cPath\=([0-9\_]+)/';
					
					if (preg_match($pattern, $parameters, $matches)) {
						$cpath_being_linked_to = $matches[1];
					}
					
					if ($cpath_being_linked_to == zen_get_product_path($product_id)) {
						$parameters = str_replace('cPath=' . $cpath_being_linked_to . '&', '', $parameters);
						$parameters = str_replace('cPath=' . $cpath_being_linked_to, '', $parameters);
					}
				}
			}
		} else if ($main_page == FILENAME_EZPAGES || $main_page == FILENAME_EZPAGES_POPUP) {
			// This link is to an EZ-Pages page
			
			// Get the page ID
			$pattern = '/[&\?]?(id=([0-9]+))/i';
			
			if (preg_match($pattern, $parameters, $matches)) {
				$ez_page_id_query_pair = $matches[1];
				$ez_page_id = $matches[2];
				
				$columns_to_retrieve = array(
					'uri'
					);
				
				$selections = array(
					'main_page' => $main_page,
					'associated_db_id' => (int) $ez_page_id,
					'language_id' => (int) $_SESSION['languages_id'],
					'current_uri' => 1
					);
				
				$uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
				
				if (!$uri_mapping_result->EOF) {
					$link = $uri_mapping_result->fields['uri'];
				} else if (defined('CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES') &&
						CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES == 1) {
					// No URI mapping exists for this EZ-Page. Attempt to generate one now
					$link = $this->_autogenEZPageURIMapping($ez_page_id, $main_page);
				}
				
				if ($link != '') {
					$parameters = str_replace($ez_page_id_query_pair . '&', '', $parameters);
					$parameters = str_replace($ez_page_id_query_pair, '', $parameters);
				}
			}
		} else {
			// This link is some other Zen Cart page
			
			// Attempt to match a page with the exact same parameters first
			$columns_to_retrieve = array(
				'uri'
				);
			
			$selections = array(
				'main_page' => $main_page,
				'associated_db_id' => 'null',
				'query_string_parameters' => $parameters,
				'language_id' => (int) $_SESSION['languages_id'],
				'current_uri' => 1
				);
			
			$uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
			
			if (!$uri_mapping_result->EOF) {
				$link = $uri_mapping_result->fields['uri'];
				
				// Don't add any parameters as they will be initialised later if URI being mapped to is loaded
				$parameters = '';
				
			} else {
				 // Attempt to match a page with no parameters
				$columns_to_retrieve = array(
					'uri'
					);
				
				$selections = array(
					'main_page' => $main_page,
					'associated_db_id' => 'null',
					'query_string_parameters' => 'null',
					'language_id' => (int) $_SESSION['languages_id'],
					'current_uri' => 1
					);
				
				$uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
				
				if (!$uri_mapping_result->EOF) {
					$link = $uri_mapping_result->fields['uri'];
				}
			}
		}
		
		if ($link != '') {
			$link = $base_link . $link;
			
			if (strlen($parameters) > 0) {
				// Must add the parameters to the link
				$link .= '?';
				
				while (substr($parameters, 0, 1) == '?' || substr($parameters, 0, 1) == '&') {
					$parameters = substr($parameters, 1, strlen($parameters) - 1);
				}
				
				$link .= zen_output_string($parameters);
				
				$separator = '&';
			} else {
				$separator = '?';
			}
			
			// Perform standard Zen Cart functionality for links
			
			// Add the session ID when moving from different HTTP and HTTPS servers, or when SID is defined
			if (($add_session_id == true) && ($session_started == true) && (SESSION_FORCE_COOKIE_USE == 'False')) {
				if (defined('SID') && zen_not_null(constant('SID'))) {
					$sid = constant('SID');
				} elseif ( ( ($request_type == 'NONSSL') && ($connection == 'SSL') && (ENABLE_SSL == 'true') ) || ( ($request_type == 'SSL') && ($connection == 'NONSSL') ) ) {
					if ($http_domain != $https_domain) {
						$sid = zen_session_name() . '=' . zen_session_id();
					}
				}
			}
			
	// clean up the link before processing
			while (strstr($link, '&&')) $link = str_replace('&&', '&', $link);
			while (strstr($link, '&amp;&amp;')) $link = str_replace('&amp;&amp;', '&amp;', $link);
			
			if ( (SEARCH_ENGINE_FRIENDLY_URLS == 'true') && (false/*@TODO $search_engine_safe*/ == true) ) {
				while (strstr($link, '&&')) $link = str_replace('&&', '&', $link);
				
				$link = str_replace('&amp;', '/', $link);
				$link = str_replace('?', '/', $link);
				$link = str_replace('&', '/', $link);
				$link = str_replace('=', '/', $link);
				
				$separator = '?';
			}
			
			if (isset($sid)) {
				$link .= $separator . zen_output_string($sid);
			}

	// clean up the link after processing
			while (strstr($link, '&amp;&amp;')) $link = str_replace('&amp;&amp;', '&amp;', $link);
			
			$link = preg_replace('/&/', '&amp;', $link);
			
			// Have got the link to be used, store it in this instance
			$this->_href_link = $link;
		}
		
		return (strlen($this->_href_link) > 0);
	}
	
	// }}}
	
	
	// {{{ _autogenManufacturerURIMapping()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified manufacturer for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $manufacturer_id   The ID of the manufacturer which needs a mapping generated.
	 * @return  string    The URI mapping auto-generated by this instance or an empty string if a problem occurred.
	 */
	protected function _autogenManufacturerURIMapping($manufacturer_id)
	{
		if (!isset($this->_manufacturers_admin_instance) || is_null($this->_manufacturers_admin_instance)) {
			require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminManufacturers.php');
			
			$this->_manufacturers_admin_instance = new CeonURIMappingAdminManufacturers();
		}
		
		$this->_autogenerated_mapping = $this->_manufacturers_admin_instance->autogenManufacturerURIMapping(
			$manufacturer_id, null, $_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_MANUFACTURER_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = FILENAME_DEFAULT;
			
			$query_string_parameters = 'manufacturers_id=' . $manufacturer_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, $query_string_parameters, null);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Use it!
				return $this->_autogenerated_mapping;
			}
		}
		
		// Couldn't generate a valid mapping, let Zen Cart use a dynamic URI
		return '';
	}
	
	// }}}
	
	
	// {{{ _autogenCategoryURIMapping()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified category for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $category_id   The ID of the category which needs a mapping generated.
	 * @return  string    The URI mapping auto-generated by this instance or an empty string if a problem occurred.
	 */
	protected function _autogenCategoryURIMapping($category_id)
	{
		if (!isset($this->_categories_admin_instance) || is_null($this->_categories_admin_instance)) {
			require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminCategories.php');
			
			$this->_categories_admin_instance = new CeonURIMappingAdminCategories();
		}
		
		$this->_autogenerated_mapping = $this->_categories_admin_instance->autogenCategoryURIMapping($category_id,
			null, null, $_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_WITH_NO_NAME ||
				$this->_autogenerated_mapping ==
				CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data/invalid data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = FILENAME_DEFAULT;
			
			$associated_db_id = $category_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, null, $associated_db_id);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Use it!
				return $this->_autogenerated_mapping;
			}
		}
		
		// Couldn't generate a valid mapping, let Zen Cart use a dynamic URI
		return '';
	}
	
	// }}}
	
	
	// {{{ _autogenProductAndPRPURIMappings()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified product, and any auto-managed product related
	 * pages, for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $product_id   The ID of the product which needs a mapping generated.
	 * @param   string    $product_page_type   The page type of the product (not its type or type ID).
	 * @return  string    The URI mapping auto-generated by this instance for the product or an empty string if a
	 *                    problem occurred.
	 */
	protected function _autogenProductAndPRPURIMappings($product_id, $product_page_type)
	{
		if (!isset($this->_products_admin_instance) || is_null($this->_products_admin_instance)) {
			require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminProducts.php');
			
			$this->_products_admin_instance = new CeonURIMappingAdminProducts();
		}
		
		$this->_autogenerated_mapping = $this->_products_admin_instance->autogenProductURIMapping($product_id,
			null, null, $_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_NAME ||
				$this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_MODEL ||
				$this->_autogenerated_mapping ==
				CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data/invalid data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = $product_page_type;
			
			$associated_db_id = $product_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, null, $associated_db_id);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Use it!
				
				// Attempt to generate any product related page mappings as well though
				if (!isset($this->_page_types_to_manage) || !is_array($this->_page_types_to_manage)) {
					$page_types = array(
						'product_reviews',
						'product_reviews_info',
						'product_reviews_write',
						'tell_a_friend',
						'ask_a_question'
						);
					
					$this->_page_types_to_manage = array();
					
					foreach ($page_types as $page_type) {
						if ($this->_products_admin_instance->autoManageProductRelatedPageURI($page_type)) {
							$this->_page_types_to_manage[] = $page_type;
						}
					}
				}
				
				$base_uri = $this->_autogenerated_mapping . '/';
				
				foreach ($this->_page_types_to_manage as $page_type) {
					// Mark any existing URI mapping for this product related page as no longer being the "primary"
					// mapping, leaving it in the database so old links aren't broken.
					$columns_to_retrieve = array(
						'uri'
						);
					
					$selections = array(
						'main_page' => $page_type,
						'associated_db_id' => (int) $associated_db_id,
						'language_id' => (int) $_SESSION['languages_id'],
						'current_uri' => 1,
						);
					
					$order_by = null;
					
					$current_uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections,
						$order_by, 1);
					
					if (!$current_uri_mapping_result->EOF) {
						$this->makeURIMappingHistorical($current_uri_mapping_result->fields['uri'],
							$_SESSION['languages_id']);
					}
					
					$uri_part = $this->_products_admin_instance->getProductRelatedPageURIPart(
						$page_type, $_SESSION['languages_code']);
					
					if ($uri_part == false) {
						// Unexpected database problem encountered
						continue;
					}
					
					$uri_part = $this->_convertStringForURI($uri_part, $_SESSION['languages_code']);
					
					$uri = $base_uri . $uri_part;
					
					$main_page = constant('FILENAME_' . strtoupper($page_type));
					
					$mapping_added = $this->addURIMapping($uri, $_SESSION['languages_id'],
						$main_page, null, $product_id);
					
					// Status of attempt to add the mapping is ignored. As long as main product's mapping is
					// generated, process is considered successful
				}
				
				return $this->_autogenerated_mapping;
			}
		}
		
		// Couldn't generate a valid mapping, let Zen Cart use a dynamic URI
		return '';
	}
	
	// }}}
	
	
	// {{{ _autogenEZPageURIMapping()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified EZ-Page for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $page_id   The ID of the EZ-Page which needs a mapping generated.
	 * @param   string    $ez_page_type   The page type of the EZ-Page (normal or popup).
	 * @return  string    The URI mapping auto-generated by this instance or an empty string if a problem occurred.
	 */
	protected function _autogenEZPageURIMapping($page_id, $ez_page_type)
	{
		if (!isset($this->_ez_pages_admin_instance) || is_null($this->_ez_pages_admin_instance)) {
			require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminEZPages.php');
			
			$this->_ez_pages_admin_instance = new CeonURIMappingAdminEZPages();
		}
		
		$this->_autogenerated_mapping = $this->_ez_pages_admin_instance->autogenEZPageURIMapping($page_id, null,
			$_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping ==
				CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_EZ_PAGE_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = $ez_page_type;
			
			$associated_db_id = $page_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, null, $associated_db_id);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Use it!
				return $this->_autogenerated_mapping;
			}
		}
		
		// Couldn't generate a valid mapping, let Zen Cart use a dynamic URI
		return '';
	}
	
	// }}}
	
	
	// {{{ _validateMapping()
	
	/**
	 * Makes sure any auto-generated URI mapping is valid. Checks its format and that it doesn't clash with any
	 * existing mappings.
	 *
	 * @access  protected
	 * @return  boolean   Whether or not the mapping is valid.
	 */
	protected function _validateMapping()
	{
		// Make sure URI mapping is relative to root of site and does not have a
		// trailing slash or any illegal characters
		$this->_autogenerated_mapping = $this->_cleanUpURIMapping($this->_autogenerated_mapping);
		
		if (strlen($this->_autogenerated_mapping) > 0) {
			// New mapping has been generated
			
			// Check that the mapping just generated doesn't clash with any existing mappings, so the user can be
			// notified
			$columns_to_retrieve = array(
				'uri'
				);
			
			$selections = array(
				'uri' => zen_db_prepare_input($this->_autogenerated_mapping),
				'current_uri' => 1,
				'language_id' => $_SESSION['languages_id']
				);
			
			$existing_uri_mapping_result =
				$this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
			
			if ($existing_uri_mapping_result->EOF) {
				// Mapping format valid and doesn't clash!
				return true;
			}
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ getHREFLink()
	
	/**
	 * Simply returns the HREF link value built by this instance.
	 *
	 * @access  public
	 * @return  string    The HREF link built by this instance.
	 */
	public function getHREFLink()
	{
		return $this->_href_link;
	}
	
	// }}}
}

// }}}
