<?php

/**
 * Ceon URI Mapping URI Handler Class. 
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingHandler.php 1054 2012-09-22 15:45:15Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingHandlerBase.php');


// {{{ CeonURIMappingHandler

/**
 * Standard version of Ceon URI Mapping URI Handler class which provides maximum compatibility and flexibility when
 * mapping the current URI to a Zen Cart page.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingHandler extends CeonURIMappingHandlerBase
{
	// {{{ properties
	
	/**
	 * A mapping generated by this instance.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_autogenerated_mapping = null;
	
	/**
	 * A URI generated by this instance, based on a newly created auto-generated mapping.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_autogenerated_uri = null;
	
	// }}}
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingHandler class. Checks the server and module's configuration,
	 * then attempts to map the current URI to the appropriate Zen Cart page.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ _handleStaticURI()
	
	/**
	 * Attempts to match the current URI against a static URI in the Ceon URI Mapping database.
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @return  none
	 */
	protected function _handleStaticURI()
	{
		global $db;
		
		// Make sure that request URI has no invalid characters in it
		$uri_to_match = preg_replace('/[^a-zA-Z0-9_\-\.\/%]/', '', $this->_request_uri);
		
		// Remove any trailing slashes
		while (substr($uri_to_match, -1) == '/') {
			$uri_to_match = substr($uri_to_match, 0, strlen($uri_to_match) -1);
		}
		
		$match_criteria_sql = "um.uri = '" . zen_db_prepare_input($uri_to_match) . "'";
		
		$columns_to_retrieve = array(
			'language_id',
			'main_page',
			'query_string_parameters',
			'associated_db_id',
			'alternate_uri',
			'current_uri',
			'redirection_type_code'
			);
		
		$selections = array(
			'uri' => zen_db_prepare_input($uri_to_match)
			);
		
		$order_by = '
			current_uri DESC,
			language_id,
			date_added DESC';
		
		$match_uri_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by);
		
		if ($match_uri_result->EOF) {
			// URI not found, attempt to create a historical URI if appropriate, then redirect to the appropriate
			// page?
			if (substr(strtolower(CEON_URI_MAPPINGS_MANAGER_ADD_HISTORICAL_URIS_FOR_UNMAPPED_SEO_MODULE_PAGES), 0,
					4) != 'none') {
				$this->_attemptToMapUnmappedStaticURIAsHistoricalURI($uri_to_match);
			}
			
			// URI couldn't be mapped, should the index page be shown or the 404 page?
			if (MISSING_PAGE_CHECK == 'On' || MISSING_PAGE_CHECK == 'true') {
				$_GET['main_page'] = 'index';
			} else if (MISSING_PAGE_CHECK == 'Page Not Found') {
				header('HTTP/1.1 404 Not Found');
				$_GET['main_page'] = 'page_not_found';
			}
			
			return;
		}
		
		// Have matched the URI!
		
		// Use the current mapping that specifically maps the language ID originally being used when this URI was
		// accessed, otherwise simply use the most recent current mapping, falling back to the most recent mapping
		// for the original language ID and, as a last resort, the most recent mapping for any ID
		$current_and_original_language_mapping = null;
		$current_mappings = array();
		$original_language_mappings = array();
		$other_mappings = array();
		
		while (!$match_uri_result->EOF) {
			$current_mapping_info = $match_uri_result->fields;
			
			if ($current_mapping_info['current_uri'] == 1 &&
					$current_mapping_info['language_id'] == $this->_original_language_id) {
				// Optimal mapping found
				$current_and_original_language_mapping = $current_mapping_info;
				break;
			}
			
			if ($current_mapping_info['current_uri'] == 1) {
				$current_mappings[] = $current_mapping_info;
			} else if ($current_mapping_info['language_id'] == $this->_original_language_id) {
				$original_language_mappings[] = $current_mapping_info;
			} else {
				$other_mappings[] = $current_mapping_info;
			}
			
			$match_uri_result->MoveNext();
		}
		
		// Use the details for the best possible match for the URI (criteria as above)
		if (!is_null($current_and_original_language_mapping)) {
			$mapping_info = $current_and_original_language_mapping;
		} else if (sizeof($current_mappings) > 0) {
			$mapping_info = $current_mappings[0];
		} else if (sizeof($original_language_mappings) > 0) {
			$mapping_info = $original_language_mappings[0];
		} else {
			$mapping_info = $other_mappings[0];
		}
		
		$language_id = (int)$mapping_info['language_id'];
		$main_page = $mapping_info['main_page'];
		$query_string_parameters = $mapping_info['query_string_parameters'];
		$associated_db_id = is_null($mapping_info['associated_db_id']) ? null : (int)$mapping_info['associated_db_id'];
		$alternate_uri = $mapping_info['alternate_uri'];
		$current_uri = $mapping_info['current_uri'];
		$redirection_type_code = $mapping_info['redirection_type_code'];
		
		if (!is_null($alternate_uri)) {
			$alternate_uri = trim($alternate_uri);
			
			if (strlen($alternate_uri) == 0) {
				$alternate_uri = null;
			}
		}
		
		if (!is_null($alternate_uri)) {
			// Alternative URI is a full URI to be redirected to
			
			// Assume that a form would never be posted to an alternative URI, so don't perform POST check
			switch ($redirection_type_code) {
				case 301:
					header('HTTP/1.1 301 Moved Permanently');
					break;
				case 302:
					header('HTTP/1.1 302 Found');
					break;
				case 303:
					header('HTTP/1.1 303 See Other');
					break;
				case 307:
					header('HTTP/1.1 307 Temporary Redirect');
					break;
			}
			
			header('Location: ' . $alternate_uri);
			
			zen_exit();
			
		} else if (is_null($main_page)) {
			// No value found for main page, this is a corrupted URI mapping!
			header('HTTP/1.1 404 Not Found');
			
			$_GET['main_page'] = 'page_not_found';
			
			// Add an error message to the message stack session variable as the message stack
			// itself isn't available at this point in the initsystem
			if (!isset($_SESSION['messageToStack']) || !is_array($_SESSION['messageToStack'])) {
				$_SESSION['messageToStack'] = array();
			}
			
			$_SESSION['messageToStack'][] = array(
				'class' => 'heading',
				'text' => 'Unable to map identified URI: ' . $this->_request_uri,
				'type' => 'error'
				);
			
			return;
		}
		
		// Have matched a Zen Cart page to initialise
		
		// Is the URI a current URI or should an attempt be made to find the current URI to redirect to? Or has the
		// user just changed the language so an attempt should be made to find the current URI for the new
		// language?
		// Although don't perform redirect checks if a form is being posted, as any redirect would break the post
		if (count($_POST) == 0 && ($current_uri != 1 || $this->_language_changed)) {
			// Attempt to find the current URI mapping for the user's chosen language
			$redirection_check_language_id = $_SESSION['languages_id'];
			
			$redirection_uri = $this->_getCurrentURI($main_page, $associated_db_id, $query_string_parameters,
				$redirection_check_language_id);
			
			if ($redirection_uri == false) {
				// Didn't find a current URI mapping for the user's chosen language
				
				// If the URI mapping identified so far is a historical mapping which uses a different language,
				// try to find the current URI mapping for that language
				if ($current_uri != 1 && $language_id != $_SESSION['languages_id']) {
					$redirection_check_language_id = $language_id;
					
					$redirection_uri = $this->_getCurrentURI($main_page, $associated_db_id,
						$query_string_parameters, $redirection_check_language_id);
					
					if ($redirection_uri == false) {
						// Didn't find a current URI mapping for the language the identified mapping is using
						
					} else {
						// If using a different language from the session (and the user hasn't just selected a new
						// language), must use the new language
						if ($language_id != $_SESSION['languages_id'] && !$this->_language_changed) {
							$language_info_query = "
								SELECT
									directory,
									code
								FROM
									" . TABLE_LANGUAGES . " 
								WHERE
									languages_id = '" . (int) $language_id . "';";
							
							$language_info_result = $db->Execute($language_info_query);
							
							$_SESSION['language'] = $language_info_result->fields['directory'];
							$_SESSION['languages_id'] = $language_id;
							$_SESSION['languages_code'] = $language_info_result->fields['code'];
							
							// Redirecting to a new URI, must record that the language is being changed so that the
							// correct currency can be used
							$_SESSION['ceon_uri_mapping_language_changed'] = true;
						}
					}
				}
			}
			
			if ($current_uri == 1 && $this->_language_changed) {
				// Must record that the language is being changed so that the correct currency can  be used
				$_SESSION['ceon_uri_mapping_language_changed'] = true;
			}
			
			// Don't redirect to another URI if the URI is the same but just for a different language
			if ($this->_language_changed && $redirection_uri == $uri_to_match) {
				$redirection_uri = false;
			}
			
			if ($redirection_uri != false) {
				// Have found a current URI mapping for this page, force a redirect to it
				
				// Include any extra info in the GET variables
				if (!is_null($associated_db_id) || is_null($query_string_parameters)) {
					$query_string = $this->_buildQueryString();
					
				} else if (!is_null($query_string_parameters)) {
					// Query string variables will be rebuilt after redirecting so no need to append them
					$query_string = '';
				}
				
				$redirection_uri .= $query_string;
				
				header('HTTP/1.1 301 Moved Permanently');
				header('Location: ' . $redirection_uri);
				
				zen_exit();
				
			} else if (!$this->_language_changed) {
				// There is no current URI for this language, this is a URI from the historical database, being
				// used to prevent broken links
				$this->_handleHistoricalURIWithNoCurrentMapping($main_page, $associated_db_id,
					$query_string_parameters);
			}
		} else if ($language_id != $_SESSION['languages_id']) {
			// The user hasn't just selected a new language, but the URI uses a different language from the
			// session, so must switch the session to use the new language
			$language_info_query = "
				SELECT
					directory,
					code
				FROM
					" . TABLE_LANGUAGES . " 
				WHERE
					languages_id = '" . (int) $language_id . "';";
			
			$language_info_result = $db->Execute($language_info_query);
			
			$_SESSION['language'] = $language_info_result->fields['directory'];
			$_SESSION['languages_id'] = $language_id;
			$_SESSION['languages_code'] = $language_info_result->fields['code'];
			
			$this->_language_changed = true;
		}
		
		// This URI is the current URI for this page. Set up the Zen Cart environment for the page
		$_GET['main_page'] = $main_page;
		
		if (!is_null($associated_db_id)) {
			// Page is a category, EZ-Page or a product info page, must initialise the appropriate ID
			if ($main_page == FILENAME_DEFAULT) {
				// This is a category page
				// Must build the full contextual link to the category, so any hierarchy is respected
				$categories = array();
				
				zen_get_parent_categories($categories, $associated_db_id);
				
				$categories = array_reverse($categories);
				
				$cPath = implode('_', $categories);
				
				if (zen_not_null($cPath)) {
					$cPath .= '_';
				}
				
				$cPath .= $associated_db_id;
				
				$_GET['cPath'] = $cPath;
				
			} else if ($main_page == FILENAME_EZPAGES || $main_page == FILENAME_EZPAGES_POPUP) {
				// Have found an EZ-Page. Should it be displayed as a normal page or in a popup?
				if ($main_page == FILENAME_EZPAGES_POPUP || isset($_GET['ezpopup'])) {
					$_GET['main_page'] = FILENAME_EZPAGES_POPUP;
				} else {
					$_GET['main_page'] = FILENAME_EZPAGES;
				}
				
				$_GET['id'] = $associated_db_id;
				
			} else if ($main_page == FILENAME_ASK_A_QUESTION) {
				// Have found an Ask A Question Page.
				if (isset($_GET['pid']) && strpos($_GET['pid'], ':') !== false) {
					// Query string parameter includes information about selected attributes (which is added when
					// linking from shopping cart to a product info page) so don't override the parameter's value
					
				} else {
					$_GET['pid'] = $associated_db_id;
				}
				// Rebuild the cPath variable for this page if it doesn't exist
				if (!isset($_GET['cPath'])) {
					$_GET['cPath'] = zen_get_product_path($_GET['pid']);
				}
				
			} else {
				// This is a product related page
				if (isset($_GET['products_id']) && strpos($_GET['products_id'], ':') !== false) {
					// Query string parameter includes information about selected attributes (which is added when
					// linking from shopping cart to a product info page) so don't override the parameter's value
					
				} else {
					$_GET['products_id'] = $associated_db_id;
				}
				
				// Rebuild the cPath variable for this page if it doesn't exist
				if (!isset($_GET['cPath'])) {
					$_GET['cPath'] = zen_get_product_path($_GET['products_id']);
				}
			}
		} else if (!is_null($query_string_parameters)) {
			// Page is a Zen Cart page with preset parameters, must initialise the parameters
			$query_string_pairs = explode('&', $query_string_parameters);
			
			foreach ($query_string_pairs as $query_string_pair) {
				$parameter_parts = explode('=', $query_string_pair);
				
				// Parameter from database overrides any in query string
				if (count($parameter_parts) == 2) {
					$_GET[$parameter_parts[0]] = urldecode($parameter_parts[1]);
				}
			}
		}
		
		// Make this URI the canonical URI for the page
		global $ceon_uri_mapping_canonical_uri;
		
		$ceon_uri_mapping_canonical_uri = HTTP_SERVER . $uri_to_match;
		
		// A product review's page needs the ID included as part of the canonical URI
		if (defined('FILENAME_PRODUCT_REVIEWS_INFO') && $main_page == FILENAME_PRODUCT_REVIEWS_INFO &&
				isset($_GET['reviews_id'])) {
			$ceon_uri_mapping_canonical_uri .= '?reviews_id=' . $_GET['reviews_id'];
		}
		if (isset($GLOBALS['zco_notifier'])) {
			$GLOBALS['zco_notifier']->notify('CEON_CLASS_HANDLER_HANDLE_STATIC_URI_END', compact('mapping_info', 'uri_to_match'));
		}
	}
	
	// }}}
	
	
	// {{{ _attemptToMapUnmappedStaticURIAsHistoricalURI()
	
	/**
	 * Attempts to match the static URI against known URI formats from other software. If a match is made, a
	 * historical URI is created and the user redirected to the appropriate page.
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @param   string    $uri   The URI to be matched against.
	 * @return  none
	 */
	protected function _attemptToMapUnmappedStaticURIAsHistoricalURI($uri)
	{
		global $db;
		
		$redirection_uri = null;
		
		$product_id = null;
		$category_id = null;
		$ez_page_id = null;
		$manufacturer_id = null;
		$product_review_product_id = null;
		$product_review_info_product_id = null;
		
		if (strpos(CEON_URI_MAPPINGS_MANAGER_ADD_HISTORICAL_URIS_FOR_UNMAPPED_SEO_MODULE_PAGES, 'Ultimate') !==
				false) {
			if (preg_match('|\-c\-([0-9\_]+)\.html$|', $uri, $matches)) {
				// Matches Ultimate SEO URLs category URI format
				$category_id = array_pop(explode('_', $matches[1]));
				
			} else if (preg_match('|\-p\-([0-9]+)\.html$|', $uri, $matches)) {
				// Matches Ultimate SEO URLs product URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-ezp\-([0-9]+)\.html$|', $uri, $matches)) {
				// Matches Ultimate SEO URLs EZ-Page URI format
				$ez_page_id = $matches[1];
				
			} else if (preg_match('|\-m\-([0-9]+)\.html$|', $uri, $matches)) {
				// Matches Ultimate SEO URLs manufacturer URI format
				$manufacturer_id = $matches[1];
				
			} else if (preg_match('|\-pr\-([0-9]+)\.html$|', $uri, $matches)) {
				// Matches Ultimate SEO URLs product review page URI format
				$product_review_product_id = $matches[1];
				
			} else if (preg_match('|\-pri\-([0-9]+)\.html$|', $uri, $matches)) {
				// Matches Ultimate SEO URLs product review info page URI format
				$product_review_info_product_id = $matches[1];
			}
		}
		
		if (strpos(CEON_URI_MAPPINGS_MANAGER_ADD_HISTORICAL_URIS_FOR_UNMAPPED_SEO_MODULE_PAGES, 'Simple') !==
				false) {
			if (preg_match('|\-c\-([0-9\_]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs category URI format
				$category_id = array_pop(explode('_', $matches[1]));
				
			} else if (preg_match('|\-p\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs standard product URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-m\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs music info product URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-g\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs document general URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-d\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs document product URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-f\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs free shipping product URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-w\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs document website info URI format
				$product_id = $matches[1];
				
			} else if (preg_match('|\-page\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs EZ-Page URI format
				$ez_page_id = $matches[1];
				
			} else if (preg_match('|\-manufacturer\-([0-9]+)$|', $uri, $matches)) {
				// Matches Simple SEO URLs manufacturer URI format
				$manufacturer_id = $matches[1];
			}
		}
		
		if (!is_null($category_id)) {
			// A category page URI has been identified. Create a historical URI for the category and redirect to
			// the category's page
			$associated_db_id = $category_id;
			
			$main_page = FILENAME_DEFAULT;
			
			// Impose basic security check/limit
			if (!$this->_tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($_SESSION['languages_id'],
					$main_page, null, $associated_db_id)) {
				if ($this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, null, $associated_db_id)) {
					$this->makeURIMappingHistorical($uri, $_SESSION['languages_id']);
				}
			}
			
			$params = 'cPath=' . $category_id;
			
			$redirection_uri = zen_href_link($main_page, $params);
		}
		
		if (!is_null($product_id)) {
			// A product page URI has been identified. Create a historical URI for the product and redirect to the
			// product's page
			$main_page = zen_get_info_page($product_id);
			
			$associated_db_id = $product_id;
			
			// Impose basic security check/limit
			if (!$this->_tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($_SESSION['languages_id'],
					$main_page, null, $associated_db_id)) {
				if ($this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, null, $associated_db_id)) {
					$this->makeURIMappingHistorical($uri, $_SESSION['languages_id']);
				}
			}
			
			$params = 'products_id=' . $product_id;
			
			$redirection_uri = zen_href_link($main_page, $params);
		}
		
		if (!is_null($ez_page_id)) {
			// An EZ-Page URI has been identified. Create a historical URI for the EZ-Page and redirect to the
			// EZ-Page
			$associated_db_id = $ez_page_id;
			
			$main_page = FILENAME_EZPAGES;
			
			// Impose basic security check/limit
			if (!$this->_tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($_SESSION['languages_id'],
					$main_page, null, $associated_db_id)) {
				if ($this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, null, $associated_db_id)) {
					$this->makeURIMappingHistorical($uri, $_SESSION['languages_id']);
				}
			}
			
			$params = 'id=' . $ez_page_id;
			
			$redirection_uri = zen_href_link($main_page, $params);
		}
		
		if (!is_null($manufacturer_id)) {
			// A manufacturer page URI has been identified. Create a historical URI for the manufacturer and
			// redirect to the manufacturer's page
			$main_page = FILENAME_DEFAULT;
			
			$query_string_parameters = 'manufacturers_id=' . $manufacturer_id;
			
			// Impose basic security check/limit
			if (!$this->_tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($_SESSION['languages_id'],
					$main_page, $query_string_parameters)) {
				if ($this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, $query_string_parameters)) {
					$this->makeURIMappingHistorical($uri, $_SESSION['languages_id']);
				}
			}
			
			$params = 'manufacturers_id=' . $manufacturer_id;
			
			$redirection_uri = zen_href_link($main_page, $params);
		}
		
		if (!is_null($product_review_product_id)) {
			// A product reviews page URI has been identified. Create a historical URI for the product reviews page
			// and redirect to it
			$main_page = FILENAME_PRODUCT_REVIEWS;
			
			$associated_db_id = $product_review_product_id;
			
			// Impose basic security check/limit
			if (!$this->_tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($_SESSION['languages_id'],
					$main_page, null, $associated_db_id)) {
				if ($this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, null, $associated_db_id)) {
					$this->makeURIMappingHistorical($uri, $_SESSION['languages_id']);
				}
			}
			
			$params = 'products_id=' . $product_review_product_id;
			
			$redirection_uri = zen_href_link($main_page, $params);
		}
		
		if (!is_null($product_review_info_product_id)) {
			// A product review info page URI has been identified. Create a historical URI for the product review
			// info page and redirect to it
			$main_page = FILENAME_PRODUCT_REVIEWS_INFO;
			
			$associated_db_id = $product_review_info_product_id;
			
			// Impose basic security check/limit
			if (!$this->_tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($_SESSION['languages_id'],
					$main_page, null, $associated_db_id)) {
				if ($this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, null, $associated_db_id)) {
					$this->makeURIMappingHistorical($uri, $_SESSION['languages_id']);
				}
			}
			
			$params = 'products_id=' . $product_review_info_product_id;
			
			$redirection_uri = zen_href_link($main_page, $params);
		}
		
		if (!is_null($redirection_uri)) {
			$redirection_uri = str_replace('&amp;', '&', $redirection_uri);
			
			header('HTTP/1.1 301 Moved Permanently');
			header('Location: ' . $redirection_uri);
			
			zen_exit();
		}
	}
	
	// }}}
	
	
	// {{{ _tooManyHistoricalURIsForPageForAutoHistoricalAutoGen()
	
	/**
	 * Bacsic defence against malicious usage of the automatic historical URI creation for other SEO modules' old
	 * URIs. Checks whether the number of historical URIs which has been created for the given page is too high for
	 * any more to be created automatically.
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @param   integer   $language_id               The Zen Cart language ID for the mappings.
	 * @param   string    $main_page                 The Zen Cart page type.
	 * @param   string    $query_string_parameters   The query string parameters for the page.
	 * @param   integer   $associated_db_id          The ID of any database record for the page.
	 * @return  boolean   True if the limit for the number of historical URIs for the given page has been reached,
	 *                    false otherwise.
	 */
	protected function _tooManyHistoricalURIsForPageForAutoHistoricalAutoGen($language_id, $main_page,
		$query_string_parameters, $associated_db_id = null)
	{
		global $db;
		
		$columns_to_retrieve = array(
			'COUNT(uri) AS num_historical_uris'
			);
		
		if (!is_null($query_string_parameters)) {
			$selections = array(
				'main_page' => $main_page,
				'query_string_parameters' => $query_string_parameters,
				'language_id' => (int) $language_id,
				'current_uri' => 0,
				);
		} else {
			$selections = array(
				'main_page' => $main_page,
				'associated_db_id' => (int) $associated_db_id,
				'language_id' => (int) $language_id,
				'current_uri' => 0,
				);
		}
		
		$num_historical_uris_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections);
		
		if (!$num_historical_uris_result->EOF && $num_historical_uris_result->fields['num_historical_uris'] > 9) {
			// 10 or more historical URIs.. too many for check!
			return true;
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _handleZCDynamicURI()
	
	/**
	 * Attempts to find and redirect to a static URI for the current, dynamic URI. If none is found, Zen Cart can
	 * proceed as normal.
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @return  none
	 */
	protected function _handleZCDynamicURI()
	{
		global /*$db, */$ceon_uri_mapping_product_pages, $ceon_uri_mapping_product_related_pages;
		
		$associated_db_id = null;
		$query_string_to_match = null;
		
		if ($_GET['main_page'] == FILENAME_DEFAULT && (isset($_GET['cPath']) || isset($_GET['manufacturers_id']) ||
				(isset($_GET['typefilter']) && 	$_GET['typefilter'] != '' &&
				isset($_GET[$_GET['typefilter'] . '_id']) && $_GET[$_GET['typefilter'] . '_id'] != ''))) {
			if (isset($_GET['manufacturers_id'])) {
				// This is a manufacturer page - get the URI for the manufacturers page if there is one
				$query_string_to_match = 'manufacturers_id=' . $_GET['manufacturers_id'];
				
			} else if (isset($_GET['typefilter']) && $_GET['typefilter'] != '' &&
					isset($_GET[$_GET['typefilter'] . '_id']) && $_GET[$_GET['typefilter'] . '_id'] != '') {
				// This is a filtered page - get the URI for the filtered page if there is one
				$query_string_to_match = 'typefilter=' . $_GET['typefilter'] . '&' . $_GET['typefilter'] . '_id=' .
					$_GET[$_GET['typefilter'] . '_id'];
				
			} else if (isset($_GET['cPath'])) {
				// This is a category page - get the URI for the category ID
				$category_parts = explode('_', $_GET['cPath']);
				$category_id = $category_parts[count($category_parts) - 1];
				
				$associated_db_id = $category_id;
			}
		} else if (in_array($_GET['main_page'], $ceon_uri_mapping_product_pages) ||
				in_array($_GET['main_page'], $ceon_uri_mapping_product_related_pages)) {
			$associated_db_id = !empty($_GET['products_id']) ? $_GET['products_id'] : (!empty($_GET['pid']) ? $_GET['pid'] : 0);
			
		} else if ($_GET['main_page'] == FILENAME_EZPAGES || $_GET['main_page'] == FILENAME_EZPAGES_POPUP) {
			$associated_db_id = !empty($_GET['id']) ? $_GET['id'] : 0;
			
		} else if ((count($_GET) > 1 && !isset($_GET[zen_session_name()])) ||
				(isset($_GET[zen_session_name()]) && count($_GET) > 2)) {
			// Check if there's a query string to match against for this Zen Cart page
			$query_string_to_match = $this->_query_string;
				
			// Remove main_page variable from query string parameters to be matched
			$query_string_to_match = str_replace('main_page=' . $_GET['main_page'], '', $query_string_to_match);
			
			if (substr($query_string_to_match, 0, 1) == '&') {
				$query_string_to_match = substr($query_string_to_match, 1, strlen($query_string_to_match) - 1);
			}
			
			if (substr($query_string_to_match, -1) == '&') {
				$query_string_to_match = substr($query_string_to_match, 0, strlen($query_string_to_match) - 1);
			}
		}
		
		// Attempt to find a current URI mapping for the user's chosen language
		$redirection_uri = $this->_getCurrentURI($_GET['main_page'], $associated_db_id,
			$query_string_to_match, $_SESSION['languages_id']);
		
		if ($redirection_uri != false) {
			// Have found a URI mapping for this page, use this as the "current" URI for this page by forcing a
			// redirect to it
			
			// Perform a sanity check to make sure that the URI to be redirected to isn't simply the current URI 
			if ($this->_request_uri != $redirection_uri) {
				// Fine to redirect
				
				// Remove unnecessary variables from query string
				if ($_GET['main_page'] == FILENAME_DEFAULT && isset($_GET['manufacturer_id'])) {
					// Remove the manufacturer ID from the URI as it will be regenerated
					// when the static URI is loaded
					unset($_GET['manufacturer_id']);
					
				} else if ($_GET['main_page'] == FILENAME_DEFAULT && isset($_GET['typefilter']) &&
						$_GET['typefilter'] != '' && isset($_GET[$_GET['typefilter'] . '_id']) &&
						$_GET[$_GET['typefilter'] . '_id'] != '') {
					// Remove the filter info from the URI as it will be regenerated when the static URI is loaded
					unset($_GET['typefilter']);
					unset($_GET[$_GET['typefilter'] . '_id']);
					
				} else if ($_GET['main_page'] == FILENAME_DEFAULT && isset($_GET['cPath'])) {
					// Remove the category ID from the URI as it will be regenerated when the static URI is loaded
					unset($_GET['cPath']);
					
				} else if (in_array($_GET['main_page'], $ceon_uri_mapping_product_pages) ||
						in_array($_GET['main_page'], $ceon_uri_mapping_product_related_pages)) {
					// Remove the category ID from the URI if it is simply the product's master category as it will
					// be regenerated when the static URI is loaded
					if (isset($_GET['cPath']) && $_GET['cPath'] == zen_get_product_path($_GET['products_id'])) {
						unset($_GET['cPath']);
					}
					
					unset($_GET['products_id'], $_GET['pid']);
					
				} else if ($_GET['main_page'] == FILENAME_EZPAGES ||
						$_GET['main_page'] == FILENAME_EZPAGES_POPUP) {
					unset($_GET['id']);
				}
				
				unset($_GET['main_page']);
				
				if (!is_null($query_string_to_match)) {
					// Remove the query string parameters which matched those in the string in the database as they
					// will be set automatically using the same parameter string associated with the mapped URI in
					// the database - this means they don't have to be displayed to the customer!
					// All other parameters should be passed or additional functionality on the mapped page would
					// not work correctly!
					$query_string_parameter_pairs = explode('&', $query_string_to_match);
					
					foreach ($query_string_parameter_pairs as $current_query_string_parameter_pair) {
						$key_and_value = explode('=', $current_query_string_parameter_pair);
						
						unset($_GET[$key_and_value[0]]);
					}
				}
			} else {
				// Don't redirect to the same address!
				$redirection_uri = false;
			}
		} else if (is_null($associated_db_id) && !is_null($query_string_to_match) &&
				$_GET['main_page'] != FILENAME_DEFAULT) {
			// Could this possibly be a Zen Cart page without parameters?
			$redirection_uri = $this->_getCurrentURI($_GET['main_page'], null, null, $_SESSION['languages_id']);
			
			if ($redirection_uri != false) {
				// Perform a sanity check to make sure that the URI to be redirected to isn't simply the current
				// URI 
				if ($this->_request_uri != $redirection_uri) {
					// Fine to redirect; remove unnecessary page identifier from query string
					unset($_GET['main_page']);
				} else {
					// Don't redirect to the same address!
					$redirection_uri = false;
				}
			}
		}
		
		if ($redirection_uri != false) {
			$query_string = $this->_buildQueryString();
			
			$redirection_uri .= $query_string;
			
			header('HTTP/1.1 301 Moved Permanently');
			header('Location: ' . $redirection_uri);
			
			zen_exit();
		}
		
		$this->_handleUnmappedURI($_GET['main_page'], $associated_db_id, $query_string_to_match);
	}
	
	// }}}
	
	
	// {{{ _handleHistoricalURIWithNoCurrentMapping()
	
	/**
	 * Redirects to a standard Zen Cart dynamic URI, building the URI from the parameters passed.
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @param   string    $main_page                 The name of the Zen Cart page for the URI.
	 * @param   integer   $associated_db_id          The associated database ID for the URI.
	 * @param   integer   $query_string_parameters   The query string parameters for the URI.
	 * @return  none
	 */
	protected function _handleHistoricalURIWithNoCurrentMapping($main_page, $associated_db_id, $query_string_parameters)
	{
		// Attempt to auto-generate the URI now!
		if (defined('CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES') &&
				CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES == 1 &&
				$this->_autogenerateURIMapping($main_page, $associated_db_id, $query_string_parameters)) {
			// URI was successfully generated! Redirect to it
			header('HTTP/1.1 301 Moved Permanently');
			header('Location: ' . $this->_autogenerated_uri);
			
			zen_exit();
		}
		
		global $request_type;
		
		if (!is_null($associated_db_id)) {
			// Must supply the ID information necessary for the correct info to be loaded
			switch ($main_page) {
				case FILENAME_DEFAULT:
					if (!isset($_GET['cPath'])) {
						$_GET['cPath'] = $associated_db_id;
					}
					
					break;
					
				case FILENAME_EZPAGES:
				case FILENAME_EZPAGES_POPUP:
					$_GET['id'] = $associated_db_id;
					
					break;
					
				case FILENAME_ASK_A_QUESTION:
					$_GET['pid'] = $associated_db_id;
					
					break;
					
				default:
					// This is a product related page
					$_GET['products_id'] = $associated_db_id;
					
					break;
			}
		} else if (!is_null($query_string_parameters)) {
			// Must make sure that all parameters for this URI are loaded
			$query_string_pairs = explode('&', $query_string_parameters);
			
			foreach ($query_string_pairs as $query_string_pair) {
				$parameter_parts = explode('=', $query_string_pair);
				
				// Parameter from database overrides any in query string as this is a historical URI mapping.
				if (count($parameter_parts) == 2) {
					$_GET[$parameter_parts[0]] = urldecode($parameter_parts[1]);
				}
			}
		}
		
		if ($request_type == 'SSL') {
			$redirection_uri = DIR_WS_HTTPS_CATALOG;
		} else {
			$redirection_uri = DIR_WS_CATALOG;
		}
		
		if (substr($redirection_uri, 0, 1) != '/') {
			$redirection_uri = '/' . $redirection_uri;
		}
		
		$redirection_uri .= 'index.php';
		
		// Add the page to be loaded to the list of query parameters
		$_GET['main_page'] = $main_page;
		
		$query_string = $this->_buildQueryString();
		
		$redirection_uri .= $query_string;
		
		// Record that the source of redirect was this script, avoiding any unnecessary processing
		// later
		$_SESSION['ceon_uri_mapping_redirected'] = true;
		
		header('HTTP/1.1 301 Moved Permanently');
		header('Location: ' . $redirection_uri);
		
		zen_exit();
	}
	
	// }}}
	
	
	// {{{ _handleUnmappedURI()
	
	/**
	 * Handles a Zen Cart dynamic URI which has no URI mapping.
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @param   string    $main_page                 The name of the Zen Cart page for the URI.
	 * @param   integer   $associated_db_id          The associated database ID for the URI.
	 * @param   integer   $query_string_parameters   The query string parameters for the URI.
	 * @return  none
	 */
	protected function _handleUnmappedURI($main_page, $associated_db_id, $query_string_parameters)
	{
		// Attempt to auto-generate the URI now!
		if (defined('CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES') &&
				CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES == 1 &&
				$this->_autogenerateURIMapping($main_page, $associated_db_id, $query_string_parameters)) {
			// URI was successfully generated! Redirect to it
			header('HTTP/1.1 301 Moved Permanently');
			header('Location: ' . $this->_autogenerated_uri);
			
			zen_exit();
		}
		
		if (is_null($associated_db_id) && !is_null($query_string_parameters) &&
				$_GET['main_page'] == FILENAME_DEFAULT) {
			// Build the canonical URI for a manufacturer page or a filtered page as the fallover Zen Cart
			// canonical code doesn't handle these pages
			if (isset($_GET['manufacturers_id'])) {
				global $ceon_uri_mapping_canonical_uri;
				
				$ceon_uri_mapping_canonical_uri = HTTP_SERVER . DIR_WS_CATALOG .
					'index.php?main_page=index&manufacturers_id=' . $_GET['manufacturers_id'];
				
			} else if (isset($_GET['typefilter']) && $_GET['typefilter'] != '' &&
					isset($_GET[$_GET['typefilter'] . '_id']) && $_GET[$_GET['typefilter'] . '_id'] != '') {
				global $ceon_uri_mapping_canonical_uri;
				
				$ceon_uri_mapping_canonical_uri = HTTP_SERVER . DIR_WS_CATALOG . 'index.php?main_page=index' .
					'&typefilter=' . $_GET['typefilter'] . '&' . $_GET['typefilter'] . '_id=' .
					$_GET[$_GET['typefilter'] . '_id'];
			}
		}
	}
	
	// }}}
	
	
	// {{{ _autogenerateURIMapping()
	
	/**
	 * Attempts to generate a URI mapping for a Zen Cart page which has no current URI mapping. If successful,
	 * builds and stores a valid URI to redirect the user to. 
	 *
	 * @access  protected
	 * @author  Conor Kerr <zen-cart.uri-mapping@ceon.net>
	 * @param   string    $main_page                 The name of the Zen Cart page for the URI.
	 * @param   integer   $associated_db_id          The associated database ID for the URI.
	 * @param   integer   $query_string_parameters   The query string parameters for the URI.
	 * @return  boolean   Whether or not a URI mapping was succesfully generated and a URI built to redirect to.
	 *           
	 */
	protected function _autogenerateURIMapping($main_page, $associated_db_id, $query_string_parameters)
	{
		global $ceon_uri_mapping_product_pages, $ceon_uri_mapping_product_related_pages;
		
		if ($main_page == FILENAME_DEFAULT && $associated_db_id == null &&
				strpos(strtolower($query_string_parameters), 'manufacturers_id=') !== false) {
			// Get the manufacturer ID
			$pattern = '/[&\?]?manufacturers_id=([0-9]+)/i';
			
			if (preg_match($pattern, $query_string_parameters, $matches)) {
				$manufacturer_id = $matches[1];
				
				return $this->_autogenManufacturerURIMapping($manufacturer_id);
			}
		} else if ($main_page == FILENAME_DEFAULT && !is_null($associated_db_id)) {
			
			return $this->_autogenCategoryURIMapping($associated_db_id);
			
		} else if (in_array($main_page, $ceon_uri_mapping_product_pages) && !is_null($associated_db_id)) {
			
			return $this->_autogenProductAndPRPURIMappings($associated_db_id, $main_page);
			
		} else if (($main_page == FILENAME_EZPAGES || $main_page == FILENAME_EZPAGES_POPUP) &&
				!is_null($associated_db_id)) {
			
			return $this->_autogenEZPageURIMapping($associated_db_id, $main_page);
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _autogenManufacturerURIMapping()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified manufacturer for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $manufacturer_id   The ID of the manufacturer which needs a mapping generated.
	 * @return  boolean   Whether or not a URI mapping was succesfully generated and a URI built to redirect to.
	 */
	protected function _autogenManufacturerURIMapping($manufacturer_id)
	{
		require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminManufacturers.php');
			
		$this->_manufacturers_admin_instance = new CeonURIMappingAdminManufacturers();
		
		$this->_autogenerated_mapping = $this->_manufacturers_admin_instance->autogenManufacturerURIMapping(
			$manufacturer_id, null, $_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_MANUFACTURER_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = FILENAME_DEFAULT;
			
			$query_string_parameters = 'manufacturers_id=' . $manufacturer_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, $query_string_parameters, null);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Now build the URI to redirect to
				if (isset($_GET['main_page'])) {
					unset($_GET['main_page']);
				}
				
				if (isset($_GET['manufacturers_id'])) {
					unset($_GET['manufacturers_id']);
				}
				
				$query_string = $this->_buildQueryString();
				
				$this->_autogenerated_uri = $this->_autogenerated_mapping . $query_string;
				
				return true;
			}
		}
		
		// Couldn't generate a valid mapping and URI
		return false;
	}
	
	// }}}
	
	
	// {{{ _autogenCategoryURIMapping()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified category for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $category_id   The ID of the category which needs a mapping generated.
	 * @return  boolean   Whether or not a URI mapping was succesfully generated and a URI built to redirect to.
	 */
	protected function _autogenCategoryURIMapping($category_id)
	{
		require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminCategories.php');
		
		$this->_categories_admin_instance = new CeonURIMappingAdminCategories();
		
		$this->_autogenerated_mapping = $this->_categories_admin_instance->autogenCategoryURIMapping($category_id,
			null, null, $_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_WITH_NO_NAME ||
				$this->_autogenerated_mapping ==
				CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data/invalid data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = FILENAME_DEFAULT;
			
			$associated_db_id = $category_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, null, $associated_db_id);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Now build the URI to redirect to
				if (isset($_GET['main_page'])) {
					unset($_GET['main_page']);
				}
				
				if (isset($_GET['cPath'])) {
					unset($_GET['cPath']);
				}
				
				$query_string = $this->_buildQueryString();
				
				$this->_autogenerated_uri = $this->_autogenerated_mapping . $query_string;
				
				return true;
			}
		}
		
		// Couldn't generate a valid mapping and URI
		return false;
	}
	
	// }}}
	
	
	// {{{ _autogenProductAndPRPURIMappings()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified product, and any auto-managed product related
	 * pages, for the current session's language.
	 *
	 * @access  protected
	 * @param   integer   $product_id   The ID of the product which needs a mapping generated.
	 * @param   string    $product_page_type   The page type of the product (not its type or type ID).
	 * @return  boolean   Whether or not a URI mapping was succesfully generated and a URI built to redirect to.
	 */
	protected function _autogenProductAndPRPURIMappings($product_id, $product_page_type)
	{
		require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminProducts.php');
		
		$this->_products_admin_instance = new CeonURIMappingAdminProducts();
		
		$this->_autogenerated_mapping = $this->_products_admin_instance->autogenProductURIMapping($product_id,
			null, null, $_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_NAME ||
				$this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_MODEL ||
				$this->_autogenerated_mapping ==
				CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data/invalid data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = $product_page_type;
			
			$associated_db_id = $product_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, null, $associated_db_id);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated
				
				// Attempt to generate any product related page mappings as well though
				if (!isset($this->_page_types_to_manage) || !is_array($this->_page_types_to_manage)) {
					$page_types = array(
						'product_reviews',
						'product_reviews_info',
						'product_reviews_write',
						'tell_a_friend',
						'ask_a_question'
						);
					
					$this->_page_types_to_manage = array();
					
					foreach ($page_types as $page_type) {
						if ($this->_products_admin_instance->autoManageProductRelatedPageURI($page_type)) {
							$this->_page_types_to_manage[] = $page_type;
						}
					}
				}
				
				$base_uri = $this->_autogenerated_mapping . '/';
				
				foreach ($this->_page_types_to_manage as $page_type) {
					// Mark any existing URI mapping for this product related page as no longer being the "primary"
					// mapping, leaving it in the database so old links aren't broken.
					$columns_to_retrieve = array(
						'uri'
						);
					
					$selections = array(
						'main_page' => $page_type,
						'associated_db_id' => (int) $id,
						'language_id' => (int) $_SESSION['languages_id'],
						'current_uri' => 1,
						);
					
					$order_by = null;
					
					$current_uri_mapping_result =
						$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
					
					if (!$current_uri_mapping_result->EOF) {
						$this->makeURIMappingHistorical($current_uri_mapping_result->fields['uri'],
							$_SESSION['languages_id']);
					}
					
					$uri_part = $this->_products_admin_instance->getProductRelatedPageURIPart($page_type,
						$_SESSION['languages_code']);
					
					if ($uri_part == false) {
						// Unexpected database problem encountered
						continue;
					}
					
					$uri_part = $this->_convertStringForURI($uri_part, $_SESSION['languages_code']);
					
					$uri = $base_uri . $uri_part;
					
					$main_page = constant('FILENAME_' . strtoupper($page_type));
					
					$mapping_added = $this->addURIMapping($uri, $_SESSION['languages_id'], $main_page, ($product_type = 'ask_a_question' ? 'products_id=' . (int) $id : null),
						$product_id);
					
					// Status of attempt to add the mapping is ignored. As long as main product's mapping is
					// generated, process is considered successful
				}
				
				// Now build the URI to redirect to
				if (isset($_GET['main_page'])) {
					unset($_GET['main_page']);
				}
				
				if (isset($_GET['products_id'])) {
					unset($_GET['products_id']);
				}
				
				$query_string = $this->_buildQueryString();
				
				$this->_autogenerated_uri = $this->_autogenerated_mapping . $query_string;
				
				return true;
			}
		}
		
		// Couldn't generate a valid mapping and URI
		return false;
	}
	
	// }}}
	
	
	// {{{ _autogenEZPageURIMapping()
	
	/**
	 * Attempts to auto-generate a URI mapping for the specified EZ-Page for the current session's
	 * language.
	 *
	 * @access  protected
	 * @param   integer   $page_id   The ID of the EZ-Page which needs a mapping generated.
	 * @param   string    $ez_page_type   The page type of the EZ-Page (normal or popup).
	 * @return  boolean   Whether or not a URI mapping was succesfully generated and a URI built to redirect to.
	 */
	protected function _autogenEZPageURIMapping($page_id, $ez_page_type)
	{
		require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminEZPages.php');
		
		$this->_ez_pages_admin_instance = new CeonURIMappingAdminEZPages();
		
		$this->_autogenerated_mapping = $this->_ez_pages_admin_instance->autogenEZPageURIMapping($page_id, null,
			$_SESSION['languages_code'], $_SESSION['languages_id']);
		
		if ($this->_autogenerated_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_EZ_PAGE_WITH_NO_NAME) {
			// Can't generate the URI because of missing "uniqueness" data
		} else if (!$this->_validateMapping()) {
			// Mapping format invalid or auto-generated mapping clashes with an existing mapping
		} else {
			$main_page = $ez_page_type;
			
			$associated_db_id = $page_id;
			
			$mapping_added = $this->addURIMapping($this->_autogenerated_mapping, $_SESSION['languages_id'],
				$main_page, null, $associated_db_id);
			
			if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
				// Mapping was generated! Now build the URI to redirect to
				if (isset($_GET['main_page'])) {
					unset($_GET['main_page']);
				}
				
				if (isset($_GET['id'])) {
					unset($_GET['id']);
				}
				
				$query_string = $this->_buildQueryString();
				
				$this->_autogenerated_uri = $this->_autogenerated_mapping . $query_string;
				
				return true;
			}
		}
		
		// Couldn't generate a valid mapping and URI
		return false;
	}
	
	// }}}
	
	
	// {{{ _validateMapping()
	
	/**
	 * Makes sure any auto-generated URI mapping is valid. Checks its format and that it doesn't clash with any
	 * existing mappings.
	 *
	 * @access  protected
	 * @return  boolean   Whether or not the mapping is valid.
	 */
	protected function _validateMapping()
	{
		// Make sure URI mapping is relative to root of site and does not have a trailing slash or any illegal
		// characters
		$this->_autogenerated_mapping = $this->_cleanUpURIMapping($this->_autogenerated_mapping);
		
		if (strlen($this->_autogenerated_mapping) > 0) {
			// New mapping has been generated
			
			// Check that the mapping just generated doesn't clash with any existing mappings, so the user can be
			// notified
			$columns_to_retrieve = array(
				'uri'
				);
			
			$selections = array(
				'uri' => zen_db_prepare_input($this->_autogenerated_mapping),
				'current_uri' => 1,
				'language_id' => (int) $_SESSION['languages_id'], /*(int) $languages[$i]['id'],*/
				);
			
			$existing_uri_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve,
				$selections, null, 1);
			
			if ($existing_uri_mapping_result->EOF) {
				// Mapping format valid and doesn't clash!
				return true;
			}
		}
		
		return false;
	}
	
	// }}}
}

// }}}
