<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate Product URIs Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2019 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageAutogenProducts.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Action class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMAction.php');


// {{{ CeonUMMCreateManageAutogenProducts

/**
 * Handles the auto-generation of URI mappings for the selected products.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2019 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageAutogenProducts extends CeonUMMAction
{
	// {{{ properties
	
	/**
	 * Variable tracks when the auto-generation process was started. 
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_autogeneration_start_timestamp = null;
	
	/**
	 * Whether or not debug output should be sent to a file in the cache folder, for tracking the progress of AJAX
	 * autogeneration of category/product URI mappings.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_debug_ajax_to_file = false;
	
	/**
	 * The name of the AJAX debug output file for the current auto-generation session.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_ajax_debug_file = null;
	
	/**
	 * The IDs of the products which are to have their URI mappings generated.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_products_to_generate = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 */
	public function __construct($use_ajax_progress_indicators, $ajax_progress_refresh_sec,
		$standard_progress_refresh_sec)
	{
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' .
			'ceon_umm_create_manage_autogen_products.php');
		
		if (!defined('TEXT_SET_MAPPING_TEMPLATES') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' . 'ceon_umm_create_manage_autogen_products.php');
		}
		
		parent::__construct($use_ajax_progress_indicators, $ajax_progress_refresh_sec,
			$standard_progress_refresh_sec);
	}
	
	// }}}
	
	
	// {{{ autogenerateMappingsForSelectedProducts()
	
	/**
	 * Handles the autogeneration of mappings for the selected products.
	 *
	 * @access  public
	 * @return  string    The HTML/JavaScript source for the output of the current stage of the autogeneration
	 *                    procedure.
	 */
	public function autogenerateMappingsForSelectedProducts()
	{
		global $db, $languages, $num_languages;
		
		// Start the control timer
		$this->_autogeneration_start_timestamp = time();
		
		if ($this->_debug_ajax_to_file) {
			$_SESSION['ceon_umm_ajax_debug_file'] = (defined('DIR_FS_LOGS') ? DIR_FS_LOGS : DIR_FS_SQL_CACHE) . '/ceon_umm_debug-' .
				date('Ymd his', time()) . '.txt';
			
			$this->_ajax_debug_file = fopen($_SESSION['ceon_umm_ajax_debug_file'], 'a');
			
			fwrite($this->_ajax_debug_file, "Starting at " . date('Ymd His', time()) . "\n\n");
		}
		
		if (!isset($this->_posted_data['prods-to-gen'])) {
			// Just starting autogeneration process. Initialise environment
			$content = $this->_initialiseProductsAutogenerationEnvironment();
			
			if (is_string($content)) {
				// Couldn't initialise environment. Output response built
				$this->_output = $content;
				
				return;
			}
		} else {
			// Executing an auto-generation stage for output to the browser. Determine the stage and configure the
			// environment appropriately
			// @TODO
			$this->_configureProductsAutogenerationEnvironment();
		}
		
		$this->_autogenerateProductsURIMappings();
		
		$content = $this->_buildProductsAutogenerationPanel();
		
		if ($this->_use_ajax_progress_indicators == true) {
			// Only run the AJAX controller if there is work still to be done
			if ($this->_num_products_inc > $this->_num_products_processed) {
				$content .= $this->_buildProductsAutogenAJAXController();
			}
		}
		
		if ($this->_debug_ajax_to_file) {
			fwrite($this->_ajax_debug_file, "\n\nStopping at " . date('Ymd His', time()) . "\n\n");
			
			fclose($this->_ajax_debug_file);
		}
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _initialiseProductsAutogenerationEnvironment()
	
	/**
	 * Sets up the autogeneration process' initial environment.
	 *
	 * @access  protected
	 * @return  true|string    True if the environment was initialised, HTML to be displayed within the
	 *                         Create/Manage URIs panel if a problem occurred.
	 */
	protected function _initialiseProductsAutogenerationEnvironment()
	{
		global $db, $languages, $num_languages;
		
		// Set up the environment for the auto-generation process
		$this->_products_to_generate = array();
		
		// Get the list of products selected
		$selected_products = array();
		
		if (!empty($this->_posted_data['product'])) {
			foreach ($this->_posted_data['product'] as $product_id) {
				$selected_products[] = (int) $product_id;
			}
		}
		
		if (sizeof($selected_products) == 0) {
			// No products selected!
			$content = '<h1>' . TEXT_PRODUCT_MAPPINGS_TITLE . '</h1>' . "\n";
			
			$content = '<p>' . TEXT_ERROR_NO_PRODUCTS_SELECTED_FOR_AUTOGENERATION . '</p>' . "\n";
			
			$params = zen_get_all_get_params();
			
			$category_name_result = $db->Execute("
				SELECT
					cd.categories_name
				FROM
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				WHERE
					cd.categories_id = '" . (int) $_GET['sub-cat-id'] . "'
				AND
					cd.language_id = '" . (int) $_SESSION['languages_id'] . "';");
			
			if (!$category_name_result->EOF) {
				$category_name = $category_name_result->fields['categories_name'];
			}
			
			$back_up_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			
			$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to.png') .
				' <a href="' . $back_up_link . '">' .
				sprintf(TEXT_BACK_TO, htmlentities($category_name, ENT_COMPAT, CHARSET)) .
				'</a></p>' . "\n";
			
			return $content;
		}
		
		$this->_include_disabled_products = false;
		
		if (isset($this->_posted_data['inc-disabled'])) {
			$this->_include_disabled_products = true;
		}
		
		if (!$this->_include_disabled_products) {
			// Make sure only enabled products are handled
			$disabled_products = array();
			
			$selected_product_ids = implode(',', $selected_products);
			
			$products_disabled_result = $db->Execute("
				SELECT
					products_id
				FROM
					" . TABLE_PRODUCTS . " p
				WHERE
					p.products_id IN (" . $selected_product_ids . ")
				AND
					p.products_status = '0';");
			
			while (!$products_disabled_result->EOF) {
				$disabled_products[] = $products_disabled_result->fields['products_id'];
				
				$products_disabled_result->MoveNext();
			}
		}
		
		// Count the number of products that are included in the auto-generation procedure to be run
		$this->_num_products_inc = 0;
		
		foreach ($selected_products as $product_id) {
			
			if (!$this->_include_disabled_products && in_array($product_id, $disabled_products)) {
				// Don't include this disabled product
				continue;
			}
			
			$this->_products_to_generate[] = $product_id;
			
			$this->_num_products_inc++;
		}
		
		// Only starting
		$this->_cur_product_pos = 0;
		$this->_num_products_processed = 0;
		$this->_num_product_mappings_generated = 0;
		$this->_num_product_mappings_unchanged = 0;
		$this->_num_product_mappings_failed = 0;
		$this->_processing_cat_mixed_with_products = 0;
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _autogenerateProductsURIMappings()
	
	/**
	 * Autogenerates the mappings for the selected products.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _autogenerateProductsURIMappings()
	{
		global $db;
		
		$this->_product_mappings_generated_messages = array();
		$this->_product_mappings_error_messages = array();
		
		if (empty($this->_products_to_generate)) {
			return true;
		}
		
		$products_result = $db->Execute("
			SELECT
				p.products_id,
				p.products_type
			FROM
				" . TABLE_PRODUCTS . " p
			WHERE
				p.products_id IN (" . implode(',', $this->_products_to_generate) . ")
			ORDER BY
				p.products_id;");
		
		// Skip past any products which have already been processed in a process which is being resumed
		$skipped_generated_products = true;
		
		if ($this->_cur_product_pos != 0) {
			$skipped_generated_products = false;
		}
		
		while (!$products_result->EOF) {
			$product_id = $products_result->fields['products_id'];
			
			if (!$skipped_generated_products) {
				// If the current product ID hasn't been matched yet, then given the
				// ordering, this product must already have been processed
				if ($product_id == $this->_cur_product_pos) {
					// This was the last product to be processed
					$skipped_generated_products = true;
				}
				
				$products_result->MoveNext();
				
				continue;
			}
			
			$this->_autogenerateProductURIMapping($product_id, $products_result->fields['products_type'],
				(int) $_GET['sub-cat-id']);
			
			$this->_cur_product_pos = $products_result->fields['products_id'];
			
			if ($this->_autogenerationTimeHasExpired()) {
				return false;
			}
			
			$products_result->MoveNext();
		}
		
		// Reaching this point without a timeout means that all products have been processed. Reset pointer
		// position
		$this->_cur_product_pos = 0;
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _autogenerationTimeHasExpired()
	
	/**
	 * Simply checks if the current stage of the auto-generation process has still got time to run.
	 *
	 * @access  protected
	 * @return  boolean   True if the current stage has run out of time, false otherwise.
	 */
	protected function _autogenerationTimeHasExpired()
	{
		if ($this->_use_ajax_progress_indicators) {
			return (time() > ($this->_autogeneration_start_timestamp + $this->_ajax_progress_refresh_sec));
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ _autogenerateProductURIMapping()
	
	/**
	 * Auto-generates the URI Mapping for the specified product. Information about the status of the
	 * auto-generation is stored in the appropriate reporting array and the respective counter variable is updated.
	 *
	 * @access  protected
	 * @param   integer   $id   The ID of the product to be autogenerated.
	 * @param   integer   $product_type_id    The product's type ID.
	 * @param   integer   $master_categories_id   The product's master category's ID.
	 * @param   string    $mapping_template   The mapping template for the product. Can be used to speed up
	 *                                        auto-generation process by pre-populating shared placement tags.
	 * @return  none
	 */
	protected function _autogenerateProductURIMapping($id, $product_type_id, $master_categories_id, $mapping_template = null)
	{
		global $languages, $num_languages;
		
		$this->_num_products_processed++;
		
		if (!isset($this->_products_admin_instance) || is_null($this->_products_admin_instance)) {
			require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminProducts.php');
			
			$this->_products_admin_instance = new CeonURIMappingAdminProducts();
		}
		
		// Build the information for the pages that must have URIs managed
		$product_type = zen_get_handler_from_type($product_type_id);
		
		$product_page_type = $product_type . '_info';
		
		if (!isset($this->_page_types_to_manage) || !is_array($this->_page_types_to_manage)) {
			$page_types = array(
				'product_reviews',
				'product_reviews_info',
				'product_reviews_write',
				'tell_a_friend',
				'ask_a_question'
				);
			
			$this->_page_types_to_manage = array();
			
			foreach ($page_types as $page_type) {
				if ($this->_products_admin_instance->autoManageProductRelatedPageURI($page_type)) {
					$this->_page_types_to_manage[] = $page_type;
				}
			}
		}
		
		if (!isset($this->_num_product_prp_mappings_generated)) {
			$this->_num_product_prp_mappings_generated = 0;
		}
		
		if (!isset($this->_num_product_prp_mappings_unchanged)) {
			$this->_num_product_prp_mappings_unchanged = 0;
		}
		
		for ($i = 0; $i < $num_languages; $i++) {
			$uri_mapping = $this->_products_admin_instance->autogenProductURIMapping($id, null, null,
				$languages[$i]['code'], $languages[$i]['id'], null, $mapping_template);
			
			if ($uri_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_NAME ||
					$uri_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_MODEL ||
					$uri_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME) {
				// Can't generate the URI because of missing "uniqueness" data or invalid data
				
				$this->_num_product_mappings_failed++;
				
				if ($uri_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_NAME) {
					$message = TEXT_ERROR_AUTOGENERATION_PRODUCT_HAS_NO_NAME;
				} else if ($uri_mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_MODEL) {
					$message = TEXT_ERROR_AUTOGENERATION_PRODUCT_HAS_NO_MODEL;
				} else {
					$message = TEXT_ERROR_AUTOGENERATION_PRODUCT_CATEGORY_HAS_NO_NAME;
				}
				
				$failure_message = sprintf($message, ucwords($languages[$i]['name']));
				
				// Build a link to the product's edit page so the user can fix the problem
				// immediately
				$product_type_admin_handler = $product_type . '.php';
				
				if (!file_exists(DIR_FS_ADMIN . $product_type_admin_handler) && (PROJECT_VERSION_MAJOR > '1' || version_compare(PROJECT_VERSION_MAJOR . PROJECT_VERSION_MINOR, '1.5.6', '>='))) {
					$product_type_admin_handler = 'product' . '.php';
				}
				
				$edit_link = zen_href_link($product_type_admin_handler, 'pID=' . $id . '&product_type=' .
					$product_type_id .  '&cPath=' . $master_categories_id . '&action=new_product', 'NONSSL', true,
					true, false, false);
				
				$edit_link_title = TEXT_EDIT_PRODUCT . ' ' . $id;
				
				$this->_product_mappings_error_messages[] = array(
					'message' => $failure_message,
					'link' => HTTP_SERVER . $uri,
					'link_title' => $uri,
					'edit_link' => $edit_link,
					'edit_link_title' => $edit_link_title
					);
				
				continue;
			}
			
			// Make sure URI mapping is relative to root of site and does not have a trailing slash
			// or any illegal characters
			$uri_mapping = $this->_cleanUpURIMapping($uri_mapping);
			
			// Get the current URI mapping for this product
			$columns_to_retrieve = array(
				'uri'
				);
			
			$selections = array(
				'main_page' => zen_db_prepare_input($product_page_type),
				'associated_db_id' => (int) $id,
				'current_uri' => 1,
				'language_id' => (int) $languages[$i]['id']
				);
			
			$existing_uri_mapping_result =
				$this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
			
			$prev_uri_mapping = '';
			
			if (!$existing_uri_mapping_result->EOF) {
				$prev_uri_mapping = $existing_uri_mapping_result->fields['uri'];
			}
			
			$insert_uri_mapping = false;
			$update_uri_mapping = false;
			
			if ($uri_mapping != '') {
				// Check if the URI mapping is being updated or does not yet exist
				if ($prev_uri_mapping == '') {
					$insert_uri_mapping = true;
				} else if ($prev_uri_mapping != $uri_mapping) {
					$update_uri_mapping = true;
				}
			}
			
			if ($insert_uri_mapping || $update_uri_mapping) {
				
				$uri = $uri_mapping;
				
				$main_page = $product_page_type;
				
				// Check that the mapping just generated doesn't clash with any existing mapping
				$mapping_clashed = false;
				
				$columns_to_retrieve = array(
					'main_page',
					'associated_db_id',
					'query_string_parameters'
					);
				
				$selections = array(
					'uri' => zen_db_prepare_input($uri),
					'current_uri' => 1,
					'language_id' => (int) $languages[$i]['id']
					);
				
				$order_by = null;
				
				$existing_uri_mapping_result =
					$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
				
				// If the existing mapping is simply having some capitalisation changed then a case insensitive
				// comparison might result in a false positive for a mapping clash, so prevent that by checking
				// the mapping's settings don't match.
				if (!$existing_uri_mapping_result->EOF &&
						!($existing_uri_mapping_result->fields['main_page'] == $main_page &&
						$existing_uri_mapping_result->fields['associated_db_id'] == $id &&
						is_null($existing_uri_mapping_result->fields['query_string_parameters']))) {
					// This mapping clashes with an existing mapping
					$mapping_clashed = true;
				}
				
				if ($mapping_clashed) {
					if ($this->_products_admin_instance->_mappingClashAutoAppendInteger()) {
						// Attempt to find a unique variation of the mapping by appending an auto-incrementing
						// integer
						
						// Must check if the product has already had its URI mapping auto-generated with an
						// integer auto-appended. If so, and this URI mapping is the current URI, then URI doesn't
						// actually need to be updated
						
						$uniqueness_integer = 1;
						
						$base_uri = $uri;
						
						$unique_mapping_found = false;
						
						while (!$unique_mapping_found) {
							$uri = $base_uri . $uniqueness_integer;
							
							$columns_to_retrieve = array(
								'main_page',
								'associated_db_id'
								);
							
							$selections = array(
								'uri' => zen_db_prepare_input($uri),
								'current_uri' => 1,
								'language_id' => (int) $languages[$i]['id']
								);
							
							$order_by = null;
							
							$existing_uri_mapping_result =
								$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
							
							if (!$existing_uri_mapping_result->EOF) {
								// Perform the sanity check to see if this matches the product's current URI. If
								// so, can take it that it was previously auto-appended
								if ($existing_uri_mapping_result->fields['main_page'] == $main_page &&
										$existing_uri_mapping_result->fields['associated_db_id'] ==
										$id) {
									// Match found so assuming that the URI hasn't actually changed
									$this->_num_product_mappings_unchanged++;
									
									$this->_num_product_prp_mappings_unchanged +=
										sizeof($this->_page_types_to_manage);
									
									// Break out of this loop and advance to next URI language
									continue 2;
								}
							} else {
								// This variation of the mapping doesn't clash with an existing
								// mapping, use it!
								$unique_mapping_found = true;
							}
							
							$uniqueness_integer++;
						}
					} else {
						// No attempt should be made to create a unique variation of the mapping, instead the user
						// must be warned that the mapping was not created for this product
						$this->_num_product_mappings_failed++;
						
						$failure_message = sprintf(TEXT_ERROR_PRODUCT_MAPPING_CLASHED,
							ucwords($languages[$i]['name']));
						
						// Build a link to the product's edit page so the user can fix the problem immediately
						$product_type_admin_handler = $product_type . '.php';
						
						if (!file_exists(DIR_FS_ADMIN . $product_type_admin_handler) && (PROJECT_VERSION_MAJOR > '1' || version_compare(PROJECT_VERSION_MAJOR . PROJECT_VERSION_MINOR, '1.5.6', '>='))) {
							$product_type_admin_handler = 'product' . '.php';
						}
						
						$edit_link = zen_href_link($product_type_admin_handler, 'pID=' . $id . '&product_type=' .
							$product_type_id .  '&cPath=' . $master_categories_id . '&action=new_product',
							'NONSSL', true, true, false, false);
						
						$edit_link_title = TEXT_EDIT_PRODUCT . ' ' . $id;
						
						$this->_product_mappings_error_messages[] = array(
								'message' => $failure_message,
								'link' => HTTP_SERVER . $uri,
								'link_title' => $uri,
								'edit_link' => $edit_link,
								'edit_link_title' => $edit_link_title
								);
						
						continue;
					}
				}
				
				if ($update_uri_mapping) {
					// Consign previous mapping to the history, so old URI mapping isn't broken
					$this->makeURIMappingHistorical($prev_uri_mapping, $languages[$i]['id']);
				}
				
				$mapping_added = $this->addURIMapping($uri, $languages[$i]['id'], $main_page, null, $id);
				
				if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
					if (!$mapping_clashed) {
						$success_message = sprintf(TEXT_AUTOGEN_PRODUCT_MAPPING_ADDED,
							ucwords($languages[$i]['name']));
						
						$this->_product_mappings_generated_messages[] = array(
							'message' => $success_message,
							'link' => HTTP_SERVER . $uri,
							'link_title' => $uri
							);
						
					} else {
						// Build a link to the product's edit page so the user can change the auto-appended URI
						// immediately
						$success_message = sprintf(TEXT_AUTOGEN_PRODUCT_MAPPING_ADDED_WITH_AUTO_APPEND,
							ucwords($languages[$i]['name']));
						
						$product_type_admin_handler = $product_type . '.php';
						
						if (!file_exists(DIR_FS_ADMIN . $product_type_admin_handler) && (PROJECT_VERSION_MAJOR > '1' || version_compare(PROJECT_VERSION_MAJOR . PROJECT_VERSION_MINOR, '1.5.6', '>='))) {
							$product_type_admin_handler = 'product' . '.php';
						}
						
						$edit_link = zen_href_link($product_type_admin_handler, 'pID=' . $id . '&product_type=' .
							$product_type_id . '&cPath=' . $master_categories_id . '&action=new_product', 'NONSSL',
							true, true, false, false);
						
						$edit_link_title = TEXT_EDIT_PRODUCT . ' ' . $id;
						
						$this->_product_mappings_generated_messages[] = array(
							'message' => $success_message,
							'link' => HTTP_SERVER . $uri,
							'link_title' => $uri,
							'edit_link' => $edit_link,
							'edit_link_title' => $edit_link_title
							);
					}
					
					$this->_num_product_mappings_generated++;
					
				} else {
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
						$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS, ucwords($languages[$i]['name']));
						
					} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
						$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_DATA, ucwords($languages[$i]['name']));
					} else {
						$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_DB, ucwords($languages[$i]['name']));
					}
					
					$this->_num_product_mappings_failed++;
					
					$product_type_admin_handler = $product_type . '.php';
					
					if (!file_exists(DIR_FS_ADMIN . $product_type_admin_handler) && (PROJECT_VERSION_MAJOR > '1' || version_compare(PROJECT_VERSION_MAJOR . PROJECT_VERSION_MINOR, '1.5.6', '>='))) {
						$product_type_admin_handler = 'product' . '.php';
					}
					
					$edit_link = zen_href_link($product_type_admin_handler, 'pID=' . $id . '&product_type=' .
						$product_type_id .  '&cPath=' . $master_categories_id . '&action=new_product', 'NONSSL',
						true, true, false, false);
					
					$edit_link_title = TEXT_EDIT_PRODUCT . ' ' . $id;
					
					$this->_product_mappings_error_messages[] = array(
						'message' => $failure_message,
						'link' => HTTP_SERVER . $uri,
						'link_title' => $uri,
						'edit_link' => $edit_link,
						'edit_link_title' => $edit_link_title
						);
				}
				
				// Now add the URI mappings for the review pages/tell-a-friend page for this product
				$base_uri = $uri . '/';
				
				// Get the language code for the mapping's language
				$language_code = strtolower($languages[$i]['code']);
				
				foreach ($this->_page_types_to_manage as $page_type) {
					// Mark any existing URI mapping for this product related page as no longer being the "primary"
					// mapping, leaving it in the database so old links aren't broken.
					$columns_to_retrieve = array(
						'uri'
						);
					
					$selections = array(
						'main_page' => $page_type,
						'associated_db_id' => (int) $id,
						'language_id' => (int) $languages[$i]['id'],
						'current_uri' => 1,
						);
					
					$order_by = null;
					
					$current_uri_mapping_result =
						$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
					
					if (!$current_uri_mapping_result->EOF) {
						$this->makeURIMappingHistorical($current_uri_mapping_result->fields['uri'],
							$languages[$i]['id']);
					}
					
					$uri_part =
						$this->_products_admin_instance->getProductRelatedPageURIPart($page_type, $language_code);
					
					if ($uri_part == false) {
						// Unexpected database problem encountered
						continue;
					}
					
					$uri_part = $this->_convertStringForURI($uri_part, $language_code);
					
					$uri = $base_uri . $uri_part;
					
					if (!defined('FILENAME_' . strtoupper($page_type))) {
						continue;
					}
					
					$main_page = constant('FILENAME_' . strtoupper($page_type));
					
					$mapping_added = $this->addURIMapping($uri, $languages[$i]['id'], $main_page, null, $id);
					
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
						$success_message =
							sprintf(TEXT_AUTOGEN_PRODUCT_MAPPING_ADDED, ucwords($languages[$i]['name']));
						
						$this->_product_mappings_generated_messages[] = array(
							'message' => $success_message,
							'link' => HTTP_SERVER . $uri,
							'link_title' => $uri
							);
						
						$this->_num_product_prp_mappings_generated++;
						
					} else {
						if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS, ucwords($languages[$i]['name']));
							
						} else if ($mapping_added ==
								CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DATA, ucwords($languages[$i]['name']));
						} else {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_DB, ucwords($languages[$i]['name']));
						}
						
						if(empty($this->_num_product_prp_mappings_failed)) {
							$this->_num_product_prp_mappings_failed = 1;
						} else {
							$this->_num_product_prp_mappings_failed++;
						}
						
						$this->_product_mappings_error_messages[] = array(
							'message' => $failure_message,
							'link' => HTTP_SERVER . $uri,
							'link_title' => $uri
							);
					}
				}
			} else {
				$this->_num_product_mappings_unchanged++;
				
				/*if ($this->_debug_ajax_to_file) {
					fwrite($this->_ajax_debug_file, "Product Unchanged: " . $id . ' Mapping: ' .
						$uri_mapping . "\n");
				}*/
				
				$this->_num_product_prp_mappings_unchanged += sizeof($this->_page_types_to_manage);
			}
		}
	}
	
	// }}}
	
	
	// {{{ _buildProductsAutogenerationPanel()
	
	/**
	 * Outputs the HTML/JavaScript source for the output of either the initial stage of the autogeneration process,
	 * or, if AJAX is not being used, the output current for the current stage of the process. If AJAX is being
	 * used, the output of subsequent stages is handled via the appropriate AJAX method.
	 *
	 * @access  protected
	 * @return  string    The HTML/JavaScript source for the output of the initial/current stage of the
	 *                    autogeneration procedure.
	 */
	protected function _buildProductsAutogenerationPanel()
	{
		global $db;
		
		$content = '<h1 id="autogeneration-title">' . TEXT_PRODUCT_MAPPINGS_TITLE . '</h1>' . "\n";
		
		$content .= '<p>' . TEXT_NUM_PRODUCTS_PROCESSED . '<span id="num-prods-processed">' .
			$this->_num_products_processed . '</span> / <span id="num-prods">' . $this->_num_products_inc .
			'</span></p>' . "\n";
		
		if ($this->_num_products_inc == 0) {
			$percentage_completed_style_width = 100;
			
			$percentage_completed = 100;
			
		} else if ($this->_num_products_processed == 0) {
			$percentage_completed_style_width = 0;
			
			$percentage_completed = 0;
			
		} else {	
			$percentage_completed_style_width = (100 / ($this->_num_products_inc / $this->_num_products_processed));
			
			$percentage_completed = round(100 / ($this->_num_products_inc / $this->_num_products_processed), 2);
		}
		
		$progress_bar = '<div class="ProgressBar">' .
			'<div class="ProgressBarIndicator" id="categories-progress" style="width: ' .
			$percentage_completed_style_width . '%">' . $percentage_completed . '%</div>' . '</div>' . "\n";
		
		$content .= $progress_bar;
		
		$content .= '<h3 class="AutogenerationSectionTitle">' . TEXT_GENERATED_PRODUCT_MAPPINGS . '</h3>' . "\n";
		
		$content .= '<p id="num-prods-generated">' . TEXT_NUM_PRODUCT_MAPPINGS_GENERATED .
			' <span id="num-prods-generated-count">' . $this->_num_product_mappings_generated . '</span></p>' .
			"\n";
		
		$content .= '<p id="num-prods-unchanged">' .
			TEXT_NUM_PRODUCT_MAPPINGS_UNCHANGED . ' <span id="num-prods-unchanged-count">' .
			$this->_num_product_mappings_unchanged . '</span></p>' . "\n";
		
		if (sizeof($this->_product_mappings_generated_messages) > 0) {
			// Output information about the mappings generated
			// @TODO currently this only outputs information for the current stage.. handling for carrying over
			// from previous stage is not yet implemented
			$content .= '<div id="products-generated-listing">';
			
			foreach ($this->_product_mappings_generated_messages as $message) {
				$content .= '<p>' . htmlspecialchars($message['message']) . ' <a href="' .
					htmlspecialchars($message['link']) . '" target="_blank">' .
					htmlspecialchars($message['link_title']) . '</a>';
				
				if (isset($message['edit_link'])) {
					$content .= ' - ' . ' <a href="' . htmlspecialchars($message['edit_link']) .
						'" target="_blank">' . htmlspecialchars($message['edit_link_title']) . '</a>';
				}
				
				$content .= '<p>' . "\n";
			}
			
			$content .= '</div>' . "\n";
		}
		
		if (sizeof($this->_product_mappings_error_messages) > 0) {
			$content .= '<p id="num-prods-failed">';
		} else {
			$content .= '<p id="num-prods-failed" style="display: none;">';
		}
		
		$content .= TEXT_NUM_PRODUCT_MAPPINGS_FAILED . ' <span id="num-prods-failed-count">' .
			$this->_num_product_mappings_failed . '</span></p>' . "\n";
		
		if (sizeof($this->_product_mappings_error_messages) > 0) {
			// Output information about the errors encountered
			// @TODO currently this only outputs information for the current stage.. handling for carrying over
			// from previous stage is not yet implemented
			$content .= '<div id="product-errors-listing">';
			
			foreach ($this->_product_mappings_error_messages as $message) {
				$content .= '<p>' . htmlspecialchars($message['message']) . ' <a href="' .
					htmlspecialchars($message['link']) . '" target="_blank">' .
					htmlspecialchars($message['link_title']) . '</a>';
				
				if (isset($message['edit_link'])) {
					$content .= ' - ' . ' <a href="' . htmlspecialchars($message['edit_link']) .
						'" target="_blank">' . htmlspecialchars($message['edit_link_title']) . '</a>';
				}
				
				$content .= '<p>' . "\n";
			}
			
			$content .= '</div>' . "\n";
		}
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action .
			'" method="post" name="autogeneration-form" id="autogeneration-form">' . "\n";
			
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Build form parameters to pass on to next stage
		$content .= zen_draw_hidden_field('include-disabled', ($this->_include_disabled_products ? 1 : 0));
		
		$content .= zen_draw_hidden_field('products-to-gen', implode(',', $this->_products_to_generate));
		
		$content .= zen_draw_hidden_field('num-prods', $this->_num_products_inc);
		
		$content .= zen_draw_hidden_field('num-prods-proc', $this->_num_products_processed);
		
		$content .= zen_draw_hidden_field('num-prods-gen', $this->_num_product_mappings_generated);
		
		$content .= zen_draw_hidden_field('num-prods-uc', $this->_num_product_mappings_unchanged);
		
		$content .= zen_draw_hidden_field('num-prods-f', $this->_num_product_mappings_failed);
		
		$content .= zen_draw_hidden_field('cur-prod-pos', $this->_cur_product_pos);
		
		$content .= zen_draw_hidden_field('proc-mixed-cat-prods', $this->_processing_cat_mixed_with_products);
		
		// Add button for browsers with JavaScript disabled
		$content .= "<noscript>\n";
		$content .= '<input type="submit">' . "\n";
		$content .= "</noscript>\n";
		
		if (!$this->_use_ajax_progress_indicators) {
			// Build autosubmitter
			if (empty($delaypersession)) {
				$delaypersession = '0';
			}
			$content .= '<script type="text/javascript">' . "\n";
			$content .= 'window.setTimeout(\'location.href="' . $_SERVER["PHP_SELF"] .
				"?start=$linenumber&fn=" . urlencode($curfilename) .
				"&foffset=$foffset&totalqueries=$totalqueries\";',500+$delaypersession);";
			$content .= '</script>' . "\n";
		}
		
		// Build back link
		$params = zen_get_all_get_params();
		
		$category_name_result = $db->Execute("
			SELECT
				cd.categories_name
			FROM
				" . TABLE_CATEGORIES_DESCRIPTION . " cd
			WHERE
				cd.categories_id = '" . (int) $_GET['sub-cat-id'] . "'
			AND
				cd.language_id = '" . (int) $_SESSION['languages_id'] . "';");
		
		if (!$category_name_result->EOF) {
			$category_name = $category_name_result->fields['categories_name'];
		}
		
		$back_up_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to.png') .
			' <a href="' . $back_up_link . '">' .
			sprintf(TEXT_BACK_TO, htmlentities($category_name, ENT_COMPAT, CHARSET)) . '</a></p>' . "\n";
		
		$content .= '</form>' . "\n";
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _buildProductsAutogenAJAXController()
	
	/**
	 * Builds the HTML/JavaScript needed to run the autogeneration process as an AJAX process.
	 *
	 * @access  protected
	 * @return  string    The HTML/JavaScript source for the AJAX controller.
	 */
	protected function _buildProductsAutogenAJAXController()
	{
		// Output the code necessary to have the script run using AJAX
		ob_start();
		
		$params = zen_get_all_get_params();
		
		$params .= 'ajax=1';
		
		$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
	?>
<script type="text/javascript">			

// Build the parameters to be sent
function buildParams(include_disabled_products, prods_to_gen, num_prods, num_prods_proc, num_prods_gen,
		num_prods_uc, num_prods_f, cur_prod_pos);
	
	return 'include-disabled-products=' + include_disabled_products + '&prods-to-gen=' + prods_to_gen +
		'&num-prods=' + num_prods + '&num-prods-proc=' + num_prods_proc + '&num-prods-gen=' + num_prods_gen +
		'&num-prods-uc=' + num_prods_uc + '&num-prods-f=' + num_prods_f + '&cur-prod-pos=' + cur_prod_pos;
}

// Extracts text from XML element (itemname must be unique)
function getValueFromXML(xml, id) {
	if (!xml ||xml.getElementsByTagName(id)[0] == undefined || xml.getElementsByTagName(id)[0].firstChild == undefined) {
		return null;
	}
	
	return xml.getElementsByTagName(id)[0].firstChild.data;
}

var http_request = false;

var uri = '<?php echo $link; ?>';

// Build parameters for first stage
var first_stage = true;

<?php 
	$prod_gen = implode(',', $this->_products_to_generate);
	if (empty($prod_gen)) {
		$prod_gen = '0';
	}?>
var params_to_send = buildParams(<?php echo ($this->_include_disabled_products ? 1 : 0); ?>,
	<?php echo "'" . $prod_gen . "'"; ?>,
	<?php echo $this->_num_products_inc; ?>,
	<?php echo $this->_num_products_processed; ?>,
	<?php echo $this->_num_product_mappings_generated; ?>,
	<?php echo $this->_num_product_mappings_unchanged; ?>,
	<?php echo $this->_num_product_mappings_failed; ?>,
	<?php echo $this->_cur_product_pos; ?>);

function makeRequest(uri, params_to_send)
{
	http_request = false;
	
	if (typeof(console.log) == 'function') {
		debugger; //PRODUCTS
	}
	
	if (typeof zcJS == "undefined" || !zcJS) {
		if (window.XMLHttpRequest) { 
			// Mozilla,...
			http_request = new XMLHttpRequest();
		
			if (http_request.overrideMimeType) {
				http_request.overrideMimeType("text/xml");
			}
		} else if (window.ActiveXObject) { 
			// IE
			try {
				http_request = new ActiveXObject("Msxml2.XMLHTTP");
			} catch(e) {
				try {
					http_request = new ActiveXObject("Microsoft.XMLHTTP");
				} catch(e) {}
			}
		}
	} else {
	}
	
	if (!http_request && (typeof zcJS == "undefined" || !zcJS)) {
		alert("Cannot create an XMLHTTP instance");
		return false;
	}
	
	// Indicate that an AJAX request is taking place
	var page_title_el = document.getElementById('autogeneration-title');
	
	if (page_title_el != undefined) {
		ajax_request_in_progress_el = document.getElementById('ajax-request-in-progress');
		
		if (ajax_request_in_progress_el == undefined) {
			ajax_request_in_progress_el = document.createElement('div');
			ajax_request_in_progress_el.setAttribute('id', 'ajax-request-in-progress');
			
			image_el = document.createElement('img');
			image_el.setAttribute('src', '<?php echo DIR_WS_IMAGES . 'ceon-star-processing.gif'; ?>');
			
			ajax_request_in_progress_el.appendChild(image_el);
			
			var text_node = document.createTextNode('<?php echo TEXT_AJAX_REQUEST_IN_PROGRESS; ?>');
			
			ajax_request_in_progress_el.appendChild(text_node);
			
			page_title_el.appendChild(ajax_request_in_progress_el);
		}
		
		ajax_request_in_progress_el.style.display = 'block';
	}
	
	if (typeof zcJS == "undefined" || !zcJS) {
		http_request.onreadystatechange = handleServerResponse;
		http_request.open("POST", uri, true);
	
		http_request.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
		http_request.setRequestHeader("Content-length", params_to_send.length);
		http_request.setRequestHeader("Connection", "close");
	
		http_request.send(params_to_send);
	} else {
		if (typeof(console.log) == 'function') {
			debugger; //prod
		}
		var option = {
			url: "ajax.php?act=ajaxCeonURIMapping&method=reload<?php 
			$other_params = zen_get_all_get_params();
			if (zen_not_null($other_params) && $other_params != '&') {
				$other_params = rtrim($other_params, '&');
				echo "&" . $other_params;
			}
			?>",
			data: params_to_send,
			timeout : 50000,
			dataType : "xml"
		};
		zcJS.ajax(option).done(
			function (response,textStatus,jqXHR) {
				http_request = jqXHR;

				if (http_request.responseXML) {
					handleServerResponse();
				} else
				if (http_request.responseText) {
					let parser = new DOMParser();
					try {
						http_request.responseXML = parser.parseFromString(http_request.responseText, "text/xml");
						handleServerResponse();
					} catch(e) {
					}
				}
				if (http_request.responseJSON) {
					if (typeof(console.log) == 'function') {
						debugger; // products
					}
					http_request.responseXML = json2xml(JSON.stringify(http_request.responseJSON.data));
					let parser = new DOMParser();
					http_request.responseXML = parser.parseFromString(http_request.responseXML, "text/xml");
					handleServerResponse();
				}
			}
		).fail( function(jqXHR,textStatus,errorThrown) {
			if (jqXHR.statusText == 'timeout') {
				if (typeof(console.log) == 'function') {
					debugger;
				}
				if (!this.tryCount) {
					this.tryCount = 1;
				} else {
					this.tryCount++;
				}
				if (this.tryCount <= 3) {
					if (typeof(console.log) == 'function') {
						debugger;
					}
					makeRequest(uri, params_to_send);
					return;
				}
				this.tryCount = 0;
				return;
			}
			if (typeof(console.log) == 'function') {
				console.log('In here');
				console.log(jqXHR);
				console.log(textStatus);
				console.log(errorThrown);
			}
		});
	}
}

function handleServerResponse() 
{
	// Waiting for correct response
	if (typeof(console.log) == 'function') {
		debugger; //products
	}
	if (http_request.readyState != 4) {
		return;
	}
	
	if (http_request.status != 200) {
		alert("Page unavailable, or wrong url!");
		return;
	}
	
	var xml = http_request.responseXML;
	if (!xml) {
		alert('not XML');
		if (typeof(console.log) == 'function') {
			debugger; //prod
		}
		return;
	}
	
	// Check that expected XML was received
	if (xml.getElementsByTagName('root').length == 0) {
		var text = http_request.responseText;
		if (typeof(console.log) == 'function') {
			debugger; //prod
		}
		document.open();
		document.write(text);
		document.close();
		
		return;
	}
	
	// Build parameters for next stage
	var include_disabled_products = parseInt(getValueFromXML(xml, 'include_disabled_products'));
	var products_to_generate = getValueFromXML(xml, 'products_to_generate');
	var num_products_inc = parseInt(getValueFromXML(xml, 'num_products_inc'));
	var num_products_processed = parseInt(getValueFromXML(xml, 'num_products_processed'));
	var num_product_mappings_generated = parseInt(getValueFromXML(xml, 'num_product_mappings_generated'));
	var num_product_mappings_unchanged = parseInt(getValueFromXML(xml, 'num_product_mappings_unchanged'));
	var num_product_mappings_failed = parseInt(getValueFromXML(xml, 'num_product_mappings_failed'));
	var cur_product_pos = parseInt(getValueFromXML(xml, 'cur_product_pos'));
	
	var num_products_processed_el = document.getElementById('num-prods-processed');
	
	if (num_products_processed_el != undefined) {
		var text_node = document.createTextNode(num_products_processed);
		
		num_products_processed_el.removeChild(num_products_processed_el.firstChild);
		
		num_products_processed_el.appendChild(text_node);
	}
	
	// Update products progress bar
	var progress_bar_el = document.getElementById('products-progress');
	
	if (progress_bar_el != undefined) {
		// Adjust for padding width of 2% each side
		if (num_products_inc == 0) {
			percentage_completed = 100;
		} else if (num_products_processed == 0) {
			percentage_completed = 0;
		} else {	
			percentage_completed = (100 / (num_products_inc / num_products_processed));
		}
		
		progress_bar_el.style.width = percentage_completed.toFixed(2) + '%';
		
		if (num_products_inc == 0) {
			percentage_completed = 100;
		} else if (num_products_processed == 0) {
			percentage_completed = 0;
		} else {
			percentage_completed = (100 / (num_products_inc / num_products_processed));
		}
		
		var text_node = document.createTextNode(percentage_completed.toFixed(2) + '%');
		
		progress_bar_el.removeChild(progress_bar_el.firstChild);
		
		progress_bar_el.appendChild(text_node);
	}
	
	
	var num_product_mappings_generated_count_el = document.getElementById('num-prods-generated-count');
	
	if (num_product_mappings_generated_count_el != undefined) {
		var text_node = document.createTextNode(num_product_mappings_generated);
		
		num_product_mappings_generated_count_el.removeChild(
			num_product_mappings_generated_count_el.firstChild);
		
		num_product_mappings_generated_count_el.appendChild(text_node);
	}
	
	
	var num_product_mappings_unchanged_el = document.getElementById('num-prods-unchanged');
	
	var num_product_mappings_unchanged_count_el = document.getElementById('num-prods-unchanged-count');
	
	if (num_product_mappings_unchanged > 0 && num_product_mappings_unchanged_el != undefined &&
			num_product_mappings_unchanged_count_el != undefined) {
		
		num_product_mappings_unchanged_el.style.display = 'block';
		
		var text_node = document.createTextNode(num_product_mappings_unchanged);
		
		num_product_mappings_unchanged_count_el.removeChild(
			num_product_mappings_unchanged_count_el.firstChild);
		
		num_product_mappings_unchanged_count_el.appendChild(text_node);
	}
	
	// Output information about any mappings that were generated
	if (num_product_mappings_generated > 0) {
		// Create the listing div if it doesn't exist yet
		var products_generated_listing_el = document.getElementById('products-generated-listing');
		
		if (products_generated_listing_el == undefined) {
			products_generated_listing_el = document.createElement('div');
			products_generated_listing_el.setAttribute('id', 'products-generated-listing');
			
			num_product_mappings_unchanged_el.appendChild(products_generated_listing_el);
		}
		
		product_mappings_generated = xml.getElementsByTagName('products_generated');
		
		if (product_mappings_generated != undefined && product_mappings_generated[0] != undefined &&
				products_generated_listing_el != undefined) {
			
			product_mappings_generated = product_mappings_generated[0].getElementsByTagName('product');
			
			if (product_mappings_generated != undefined) {
				// Build info needed for auto-scrolling
				var start_height = 0;
				
				if (products_generated_listing_el.scrollHeight > 0) {
					start_height = products_generated_listing_el.scrollHeight;
				} else if (products_generated_listing_el.offsetHeight > 0) {
					start_height = products_generated_listing_el.offsetHeight;
				}
				
				for (var prod_i = 0; prod_i < product_mappings_generated.length; prod_i++) {
					message = getValueFromXML(product_mappings_generated[prod_i], 'message');
					
					link = getValueFromXML(product_mappings_generated[prod_i], 'link');
					
					link_title = getValueFromXML(product_mappings_generated[prod_i], 'link_title');
					
					edit_link = getValueFromXML(product_mappings_generated[prod_i], 'edit_link');
					
					edit_link_title =
						getValueFromXML(product_mappings_generated[prod_i], 'edit_link_title');
					
					if (message != undefined) {
						var p_el = document.createElement('p');
						
						p_el.appendChild(document.createTextNode(message + ' '));
						
						if (link != undefined && link_title != undefined) {
							var link_el = document.createElement('a');
							link_el.setAttribute('href', link);
							link_el.setAttribute('target', '_blank');
							link_el.appendChild(document.createTextNode(link_title));
							
							p_el.appendChild(link_el);
						}
						
						if (edit_link != undefined && edit_link_title != undefined) {
							var edit_link_el = document.createElement('a');
							edit_link_el.setAttribute('href', edit_link);
							edit_link_el.setAttribute('target', '_blank');
							edit_link_el.appendChild(document.createTextNode(edit_link_title));
							
							p_el.appendChild(document.createTextNode(' - '));
							
							p_el.appendChild(edit_link_el);
						}
						
						products_generated_listing_el.appendChild(p_el);
					}
				}
				
				// Scroll to the bottom if possible
				var current_height = 0;
				
				if (products_generated_listing_el.scrollHeight > 0) {
					current_height = products_generated_listing_el.scrollHeight;
				} else if (products_generated_listing_el.offsetHeight > 0) {
					current_height = products_generated_listing_el.offsetHeight;
				}
				
				// Height of updated div must allowed for in test to see if user has scrolled the
				// div
				var added_height = current_height - start_height;
				
				if (first_stage || (current_height - products_generated_listing_el.scrollTop -
						((products_generated_listing_el.style.pixelHeight) ?
						products_generated_listing_el.style.pixelHeight :
						products_generated_listing_el.offsetHeight) < (added_height + 25))) {
					products_generated_listing_el.scrollTop = current_height;
				}
			}
		}
	}
	
	// Output information about any errors that were encountered
	if (num_product_mappings_failed > 0) {
		var num_product_mappings_failed_el = document.getElementById('num-prods-failed');
		
		var num_product_mappings_failed_count_el = document.getElementById('num-prods-failed-count');
		
		if (num_product_mappings_failed_el != undefined && num_product_mappings_failed_count_el != undefined) {
			// Make sure info is visible
			num_product_mappings_failed_el.style.display = 'block';
			
			var text_node = document.createTextNode(num_product_mappings_failed);
			
			if (num_product_mappings_failed_count_el.firstChild != undefined) {
				num_product_mappings_failed_count_el.removeChild(num_product_mappings_failed_count_el.firstChild);
			}
			
			num_product_mappings_failed_count_el.appendChild(text_node);
			
			// Create the listing div if it doesn't exist yet
			var product_errors_listing_el = document.getElementById('product-errors-listing');
			
			if (product_errors_listing_el == undefined) {
				product_errors_listing_el = document.createElement('div');
				product_errors_listing_el.setAttribute('id', 'product-errors-listing');
				
				num_product_mappings_failed_el.appendChild(product_errors_listing_el);
			}
			
			product_mappings_errors = xml.getElementsByTagName('products_errors');
			
			if (product_mappings_errors != undefined && product_mappings_errors[0] != undefined) {
				product_mappings_errors = product_mappings_errors[0].getElementsByTagName('product');
				
				if (product_mappings_errors != undefined) {
					// Build info needed for auto-scrolling
					var start_height = 0;
					
					if (product_errors_listing_el.scrollHeight > 0) {
						start_height = product_errors_listing_el.scrollHeight;
					} else if (product_errors_listing_el.offsetHeight > 0) {
						start_height = product_errors_listing_el.offsetHeight;
					}
					
					for (var prod_i = 0; prod_i < product_mappings_errors.length; prod_i++) {
						message = getValueFromXML(product_mappings_errors[prod_i], 'message');
						
						link = getValueFromXML(product_mappings_errors[prod_i], 'link');
						
						link_title = getValueFromXML(product_mappings_errors[prod_i], 'link_title');
						
						edit_link = getValueFromXML(product_mappings_errors[prod_i], 'edit_link');
						
						edit_link_title = getValueFromXML(product_mappings_errors[prod_i], 'edit_link_title');
						
						if (message != undefined) {
							var p_el = document.createElement('p');
							
							p_el.appendChild(document.createTextNode(message + ' '));
							
							if (link != undefined && link_title != undefined) {
								var link_el = document.createElement('a');
								link_el.setAttribute('href', link);
								link_el.setAttribute('target', '_blank');
								link_el.appendChild(document.createTextNode(link_title));
								
								p_el.appendChild(link_el);
							}
							
							if (edit_link != undefined && edit_link_title != undefined) {
								var edit_link_el = document.createElement('a');
								edit_link_el.setAttribute('href', edit_link);
								edit_link_el.setAttribute('target', '_blank');
								edit_link_el.appendChild(document.createTextNode(edit_link_title));
								
								p_el.appendChild(document.createTextNode(' - '));
								
								p_el.appendChild(edit_link_el);
							}
							
							product_errors_listing_el.appendChild(p_el);
						}
					}
					
					// Scroll to the bottom if possible
					var current_height = 0;
					
					if (product_errors_listing_el.scrollHeight > 0) {
						current_height = product_errors_listing_el.scrollHeight;
					} else if (product_errors_listing_el.offsetHeight > 0) {
						current_height = product_errors_listing_el.offsetHeight;
					}
					
					// Height of updated div must allowed for in test to see if user has scrolled the div
					var added_height = current_height - start_height;
					
					if (first_stage || (current_height - product_errors_listing_el.scrollTop -
							((product_errors_listing_el.style.pixelHeight) ?
							product_errors_listing_el.style.pixelHeight :
							product_errors_listing_el.offsetHeight) < (added_height + 25))) {
						product_errors_listing_el.scrollTop = current_height;
					}
				}
			}
		}
	}
	
	if (first_stage == true) {
		first_stage = false;
	}
	
	// Has the process completed?
	if (num_products_inc <= num_products_processed) {
		return;
	}
	
	params_to_send = buildParams(include_disabled_products, products_to_generate, num_products_inc,
		num_products_processed, num_product_mappings_generated, num_product_mappings_unchanged,
		num_product_mappings_failed, cur_product_pos);
	
	// Begin next stage of process
	makeRequest(uri, params_to_send);
}

// Begin process!
window.setTimeout("makeRequest(uri, params_to_send)", 50);

</script>
	<?php
		$content = ob_get_clean();
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ handleProductsAJAXRequest()
	
	/**
	 * Handles an AJAX request for the products auto-generation section.
	 *
	 * @access  public
	 * @return  none
	 */
	public function handleProductsAJAXRequest()
	{
		// Start the control timer
		$this->_autogeneration_start_timestamp = time();
		
		if ($this->_debug_ajax_to_file) {
			$this->_ajax_debug_file = fopen($_SESSION['ceon_umm_ajax_debug_file'], 'a');
			
			fwrite($this->_ajax_debug_file, "Starting AJAX REQUEST at " . date('Ymd His', time()) . "\n\n");
		}
		
		// Get the data posted for this request
		$this->_include_disabled_products =
			($this->_posted_data['include-disabled-productss'] == 1 ? true : false);
		
		$this->_products_to_generate = explode(',', $this->_posted_data['prods-to-gen']);
		
		$this->_num_products_inc = $this->_posted_data['num-prods'];
		$this->_num_products_processed = $this->_posted_data['num-prods-proc'];
		$this->_num_product_mappings_generated = $this->_posted_data['num-prods-gen'];
		$this->_num_product_mappings_unchanged = $this->_posted_data['num-prods-uc'];
		$this->_num_product_mappings_failed = $this->_posted_data['num-prods-f'];
		$this->_cur_product_pos = $this->_posted_data['cur-prod-pos'];
		
		// Carry out next stage of process, if there is one
		if ($this->_num_products_inc > $this->_num_products_processed) {
			$this->_autogenerateProductsURIMappings();
		}
		
		// Return XML for request
		header('Content-Type: application/xml');
		header('Cache-Control: no-cache');
		
		ob_start();
		
		echo '<?xml version="1.0" encoding="' . CHARSET .'"?>';
		echo '<root>';
		echo '<include_disabled_products>';
		echo ($this->_include_disabled_products ? 1 : 0);
		echo '</include_disabled_products>';
		echo '<products_to_generate>';
		$prod_pos = implode(',', $this->_categories_to_products);
		if (empty($prod_pos)) {
			$prod_pos = '0';
		}
		echo $prod_pos;
		echo '</products_to_generate>';
		echo '<num_products_inc>';
		echo $this->_num_products_inc;
		echo '</num_products_inc>';
		echo '<num_products_processed>';
		echo $this->_num_products_processed;
		echo '</num_products_processed>';
		echo '<num_product_mappings_generated>';
		echo $this->_num_product_mappings_generated;
		echo '</num_product_mappings_generated>';
		echo '<num_product_mappings_unchanged>';
		echo $this->_num_product_mappings_unchanged;
		echo '</num_product_mappings_unchanged>';
		echo '<num_product_mappings_failed>';
		echo $this->_num_product_mappings_failed;
		echo '</num_product_mappings_failed>';
		echo '<cur_product_pos>';
		echo $this->_cur_product_pos;
		echo '</cur_product_pos>';
		
		if (sizeof($this->_product_mappings_generated_messages) > 0) {
			echo '<products_generated>';
			
			foreach ($this->_product_mappings_generated_messages as $message) {
				echo '<cat><message>';
				echo htmlspecialchars($message['message']);
				echo '</message><link>';
				echo htmlspecialchars($message['link']);
				echo '</link><link_title>';
				echo htmlspecialchars($message['link_title']);
				echo '</link_title>';
				
				if (isset($message['edit_link'])) {
					echo '<edit_link>';
					echo htmlspecialchars($message['edit_link']);
					echo '</edit_link><edit_link_title>';
					echo htmlspecialchars($message['edit_link_title']);
					echo '</edit_link_title>';
				}
				
				echo '</cat>';
			}
			
			echo '</products_generated>';
		}
		
		if (sizeof($this->_product_mappings_error_messages) > 0) {
			echo '<products_errors>';
			
			foreach ($this->_product_mappings_error_messages as $message) {
				echo '<cat><message>';
				echo htmlspecialchars($message['message']);
				echo '</message><link>';
				echo htmlspecialchars($message['link']);
				echo '</link><link_title>';
				echo htmlspecialchars($message['link_title']);
				echo '</link_title>';
				
				if (isset($message['edit_link'])) {
					echo '<edit_link>';
					echo htmlspecialchars($message['edit_link']);
					echo '</edit_link><edit_link_title>';
					echo htmlspecialchars($message['edit_link_title']);
					echo '</edit_link_title>';
				}
				
				echo '</cat>';
			}
			
			echo '</products_errors>';
		}
		
		echo "</root>";	
		
		$xml = ob_get_clean();
		
		echo $xml;
		
		include('includes/application_bottom.php');
		
		exit();
	}
	
	// }}}
}

// }}}
