<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate Categories/Products URIs Listing Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageCategoriesProducts.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Listing class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMListing.php');


// {{{ CeonUMMCreateManageCategoriesProducts

/**
 * Displays a listing of categories/subcategories/products with controls to manage the URIs/URI
 * mapping templates (template management for categories only).
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageCategoriesProducts extends CeonUMMListing
{
	// {{{ properties
	
	/**
	 * The ID of a category being managed or of the category containing a product being managed.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_category_id = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' .
			'ceon_umm_create_manage_categories_products.php');
		
		if (!defined('TEXT_CATEGORY_PRODUCT_MAPPINGS_TITLE') &&
				$_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' .
				'ceon_umm_create_manage_categories_products.php');
		}
		
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ buildCategoryListing()
	
	/**
	 * Builds a list of all the subcategories for the current category, information about any
	 * mappings and mapping templates for the category and its subcategories, and gadgets to enable
	 * actions to be carried out against the category or any of the subcategories.
	 *
	 * @access  public
	 * @return  none
	 */
	public function buildCategoryListing()
	{
		global $db, $languages, $num_languages;
		
		$content = '';
		
		// Variables hold information about the parent category
		$main_category_name = '';
		$main_category_mappings = array();
		
		// Default to the top category for the site
		$this->_category_id = (isset($_GET['cat-id']) ? (int) $_GET['cat-id'] : 0);
		
		if ($this->_category_id != 0) {
			$main_category_name =
				zen_get_category_name($this->_category_id, $_SESSION['languages_id']);
			
			// Get the URI mapping(s) for the current parent category, if any
			$columns_to_retrieve = array(
				'uri',
				'language_id'
				);
			
			$selections = array(
				'main_page' => FILENAME_DEFAULT,
				'associated_db_id' => $this->_category_id,
				'current_uri' => 1
				);
			
			$main_category_mappings_result =
				$this->getURIMappingsResultset($columns_to_retrieve, $selections);
			
			while (!$main_category_mappings_result->EOF) {
				$main_category_mappings[$main_category_mappings_result->fields['language_id']] =
					$main_category_mappings_result->fields['uri'];
				
				$main_category_mappings_result->MoveNext();
			}
		}
		
		
		// Variables hold information about the default mapping templates for the level of category
		// being displayed
		$default_subcategory_templates = array();
		$default_product_templates = array();
		
		// Progress back through the list of parent categories for the category being displayed,
		// getting the most uniquely specified templates
		$current_category_id = $this->_category_id;
		
		do {
			// Get any subcategory mapping template(s) for this category
			$columns_to_retrieve = array(
				'mapping_template',
				'language_id'
				);
			
			$selections = array(
				'category_id' => $current_category_id
				);
			
			$default_mapping_templates_result = $this->getURIMappingTemplatesResultset('category',
				$columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$default_mapping_templates_result->EOF) {
				$language_id = $default_mapping_templates_result->fields['language_id'];
				
				if (!isset($default_subcategory_templates[$language_id])) {
					// A specific default template has not been set for this language
					$default_subcategory_templates[$language_id] =
						$default_mapping_templates_result->fields['mapping_template'];
				}
				
				$default_mapping_templates_result->MoveNext();
			}
			
			// Get any product mapping template(s) for this category
			$default_mapping_templates_result = $this->getURIMappingTemplatesResultset('product',
				$columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$default_mapping_templates_result->EOF) {
				$language_id = $default_mapping_templates_result->fields['language_id'];
				
				if (!isset($default_product_templates[$language_id])) {
					// A specific default template has not been set for this language
					$default_product_templates[$language_id] =
						$default_mapping_templates_result->fields['mapping_template'];
				}
				
				$default_mapping_templates_result->MoveNext();
			}
			
			if (sizeof($default_subcategory_templates) == $num_languages &&
					sizeof($default_product_templates) == $num_languages) {
				// Have got a template for each mapping type, for each language, job's done!
				break;
			}
			
			if ($current_category_id == 0) {
				// Can't move any further up, at the top!
				break;
			}
			
			$parent_category_id_result = $db->Execute("
				SELECT
					parent_id
				FROM
					" . TABLE_CATEGORIES . "
				WHERE
					categories_id = '" . (int) $current_category_id . "';");
			
			if ($parent_category_id_result->EOF) {
				// Should never happen
				break;
			} else {
				$current_category_id = $parent_category_id_result->fields['parent_id'];
			}
		} while (1);
		
		
		// Get the information for the subcategories for this category
		$subcategories = array();
		$subcategories_mappings = array();
		$subcategories_subcategory_templates = array();
		$subcategories_product_templates = array();
		
		$subcategories_result = $db->Execute("
			SELECT
				c.categories_id,
				cd.categories_name,
				c.categories_status
            FROM
				" . TABLE_CATEGORIES . " c
			LEFT JOIN
				" . TABLE_CATEGORIES_DESCRIPTION . " cd
            ON
				c.categories_id = cd.categories_id
            AND
				cd.language_id = '" . (int) $_SESSION['languages_id'] . "'
            WHERE
				c.parent_id = '" . $this->_category_id . "'
            ORDER BY
				c.sort_order, cd.categories_name");
		
		while (!$subcategories_result->EOF) {
			// Does the subcategory contain further subcategories or is it assumed to contain
			// products?
			$sub_subcategories_result = $db->Execute("
				SELECT
					c.categories_id
				FROM
					" . TABLE_CATEGORIES . " c
				WHERE
					c.parent_id = '" . $subcategories_result->fields['categories_id'] . "'
				LIMIT
					1");
			
			$subcategories[] = array(
				'id' => $subcategories_result->fields['categories_id'],
				'name' => $subcategories_result->fields['categories_name'],
				'has_subcategories' => (!$sub_subcategories_result->EOF),
				'is_disabled' =>
					($subcategories_result->fields['categories_status'] == 0 ? true: false)
				);
			
			$subcategories_result->MoveNext();
		}
		
		
		// Get the current URI mapping(s) for the subcategories, if any
		$columns_to_retrieve = array(
			'uri',
			'language_id'
			);
		
		$selections = array(
			'main_page' => FILENAME_DEFAULT,
			'current_uri' => 1
			);
		
		foreach ($subcategories as $subcategory) {
			$subcategories_mappings[$subcategory['id']] = array();
			
			$selections['associated_db_id'] = $subcategory['id'];
			
			$subcategory_mapping_result =
				$this->getURIMappingsResultset($columns_to_retrieve, $selections);
			
			while (!$subcategory_mapping_result->EOF) {
				$subcategories_mappings[$subcategory['id']]
					[$subcategory_mapping_result->fields['language_id']] =
					$subcategory_mapping_result->fields['uri'];
				
				$subcategory_mapping_result->MoveNext();
			}
		}
		
		
		// Get the subcategory mapping template(s) for the subcategories, if any
		$columns_to_retrieve = array(
			'mapping_template',
			'language_id'
			);
		
		foreach ($subcategories as $subcategory) {
			if (!$subcategory['has_subcategories']) {
				$subcategory_mapping_result->MoveNext();
				
				continue;
			}
			
			$selections = array(
				'category_id' => $subcategory['id']
				);
			
			$subcategory_mapping_templates_result = $this->getURIMappingTemplatesResultset(
				'category', $columns_to_retrieve, $selections);
			
			while (!$subcategory_mapping_templates_result->EOF) {
				if (!isset($subcategories_subcategory_templates[$subcategory['id']])) {
					$subcategories_subcategory_templates[$subcategory['id']] = array();
				}
				
				$language_id = $subcategory_mapping_templates_result->fields['language_id'];
				
				$subcategories_subcategory_templates[$subcategory['id']][$language_id] =
					$subcategory_mapping_templates_result->fields['mapping_template'];
				
				$subcategory_mapping_templates_result->MoveNext();
			}
		}
		
		
		// Get the product mapping template(s) for the subcategories, if any
		foreach ($subcategories as $subcategory) {
			$selections = array(
				'category_id' => $subcategory['id']
				);
			
			$subcategory_mapping_templates_result = $this->getURIMappingTemplatesResultset(
				'product', $columns_to_retrieve, $selections);
			
			while (!$subcategory_mapping_templates_result->EOF) {
				if (!isset($subcategories_product_templates[$subcategory['id']])) {
					$subcategories_product_templates[$subcategory['id']] = array();
				}
				
				$language_id = $subcategory_mapping_templates_result->fields['language_id'];
				
				$subcategories_product_templates[$subcategory['id']][$language_id] =
					$subcategory_mapping_templates_result->fields['mapping_template'];
				
				$subcategory_mapping_templates_result->MoveNext();
			}
		}
		
		// Do all subcategories have their own templates?
		// @TODO
		
		
		// Some stores (wrongly/unexpectedly) have products in categories that have subcategories.
		// Must check if this is the case for this category
		$products_result = $db->Execute("
			SELECT
				p.products_id
            FROM
				" . TABLE_PRODUCTS . " p
            WHERE
				p.master_categories_id = '" . $this->_category_id . "';");
		
		$num_products_in_category = 0;
		
		// Simply determine if there is more than one product (for pluralisation of text)
		while (!$products_result->EOF) {
			$num_products_in_category++;
			
			if ($num_products_in_category > 1) {
				break;
			}
			
			$products_result->MoveNext();
		}
		
		
		// Default to generating mappings for products within categories/subcategories
		$autogenerate_category_mappings = 'inc-products';
		
		// Default to excluding disabled categories/products
		$autogenerate_inc_disabled = false;
		
		// Default to all being selected
		$this->_select_all = true;
		
		$selected_cats = array();
		
		foreach ($subcategories as $subcategory) {
			$selected_cats[] = $subcategory['id'];
		}
		
		// All data has been gathered, build the page //////////////////////////////////////////////
		
		// Build the title with the current category's details
		$panel_title = TEXT_CATEGORY_PRODUCT_MAPPINGS_TITLE;
		
		$panel_intro = '<p>' . TEXT_CATEGORY_PRODUCT_MAPPINGS_INTRO . '</p>' . "\n";
		
		$panel_intro .= '<ul>' . "\n";
		
		$panel_intro .= '<li>' . TEXT_CATEGORY_PRODUCT_MAPPINGS_INTRO_AUTOGENERATE_BUTTONS .
			'</li>' . "\n";
		
		$panel_intro .= '<li class="LastIntro">' .
			TEXT_CATEGORY_PRODUCT_MAPPINGS_INTRO_CATEGORY_CONTROLS . '</li>' . "\n";
		
		$panel_intro .= '</ul>' . "\n";
		
		if ($main_category_name == '') {
			$main_category_name = TEXT_ROOT_CATEGORY;
		}
		
		$content .= '<h1>' . $panel_title . '</h1>' . "\n";
		
		$content .= $panel_intro;
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action .
			'" method="post" name="category-listing-form" id="category-listing-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Add the autogenerate controls for this category /////////////////////////////////////////
		$content .= '<div class="AutogenerateControlsWrapper" id="autogenerate-controls-top">' .
			"\n";
		
		$content .= '<fieldset class="AutogenerateControls"><legend>' .
			TEXT_AUTOGENERATE_MAPPINGS . '</legend>' . "\n";
		
		$content .= '<p class="AutogenerationOptions">' . "\n";
		
		$content .= '<input type="radio" name="autogenerate-category-mappings"' .
			' value="inc-products" id="autogenerate-category-mappings-inc-products"' .
			($autogenerate_category_mappings == 'inc-products' ? ' checked="checked"' : '') .
			' />' . "\n";
		
		$content .= '<label for="autogenerate-category-mappings-inc-products">' .
			TEXT_SELECTED_CATEGORIES_AND_ALL_PRODUCTS . '</label>' . "\n";
		
		$content .= '<br /><input type="radio" name="autogenerate-category-mappings"' .
			' value="exc-products" id="autogenerate-category-mappings-exc-products"' .
			($autogenerate_category_mappings == 'exc-products' ? ' checked="checked"' : '') .
			' />' . "\n"; 
		
		$content .= '<label for="autogenerate-category-mappings-exc-products">' .
			TEXT_SELECTED_CATEGORIES_BUT_NOT_PRODUCTS . '</label>' .'<br />' . "\n";
		
		$content .= '<br /><input type="checkbox" name="inc-disabled" value="1"' .
			' id="inc-disabled"' .
			($autogenerate_inc_disabled == true ? ' checked="checked"' : '') .
			' />' . "\n"; 
		
		$content .= '<label for="autogenerate-category-mappings-exc-products">' .
			TEXT_INC_DISABLED_CATEGORIES_PRODUCTS . '</label>' . '</p>' . "\n";
		
		// Main button to generate URI mappings for all selected categories/subcategories/products
		$button_title = TEXT_AUTOGENERATE_URI_MAPPINGS_BUTTON;
		$button_desc = TEXT_AUTOGENERATE_CATEGORIES_URI_MAPPINGS_BUTTON_HOVER_INFO;
		
		$content .= '<input type="submit" class="AutogenerateURIMappingsButton"' .
			' name="auto-gen-cats" id="auto-gen-cats"' .
			' value="' . $button_title . '" title="' . $button_desc . '" />' . "\n"; 
		
		$content .= '</fieldset>' . "\n";
		
		$content .= '</div>' . "\n";
		
		
		// Build the info for the current category /////////////////////////////////////////////////
		$content .= '<h2 class="CreateManageCategoryTitle">' .
			htmlentities($main_category_name, ENT_COMPAT, CHARSET) . '</h2>' . "\n";
		
		// Build the list of current mappings for the main category
		if ($this->_category_id != 0) {
			$content .= '<ul class="MainCategoryMappings">';
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li class="MainCategoryMapping">';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				if (isset($main_category_mappings[$languages[$i]['id']])) {
					$content .= '<a href="' . HTTP_SERVER .
						$main_category_mappings[$languages[$i]['id']] . '" target="_blank">' .
						$main_category_mappings[$languages[$i]['id']] . '</a>';
				} else {
					$content .= TEXT_NO_CURRENT_MAPPING;
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
		}
		
		// Build the list of mapping templates for the main category
		if ($this->_category_id == 0) {
			if ($num_languages == 1) {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_DEFAULT_MAPPING_TEMPLATE_FOR_SUBCATEGORIES . '&nbsp;' . '</label>' . '</p>' .
					"\n";
			} else {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_DEFAULT_MAPPING_TEMPLATES_FOR_SUBCATEGORIES . '&nbsp;' . '</label>' . '</p>' .
					"\n";
			}
		} else {
			if ($num_languages == 1) {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_MAPPING_TEMPLATE_FOR_SUBCATEGORIES . '&nbsp;' . '</label>' . '</p>' .
					"\n";
			} else {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_MAPPING_TEMPLATES_FOR_SUBCATEGORIES . '&nbsp;' . '</label>' . '</p>' .
					"\n";
			}
		}
		
		$content .= '<ul class="MainMappingTemplates">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li>';
			
			$content .= '<span class="MappingLanguageIcon">' . zen_image(DIR_WS_CATALOG_LANGUAGES .
				$languages[$i]['directory'] . '/images/' . $languages[$i]['image'],
				$languages[$i]['name']) . '</span>';
			
			$content .= $this->_addTooltipDescriptionsOfPlacementTags('category',
				$default_subcategory_templates[$languages[$i]['id']]) . '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		if ($this->_category_id == 0) {
			if ($num_languages == 1) {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_DEFAULT_MAPPING_TEMPLATE_FOR_SUBCATEGORIES_PRODUCTS . '&nbsp;' . '</label>' .
					'</p>' . "\n";
			} else {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_DEFAULT_MAPPING_TEMPLATES_FOR_SUBCATEGORIES_PRODUCTS . '&nbsp;' . '</label>' .
					'</p>' . "\n";
			}
		} else {
			if ($num_languages == 1) {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_MAPPING_TEMPLATE_FOR_SUBCATEGORIES_PRODUCTS . '&nbsp;' . '</label>' .
					'</p>' . "\n";
			} else {
				$content .= '<p class="MainMappingTemplatesIntro"><label>' .
					TEXT_MAPPING_TEMPLATES_FOR_SUBCATEGORIES_PRODUCTS . '&nbsp;' . '</label>' .
					'</p>' . "\n";
			}
		}
		
		$content .= '<ul class="MainMappingTemplates">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li>';
			
			$content .= '<span class="MappingLanguageIcon">' . zen_image(DIR_WS_CATALOG_LANGUAGES .
				$languages[$i]['directory'] . '/images/' . $languages[$i]['image'],
				$languages[$i]['name']) . '</span>';
			
			$content .= $this->_addTooltipDescriptionsOfPlacementTags('product',
				$default_product_templates[$languages[$i]['id']]) . '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		// If this category has products in it then must link to listing to manage the mappings for
		// those products
		if ($num_products_in_category > 0) {
			$content .= '<h2 class="CreateManageCategoryHasProducts">' .
				sprintf(($num_products_in_category == 1 ? TEXT_PRODUCT_IN_CATEGORY :
				TEXT_PRODUCTS_IN_CATEGORY),
				htmlentities($main_category_name, ENT_COMPAT, CHARSET)) . '</h2>' . "\n";
			
			if (sizeof($subcategories) == 1 && $num_products_in_category == 1) {
				$content .= '<p>' . TEXT_SUBCATEGORY_PRODUCT_IN_CATEGORY_DESC . '</p>' . "\n";
				
			} else if (sizeof($subcategories) == 1 && $num_products_in_category > 1) {
				$content .= '<p>' . TEXT_SUBCATEGORY_PRODUCTS_IN_CATEGORY_DESC . '</p>' . "\n";
				
			} else if (sizeof($subcategories) > 1 && $num_products_in_category == 1) {
				$content .= '<p>' . TEXT_SUBCATEGORIES_PRODUCT_IN_CATEGORY_DESC . '</p>' . "\n";
				
			} else {
				$content .= '<p>' . TEXT_SUBCATEGORIES_PRODUCTS_IN_CATEGORY_DESC . '</p>' . "\n";
			}
			
			// Build link to product listing for products in this category
			$params = zen_get_all_get_params(array('listing'));
			
			$params .= 'listing=products&sub-cat-id=' . $this->_category_id .
				'&prods-not-in-subcat=1';
			
			$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			
			$content .= '<p><a href="' . $link . '">';
			
			if ($num_products_in_category == 1) {
				$content .= TEXT_MANAGE_PRODUCT_IN_CATEGORY;
			} else {
				$content .= TEXT_MANAGE_PRODUCTS_IN_CATEGORY;
			}
			
			$content .= '</a></p>' . "\n";
		}
		
		
		// Build the output for all of the subcategories of this category, including the control
		// panels to carry out actions for individual subcategories
		
		// Build the back link
		if ($this->_category_id != 0) {
			// Build a "back to parent category" link
			$params = zen_get_all_get_params(array('listing', 'cat-id'));
			
			$params .= 'listing=categories';
			
			$parent_category_info_result = $db->Execute("
				SELECT
					c.parent_id,
					cd.categories_name
				FROM
					" . TABLE_CATEGORIES . " c
				LEFT JOIN
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				ON
					cd.categories_id = c.parent_id
				AND
					cd.language_id = '" . (int) $_SESSION['languages_id'] . "'
				WHERE
					c.categories_id = '" . (int) $this->_category_id . "';");
			
			if (!$parent_category_info_result->EOF) {
				$parent_category_id = $parent_category_info_result->fields['parent_id'];
				$parent_category_name = $parent_category_info_result->fields['categories_name'];
			}
			
			if ($parent_category_id != 0) {
				$params .= '&cat-id=' . $parent_category_id;
			} else {
				$parent_category_name = TEXT_ROOT_CATEGORY;
			}
			
			$back_up_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			
		} else {
			$params = zen_get_all_get_params(array('listing', 'cat-id'));
			
			$back_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		}
		
		if (isset($back_up_link)) {
			$content .= '<p class="BackLink TopLink">' . zen_image(DIR_WS_IMAGES . 'icons/' .
				'ceon-umm-back-to.png') . ' <a href="' . $back_up_link . '">' .
				sprintf(TEXT_BACK_TO, htmlentities($parent_category_name, ENT_COMPAT, CHARSET)) .
				'</a></p>' . "\n";
		} else {
			$content .= '<p class="BackToSection TopLink">' . zen_image(DIR_WS_IMAGES . 'icons/' .
				'ceon-umm-back-to.png') . ' <a href="' . $back_link . '">' .
				TEXT_BACK_TO_CREATE_MANAGE_URIS . '</a></p>' . "\n";
		}
		
		
		$select_all_text = addslashes(TEXT_TICK_TO_SELECT_ALL_CATEGORIES);
		$deselect_all_text = addslashes(TEXT_TICK_TO_DESELECT_ALL_CATEGORIES);
		
		$content .= <<<TABLE_JS
<script type="text/javascript">
<!--
function RowHighlighted(el)
{
	if (el.className == 'EvenRow') {
		el.className = 'EvenRowOver';
	} else if (el.className == 'OddRow') {
		el.className = 'OddRowOver';
	}
}

function RowNotHighlighted(el)
{
	if (el.className == 'EvenRowOver') {
		el.className = 'EvenRow';
	} else if (el.className == 'OddRowOver') {
		el.className = 'OddRow';
	}
}

function SelectAllCategories(toggle_el, el_id)
{
	select_or_deselect = toggle_el.checked
	
	form_el = document.getElementById('category-listing-form');
	
	for (var t, i = 0; t = form_el.elements[el_id][i++]; t.checked=select_or_deselect);
	
	if (select_or_deselect) {
		toggle_el.title = '$deselect_all_text';
	} else {
		toggle_el.title = '$select_all_text';
	}
}

function stopPropagation(e)
{
	if (window.event) {
		window.event.cancelBubble = true;
	} else {
		e.stopPropagation();
	}
}
// -->
</script>
TABLE_JS;
		
		$content .=
			'<table cellpadding="0" cellspacing="0" id="category-listing" class="Listing">' . "\n";
		
		// Build header row
		$content .= '<tr>' . "\n";
		
		$content .= '<th id="checkboxes-col">' . '<input type="checkbox" name="select-all"' .
			' value="1" onclick="javascript:SelectAllCategories(this, \'cat[]\');"' .
			($this->_select_all ? ' checked="checked"' : '') . ' title="' .
			TEXT_TICK_TO_DESELECT_ALL_CATEGORIES . '" />' . '</th>' . "\n";
		
		$content .= '<th id="id-col">' . TEXT_ID . '</th>' . "\n";
		
		$content .= '<th>' . ($this->_category_id == 0 ? TEXT_CATEGORY : TEXT_SUBCATEGORY) .
			'</th>' . "\n";
		
		$content .= '<th>' . TEXT_ACTIONS . '</th>' . "\n";
		
		$content .= '</tr>' . "\n";
		
		
		// Build subcategory rows
		$num_subcategories = sizeof($subcategories);
		
		for ($sub_cat_i = 0; $sub_cat_i < $num_subcategories; $sub_cat_i++) {
			$sub_cat_id = $subcategories[$sub_cat_i]['id'];
			
			// Build the link for this category to display it
			if ($subcategories[$sub_cat_i]['has_subcategories']) {
				$params = zen_get_all_get_params(array('cat-id'));
				
				$params .= 'cat-id=' . $sub_cat_id;
				
				$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			} else {
				$params = zen_get_all_get_params(array('listing'));
				
				$params .= 'listing=products&sub-cat-id=' . $sub_cat_id;
				
				$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			}
			
			$content .= '<tr class="' . (($sub_cat_i % 2 == 0) ? 'EvenRow' : 'OddRow') . '"' .
				' onmouseover="RowHighlighted(this);" onmouseout="RowNotHighlighted(this)"' .
				' onclick="document.location.href=\'' . $link . '\'">' . "\n";
			
			$content .= '<td onclick="stopPropagation(event)">' . "\n";
			
			$content .= '<input type="checkbox" name="cat[]" value="' . $sub_cat_id . '"' .
				(in_array($sub_cat_id, $selected_cats) ? ' checked="checked"' : '') . ' title="' .
				TEXT_TICK_TO_SELECT_CATEGORY . '" />' . "\n";
			
			$content .= '</td>' . "\n";
			
			
			$content .= '<td>' . "\n";
			
			$content .= $sub_cat_id . "\n";
			
			$content .= '</td>' . "\n";
			
			
			// Build name and current URI mapping(s) and specific mapping template(s) for this
			// subcategory ////////////
			$content .= '<td>' . "\n";
			
			$disabled = ($subcategories[$sub_cat_i]['is_disabled'] ?
				' <span title="' . TEXT_CATEGORY_IS_DISABLED . '" style="cursor: help;" >*' .
				'</span>' : '');
			
			$content .= '<h3>' . '<a href="' . $link . '">' .
				htmlentities($subcategories[$sub_cat_i]['name'], ENT_COMPAT, CHARSET) .
				$disabled . '</a></h3>' . "\n";
			
			
			// Build the list of current mappings for this subcategory
			$content .= '<ul class="CategoryMappings">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] .
					'/images/' . $languages[$i]['image'], $languages[$i]['name']) .
					'</span>';
				
				if (isset($subcategories_mappings[$sub_cat_id][$languages[$i]['id']])) {
					$content .= '<a href="' . HTTP_SERVER .
						$subcategories_mappings[$sub_cat_id][$languages[$i]['id']] .
						'" target="_blank">' .
						$subcategories_mappings[$sub_cat_id][$languages[$i]['id']] . '</a>' . "\n";
				} else {
					$content .= '<span class="NoCurrentMapping">' . TEXT_NO_CURRENT_MAPPING .
						'</span>';
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			
			// Build the list of mapping templates for this subcategory
			if (isset($subcategories_subcategory_templates[$sub_cat_id])) {
				if ($num_languages == 1 ||
						sizeof($subcategories_subcategory_templates[$sub_cat_id]) == 1) {
					$content .= '<p class="CategoryMappingTemplatesIntro"><label>' .
						TEXT_MAPPING_TEMPLATE_FOR_SUBCATEGORIES . '&nbsp;' . '</label>' .
						'</p>' . "\n";
				} else {
					$content .= '<p class="CategoryMappingTemplatesIntro"><label>' .
						TEXT_MAPPING_TEMPLATES_FOR_SUBCATEGORIES . '&nbsp;' . '</label>' .
						'</p>' . "\n";
				}
				
				$content .= '<ul class="CategoryMappingTemplates">' . "\n";
				
				for ($i = 0; $i < $num_languages; $i++) {
					if (isset($subcategories_subcategory_templates[$sub_cat_id]
							[$languages[$i]['id']])) {
						$content .= '<li>';
						
						$content .= '<span class="MappingLanguageIcon">' .
							zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] .
							'/images/' . $languages[$i]['image'], $languages[$i]['name']) .
							'</span>';
						
						$content .= $this->_addTooltipDescriptionsOfPlacementTags('category',
							$subcategories_subcategory_templates[$sub_cat_id]
							[$languages[$i]['id']]) . '</li>' . "\n";
					}
				}
				
				$content .= '</ul>' . "\n";
			}
			
			if (isset($subcategories_product_templates[$sub_cat_id])) {
				if ($subcategories[$sub_cat_i]['has_subcategories']) {
					if ($num_languages == 1 ||
							sizeof($subcategories_product_templates[$sub_cat_id]) == 1) {
						$content .= '<p class="CategoryMappingTemplatesIntro"><label>' .
							TEXT_MAPPING_TEMPLATE_FOR_SUBCATEGORIES_PRODUCTS . '&nbsp;' .
							'</label>' . '</p>' . "\n";
					} else {
						$content .= '<p class="CategoryMappingTemplatesIntro"><label>' .
							TEXT_MAPPING_TEMPLATES_FOR_SUBCATEGORIES_PRODUCTS . '&nbsp;' .
							'</label>' . '</p>' . "\n";
					}
				} else {
					if ($num_languages == 1 ||
							sizeof($subcategories_product_templates[$sub_cat_id]) == 1) {
						$content .= '<p class="CategoryMappingTemplatesIntro"><label>' .
							TEXT_MAPPING_TEMPLATE_FOR_PRODUCTS . '&nbsp;' . '</label>' .
							'</p>' . "\n";
					} else {
						$content .= '<p class="CategoryMappingTemplatesIntro"><label>' .
							TEXT_MAPPING_TEMPLATES_FOR_PRODUCTS . '&nbsp;' . '</label>' .
							'</p>' . "\n";
					}
				}
				
				$content .= '<ul class="CategoryMappingTemplates">' . "\n";
				
				for ($i = 0; $i < $num_languages; $i++) {
					if (isset($subcategories_product_templates[$sub_cat_id]
							[$languages[$i]['id']])) {
						$content .= '<li>';
						
						$content .= '<span class="MappingLanguageIcon">' .
							zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] .
							'/images/' . $languages[$i]['image'], $languages[$i]['name']) .
							'</span>';
						
						$content .= $this->_addTooltipDescriptionsOfPlacementTags('product',
							$subcategories_product_templates[$sub_cat_id]
							[$languages[$i]['id']]) . '</li>' . "\n";
					}
				}
				
				$content .= '</ul>' . "\n";
			}
			
			
			$content .= '</td>' . "\n";
			
			// Build controls for this subcategory //////////
			$content .= '<td class="ListingItemControls">' . "\n";
			
			$params = zen_get_all_get_params();
			
			$params .= 'manage-mapping-templates=' . $sub_cat_id;
			
			$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			
			if ($num_languages == 1) {
				$button_text = TEXT_CATEGORY_MANAGE_MAPPING_TEMPLATE;
				
				$button_desc = TEXT_CATEGORY_MANAGE_MAPPING_TEMPLATE_HOVER_INFO;
			} else {
				$button_text = TEXT_CATEGORY_MANAGE_MAPPING_TEMPLATES;
				
				$button_desc = TEXT_CATEGORY_MANAGE_MAPPING_TEMPLATES_HOVER_INFO;
			}
			
			$content .= '<p class="ButtonLink ManageTemplatesButton"><a href="' . $link .
				'" title="' . $button_desc . '">' . $button_text . '</a></p>' . "\n";
			
			
			$params = zen_get_all_get_params();
			
			$params .= 'manage-mappings=' . $sub_cat_id;
			
			$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			
			if ($num_languages == 1) {
				$button_text = TEXT_CATEGORY_MANAGE_MAPPING;
				
				$button_desc = TEXT_CATEGORY_MANAGE_MAPPING_HOVER_INFO;
			} else {
				$button_text = TEXT_CATEGORY_MANAGE_MAPPINGS;
				
				$button_desc = TEXT_CATEGORY_MANAGE_MAPPINGS_HOVER_INFO;
			}
			
			$content .= '<p class="ButtonLink ManageMappingsButton"><a href="' . $link .
				'" title="' . $button_desc . '">' . $button_text . '</a></p>' . "\n";
			
			
			$content .= '</td>' . "\n";
			
			$content .= '</tr>' . "\n";
		}
		
		$content .= '</table>' . "\n";
		
		
		// Build the back to top link
		$content .= '<p class="BackToTopNoSaveButton"><a href="#top">' . TEXT_BACK_TO_TOP .
			zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to-top.png') . '</a></p>' . "\n";
		
		// Output the appropriate back link
		if (isset($back_up_link)) {
			$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/' .
				'ceon-umm-back-to.png') . ' <a href="' . $back_up_link . '">' .
				sprintf(TEXT_BACK_TO, htmlentities($parent_category_name, ENT_COMPAT, CHARSET)) .
				'</a></p>' . "\n";
		} else {
			$content .= '<p class="BackToSection">' . zen_image(DIR_WS_IMAGES . 'icons/' .
				'ceon-umm-back-to.png') . ' <a href="' . $back_link . '">' .
				TEXT_BACK_TO_CREATE_MANAGE_URIS . '</a></p>' . "\n";
		}
		
		
		$content .= '</form>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ buildProductListing()
	
	/**
	 * Builds a list of all the products for the current category, information about any
	 * mappings, and gadgets to enable actions to be carried out against products in the category.
	 *
	 * @access  public
	 * @return  none
	 */
	public function buildProductListing()
	{
		global $db, $languages, $num_languages, $ceon_uri_mapping_demo, $messageStack;
		
		$content = '';
		
		$product_templates = array();
		
		// Variables hold information about the current category
		$this->_category_id = (int) $_GET['sub-cat-id'];
		
		$category_name = zen_get_category_name($this->_category_id, $_SESSION['languages_id']);
		
		// Progress back through the list of parent categories for the category being displayed,
		// getting the most uniquely specified templates
		$current_category_id = $this->_category_id;
		
		do {
			// Get any product mapping template(s) for this category
			$columns_to_retrieve = array(
				'mapping_template',
				'language_id'
				);
			
			$selections = array(
				'category_id' => $current_category_id
				);
			
			$products_mapping_templates_result = $this->getURIMappingTemplatesResultset('product',
				$columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$products_mapping_templates_result->EOF) {
				$language_id = $products_mapping_templates_result->fields['language_id'];
				
				if (!isset($product_templates[$language_id])) {
					// A specific default template has not been set for this language
					$product_templates[$language_id] =
						$products_mapping_templates_result->fields['mapping_template'];
				}
				
				$products_mapping_templates_result->MoveNext();
			}
			
			if (sizeof($product_templates) == $num_languages) {
				// Have got a template for each language, job's done!
				break;
			}
			
			if ($current_category_id == 0) {
				// Can't move any further up, at the top!
				break;
			}
			
			$parent_category_id_result = $db->Execute("
				SELECT
					parent_id
				FROM
					" . TABLE_CATEGORIES . "
				WHERE
					categories_id = '" . (int) $current_category_id . "';");
			
			if ($parent_category_id_result->EOF) {
				// Should never happen
				break;
			} else {
				$current_category_id = $parent_category_id_result->fields['parent_id'];
			}
		} while (1);
		
		
		// Get the information for the products for this category
		$products = array();
		$products_mappings = array();
		
		$products_result = $db->Execute("
			SELECT
				p.products_id,
				pd.products_name,
				p.products_model,
				pt.type_handler,
				p.products_status
            FROM
				" . TABLE_PRODUCTS . " p
			LEFT JOIN
				" . TABLE_PRODUCTS_DESCRIPTION . " pd
            ON
				p.products_id = pd.products_id
            AND
				pd.language_id = '" . (int) $_SESSION['languages_id'] . "'
			LEFT JOIN
				" . TABLE_PRODUCT_TYPES . " pt
            ON
				p.products_type = pt.type_id
            WHERE
				p.master_categories_id = '" . $this->_category_id . "'
            ORDER BY
				p.products_sort_order, pd.products_name");
		
		while (!$products_result->EOF) {
			$products[] = array(
				'id' => $products_result->fields['products_id'],
				'name' => $products_result->fields['products_name'],
				'model' => $products_result->fields['products_model'],
				'type_handler' => $products_result->fields['type_handler'],
				'is_disabled' => ($products_result->fields['products_status'] == 0 ? true : false)
				);
			
			$products_result->MoveNext();
		}
		
		
		// Get the current URI mapping(s) for the products, if any
		$columns_to_retrieve = array(
			'uri',
			'language_id'
			);
		
		$selections = array(
			'current_uri' => 1
			);
		
		foreach ($products as $product) {
			$products_mappings[$product['id']] = array();
			
			$selections['main_page'] = $product['type_handler'] . '_info';
			
			$selections['associated_db_id'] = $product['id'];
			
			$product_mapping_result =
				$this->getURIMappingsResultset($columns_to_retrieve, $selections);
			
			while (!$product_mapping_result->EOF) {
				$products_mappings[$product['id']]
					[$product_mapping_result->fields['language_id']] =
					$product_mapping_result->fields['uri'];
				
				$product_mapping_result->MoveNext();
			}
		}
		
		// Default to excluding disabled products
		$autogenerate_inc_disabled = false;
		
		// Default to all being selected
		$this->_select_all = true;
		
		$selected_products = array();
		
		foreach ($products as $product) {
			$selected_products[] = $product['id'];
		}
		
		// All data has been gathered, build the page //////////////////////////////////////////////
		
		// Build the title with the current category's details
		$panel_title = TEXT_PRODUCT_MAPPINGS_TITLE;
		
		$content .= '<h1>' . $panel_title . '</h1>' . "\n";
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action .
			'" method="post" name="product-listing-form" id="product-listing-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Add the autogenerate controls for this category /////////////////////////////////////////
		$content .= '<div class="AutogenerateControlsWrapper" id="autogenerate-controls-top">' .
			"\n";
		
		$content .= '<fieldset class="AutogenerateControls"><legend>' .
			TEXT_AUTOGENERATE_MAPPINGS . '</legend>' . "\n";
		
		$content .= '<p class="AutogenerationOptions">' . "\n";
		
		$content .= '<input type="radio" name="autogenerate-product-mappings"' .
			' value="exc-products" id="autogenerate-product-mappings"' .
			(1 == 1 ? ' checked="checked"' : '') .
			' />' . "\n"; 
		
		$content .= '<label for="autogenerate-product-mappings">' .
			TEXT_SELECTED_PRODUCTS . '</label>' . "\n";
		
		$content .= '<br /><input type="checkbox" name="inc-disabled" value="1"' .
			' id="inc-disabled"' .
			($autogenerate_inc_disabled == true ? ' checked="checked"' : '') .
			' />' . "\n"; 
		
		$content .= '<label for="autogenerate-category-mappings-exc-products">' .
			TEXT_INC_DISABLED_PRODUCTS . '</label>' . '</p>' . "\n";
		
		// Main button to generate URI mappings for all selected products
		$button_title = TEXT_AUTOGENERATE_URI_MAPPINGS_BUTTON;
		$button_desc = TEXT_AUTOGENERATE_PRODUCTS_URI_MAPPINGS_BUTTON_HOVER_INFO;
		
		$content .= '<input type="submit" class="AutogenerateURIMappingsButton"' .
			' name="auto-gen-products" id="auto-gen-products"' .
			' value="' . $button_title . '" title="' . $button_desc . '" />' . "\n"; 
		
		$content .= '</fieldset>' . "\n";
		
		$content .= '</div>' . "\n";
		
		
		// Build the info for the current category /////////////////////////////////////////////////
		$content .= '<h2 class="CreateManageCategoryTitle">' . sprintf(TEXT_PRODUCTS_IN_CATEGORY2,
			htmlentities($category_name, ENT_COMPAT, CHARSET)) . '</h2>' . "\n";
		
		// Build the list of products mapping templates for the main category
		if ($num_languages == 1) {
			$content .= '<p class="MainMappingTemplatesIntro"><label>' .
				TEXT_MAPPING_TEMPLATE_FOR_PRODUCTS . '&nbsp;' . '</label>' . '</p>' . "\n";
		} else {
			$content .= '<p class="MainMappingTemplatesIntro"><label>' .
				TEXT_MAPPING_TEMPLATES_FOR_PRODUCTS . '&nbsp;' . '</label>' . '</p>' . "\n";
		}
		
		$content .= '<ul class="MainMappingTemplates">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li>';
			
			$content .= '<span class="MappingLanguageIcon">' . zen_image(DIR_WS_CATALOG_LANGUAGES .
				$languages[$i]['directory'] . '/images/' . $languages[$i]['image'],
				$languages[$i]['name']) . '</span>';
			
			$content .= $this->_addTooltipDescriptionsOfPlacementTags('product',
				$product_templates[$languages[$i]['id']]) . '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		// Build the output for all of the products in this category, including the control panels
		// to carry out actions for individual products
		
		// Build the "back to category" link
		$params = zen_get_all_get_params(array('listing', 'sub-cat-id', 'prods-not-in-subcat'));
		
		$params .= 'listing=categories';
		
		// Don't look up parent category if products are in the category itself and not a
		// subcategory
		if (isset($_GET['prods-not-in-subcat'])) {
			$parent_category_name = $category_name;
		} else {
			$parent_category_info_result = $db->Execute("
				SELECT
					c.parent_id,
					cd.categories_name
				FROM
					" . TABLE_CATEGORIES . " c
				LEFT JOIN
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				ON
					cd.categories_id = c.parent_id
				AND
					cd.language_id = '" . (int) $_SESSION['languages_id'] . "'
				WHERE
					c.categories_id = '" . (int) $this->_category_id . "';");
			
			if (!$parent_category_info_result->EOF) {
				$parent_category_id = $parent_category_info_result->fields['parent_id'];
				$parent_category_name = $parent_category_info_result->fields['categories_name'];
			}
			
			if ($parent_category_id == 0) {
				$parent_category_name = TEXT_ROOT_CATEGORY;
			}
		}
		
		$back_up_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p class="BackLink TopLink">' . zen_image(DIR_WS_IMAGES . 'icons/' .
			'ceon-umm-back-to.png') . ' <a href="' . $back_up_link . '">' .
			sprintf(TEXT_BACK_TO, htmlentities($parent_category_name, ENT_COMPAT, CHARSET)) .
			'</a></p>' . "\n";
		
		
		$select_all_text = addslashes(TEXT_TICK_TO_SELECT_ALL_PRODUCTS);
		$deselect_all_text = addslashes(TEXT_TICK_TO_DESELECT_ALL_PRODUCTS);
		
		$content .= <<<TABLE_JS
<script type="text/javascript">
<!--
function SelectAllProducts(toggle_el, el_id)
{
	select_or_deselect = toggle_el.checked
	
	form_el = document.getElementById('product-listing-form');
	
	for (var t, i = 0; t = form_el.elements[el_id][i++]; t.checked=select_or_deselect);
	
	if (select_or_deselect) {
		toggle_el.title = '$deselect_all_text';
	} else {
		toggle_el.title = '$select_all_text';
	}
}
// -->
</script>
TABLE_JS;
		
		$content .= '<table cellpadding="0" cellspacing="0" id="product-listing" class="Listing">' .
			"\n";
		
		// Build header row
		$content .= '<tr>' . "\n";
		
		$content .= '<th id="checkboxes-col">' .
			'<input type="checkbox" name="select-all" value="1"' .
			' onclick="javascript:SelectAllProducts(this, \'product[]\');"' .
			($this->_select_all ? ' checked="checked"' : '') . ' title="' .
			TEXT_TICK_TO_DESELECT_ALL_PRODUCTS . '" />' . '</th>' . "\n";
		
		$content .= '<th id="id-col">' . TEXT_ID . '</th>' . "\n";
		
		$content .= '<th>' . TEXT_PRODUCT . '</th>' . "\n";
		
		$content .= '<th>' . TEXT_ACTIONS . '</th>' . "\n";
		
		$content .= '</tr>' . "\n";
		
		
		// Build product rows
		$num_products = sizeof($products);
		
		for ($product_i = 0; $product_i < $num_products; $product_i++) {
			$product_id = $products[$product_i]['id'];
			
			$content .= '<tr class="' . (($product_i % 2 == 0) ? 'EvenRow' : 'OddRow') . '">' .
				"\n";
			
			$content .= '<td>' . "\n";
			
			$content .= '<input type="checkbox" name="product[]" value="' . $product_id . '"' .
				(in_array($product_id, $selected_products) ? ' checked="checked"' : '') .
				' title="' . TEXT_TICK_TO_SELECT_PRODUCT . '" />' . "\n";
			
			$content .= '</td>' . "\n";
			
			
			$content .= '<td>' . "\n";
			
			$content .= $product_id . "\n";
			
			$content .= '</td>' . "\n";
			
			
			// Build name and current URI mapping(s) and specific mapping template(s) for this
			// product ////////////
			$content .= '<td>' . "\n";
			
			$disabled = ($products[$product_i]['is_disabled'] ?
				' <span title="' . TEXT_PRODUCT_IS_DISABLED . '" style="cursor: help"; >*' .
				'</span>' : '');
			
			$content .= '<h3>' . htmlentities($products[$product_i]['name'], ENT_COMPAT, CHARSET) .
				$disabled . '</h3>' . "\n";
			
			
			// Build the list of current mappings for this product
			$content .= '<ul class="ProductMappings">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				if (isset($products_mappings[$product_id][$languages[$i]['id']])) {
					$content .= '<a href="' . HTTP_SERVER .
						$products_mappings[$product_id][$languages[$i]['id']] .
						'" target="_blank">' .
						$products_mappings[$product_id][$languages[$i]['id']] . '</a>' . "\n";
				} else {
					$content .= '<span class="NoCurrentMapping">' . TEXT_NO_CURRENT_MAPPING .
						'</span>';
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			$content .= '</td>' . "\n";
			
			// Build controls for this product //////////
			$content .= '<td class="ListingItemControls">' . "\n";
			
			$params = zen_get_all_get_params();
			
			$params .= 'manage-mappings=' . $product_id;
			
			$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
			
			if ($num_languages == 1) {
				$button_text = TEXT_PRODUCT_MANAGE_MAPPING;
				
				$button_desc = TEXT_PRODUCT_MANAGE_MAPPING_HOVER_INFO;
			} else {
				$button_text = TEXT_PRODUCT_MANAGE_MAPPINGS;
				
				$button_desc = TEXT_PRODUCT_MANAGE_MAPPINGS_HOVER_INFO;
			}
			
			$content .= '<p class="ButtonLink ManageMappingsButton"><a href="' . $link .
				'" title="' . $button_desc . '">' . $button_text . '</a></p>' . "\n";
			
			
			$content .= '</td>' . "\n";
			
			$content .= '</tr>' . "\n";
		}
		
		$content .= '</table>' . "\n";
		
		
		// Build the back to top link
		$content .= '<p class="BackToTopNoSaveButton"><a href="#top">' . TEXT_BACK_TO_TOP .
			zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to-top.png') . '</a></p>' . "\n";
		
		// Build the "back to category" link
		$params = zen_get_all_get_params(array('listing', 'sub-cat-id'));
		
		$params .= 'listing=categories';
		
		$back_up_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/' .
			'ceon-umm-back-to.png') . ' <a href="' . $back_up_link . '">' .
			sprintf(TEXT_BACK_TO, htmlentities($parent_category_name, ENT_COMPAT, CHARSET)) .
			'</a></p>' . "\n";
		
		
		$content .= '</form>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
}

// }}}
