<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate Object Mappings Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageObjMappings.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Action class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMAction.php');


// {{{ CeonUMMCreateManageObjMappings

/**
 * Allows the user to manually set/change/delete the URI mappings for the specified object (page).
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageObjMappings extends CeonUMMAction
{
	// {{{ properties
	
	/**
	 * The type of object being managed.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_object_type = null;
	
	/**
	 * The ID of a category being managed or of the category containing a product being managed.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_category_id = null;
	
	/**
	 * The ID of a product being managed.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_product_id = null;
	
	/**
	 * The ID of the parent category of a category being managed.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_parent_category_id = null;
	
	/**
	 * The previous mappings for the object, or list of objects, currently being managed. Individual mappings are
	 * indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_prev_mappings = null;
	
	/**
	 * The current mappings for the object, or list of objects, currently being managed. Individual mappings are
	 * indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_mappings = array();
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 * @param   string     $object_type   The type of object to have its URIs managed.
	 * @param   integer    $object_id     The database ID of the object.
	 */
	public function __construct($object_type, $object_id)
	{
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' . 'ceon_umm_create_manage_obj_mappings.php');
		
		if (!defined('TEXT_MANAGE_MAPPING_TITLE') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' . 'ceon_umm_create_manage_obj_mappings.php');
		}
		
		// Set up the main environment control variables
		$this->_object_type = $object_type;
		
		if ($this->_object_type == 'category') {
			// Build the form for a category's mappings
			$this->_category_id = $object_id;
			
			$this->_parent_category_id = (int) $_GET['cat-id'];
			
		} else if ($this->_object_type == 'product') {
			// Build the form for a product's mappings
			$this->_product_id = $object_id;
			
			$this->_category_id = (int) $_GET['sub-cat-id'];
		}
		
		parent::__construct();
		
		// Run the actual functionality and build the output
		$this->_buildManageObjectMappingsPanel();
	}
	
	// }}}
	
	
	// {{{ _buildManageObjectMappingsPanel()
	
	/**
	 * Builds a subpanel which lets the user see and edit the mappings for the specified object.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildManageObjectMappingsPanel()
	{
		global $db, $languages, $num_languages;
		
		$this->_prev_mappings = array();
		$this->_mappings = array();
		
		if (isset($this->_posted_data['mappings-submit'])) {
			// Form has been submitted
			$this->_processMappingsFormSubmission();
		} else {
			// Set up defaults
			if ($this->_object_type == 'category') {
				$columns_to_retrieve = array(
					'uri',
					'language_id'
					);
				
				$selections = array(
					'main_page' => FILENAME_DEFAULT,
					'associated_db_id' => $this->_category_id,
					'current_uri' => 1
					);
				
				$mappings_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections);
				
				// Store any mappings for any language
				while (!$mappings_result->EOF) {
					$language_id = $mappings_result->fields['language_id'];
					
					$this->_prev_mappings[$language_id] = $mappings_result->fields['uri'];
					
					$mappings_result->MoveNext();
				}
			} else if ($this->_object_type == 'product') {
				// Get the page type for this product
				$page_handler = zen_get_handler_from_type(zen_get_products_type($this->_product_id));
				
				$columns_to_retrieve = array(
					'uri',
					'language_id'
					);
				
				$selections = array(
					'main_page' => $page_handler . '_info',
					'associated_db_id' => $this->_product_id,
					'current_uri' => 1
					);
				
				$mappings_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections);
				
				// Store any mappings for any language
				while (!$mappings_result->EOF) {
					$language_id = $mappings_result->fields['language_id'];
					
					$this->_prev_mappings[$language_id] = $mappings_result->fields['uri'];
					
					$mappings_result->MoveNext();
				}
			}
			
			// Must have defaults for each mapping
			for ($i = 0; $i < $num_languages; $i++) {
				$language_id = $languages[$i]['id'];
				
				if (!isset($this->_prev_mappings[$language_id])) {
					$this->_prev_mappings[$language_id] = '';
				}
				
				$this->_mappings[$language_id] = $this->_prev_mappings[$language_id];
			}
		}
		
		$num_previous_mappings = sizeof($this->_prev_mappings);
		$num_mappings = sizeof($this->_mappings);
		
		// Build the output for this method ////
		$content = '';
		
		if ($this->_object_type == 'category') {
			$category_name = zen_get_category_name($this->_category_id, $_SESSION['languages_id']);
			
			$object_name = $category_name;
			
		} else if ($this->_object_type == 'product') {
			$product_name = zen_get_products_name($this->_product_id, $_SESSION['languages_id']);
			
			$object_name = $product_name;
		}
		
		if ($num_languages == 1) {
			$title = sprintf(TEXT_MANAGE_MAPPING_TITLE, htmlentities($object_name, ENT_COMPAT, CHARSET));
		} else {
			$title = sprintf(TEXT_MANAGE_MAPPINGS_TITLE, htmlentities($object_name, ENT_COMPAT, CHARSET));
		}
		
		
		$content .= '<fieldset id="manage-object-mappings" class="ContainingSubPanel">' . '<legend>' . $title .
			'</legend>' . "\n";
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action . '" method="post"  name="mappings-form" id="mappings-form">' .
			"\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		if ($this->_object_type == 'category') {
			if ($num_languages == 1) {
				$intro =
					sprintf(TEXT_MANAGE_MAPPING_INTRO_CATEGORY, htmlentities($category_name, ENT_COMPAT, CHARSET));
			} else {
				$intro =
					sprintf(TEXT_MANAGE_MAPPINGS_INTRO_CATEGORY, htmlentities($category_name, ENT_COMPAT, CHARSET));
			}
		} else if ($this->_object_type == 'product') {
			if ($num_languages == 1) {
				$intro =
					sprintf(TEXT_MANAGE_MAPPING_INTRO_PRODUCT, htmlentities($product_name, ENT_COMPAT, CHARSET));
			} else {
				$intro =
					sprintf(TEXT_MANAGE_MAPPINGS_INTRO_PRODUCT, htmlentities($product_name, ENT_COMPAT, CHARSET));
			}
		}
		
		if ($num_languages == 1) {
			if ($num_previous_mappings == 1) {
				$intro2 = TEXT_THE_CURRENT_MAPPING_CHANGED_BECOMES_HISTORICAL;
				
				$intro3 = TEXT_MAKE_THE_CURRENT_MAPPING_HISTORICAL_BY_ENTERING_NOTHING;
			}
		} else {
			if ($num_previous_mappings == 1) {
				$intro2 = TEXT_A_CURRENT_MAPPING_CHANGED_BECOMES_HISTORICAL;
				
				$intro3 = TEXT_MAKE_A_CURRENT_MAPPING_HISTORICAL_BY_ENTERING_NOTHING;
				
			} else {
				$intro2 = TEXT_A_CURRENT_MAPPING_CHANGED_BECOMES_HISTORICAL;
				
				$intro3 = TEXT_MAKE_THE_CURRENT_MAPPINGS_HISTORICAL_BY_ENTERING_NOTHING;
			}
		}
		
		$content .= '<p>' . $intro . '</p>' . "\n";
		
		if (isset($intro2)) {
			$content .= '<p>' . $intro2 . '</p>' . "\n";
			
			$content .= '<p>' . $intro3 . '</p>' . "\n";
		}
		
		// Build the field(s) for the mapping(s)
		$content .= '<p><label class="Mappings">' . "\n";
		
		if ($this->_object_type == 'category') {
			if ($num_languages == 1) {
				$content .= TEXT_LABEL_CATEGORY_MAPPING;
			} else {
				$content .= TEXT_LABEL_CATEGORY_MAPPINGS;
			}
		} else if ($this->_object_type == 'product') {
			if ($num_languages == 1) {
				$content .= TEXT_LABEL_PRODUCT_MAPPING;
			} else {
				$content .= TEXT_LABEL_PRODUCT_MAPPINGS;
			}
		}
		
		$content .= '</label></p>' . "\n";
		
		$content .= '<ul class="MappingsFields">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li><span class="MappingLanguageIcon">' .
				zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
			
			$content .= ' <input type="text" name="mappings[' . $languages[$i]['id'] . ']" value="' .
				$this->_mappings[$languages[$i]['id']] . '" maxlength="255" class="Textfield" />';
			
			if (isset($this->_error_messages['mappings']) &&
					isset($this->_error_messages['mappings'][$languages[$i]['id']])) {
				foreach ($this->_error_messages['mappings'][$languages[$i]['id']] as $error_message) {
					$content .= '<p class="MappingError">' . $error_message . '</p>' . "\n";
				}
			}
			
			$content .= '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		if ($num_languages == 1) {
			$button_text = TEXT_UPDATE_MAPPING;
		} else {
			$button_text = TEXT_UPDATE_MAPPINGS;
		}
		
		$content .= '<input type="submit" class="MappingsSubmit" name="mappings-submit" id="mappings-submit"' .
			' value="' . $button_text . '" />' . "\n"; 
		
		// Store the previous mappings temporarily so they can be compared to see if any have changed
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<input type="hidden" name="prev-mappings[' . $languages[$i]['id'] . ']" value="' .
				$this->_prev_mappings[$languages[$i]['id']] . '" />' . "\n"; 
		}
		
		$content .= '</form>' . "\n";
		
		
		$content .= '</fieldset>' . "\n";
		
		
		// Build the back link
		$params = zen_get_all_get_params(array('manage-mappings'));
		
		$back_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		if ($this->_object_type == 'category') {
			// Get the name of the category to go back to
			if ($this->_parent_category_id == 0) {
				$category_name = TEXT_ROOT_CATEGORY;
			} else {
				$category_info_result = $db->Execute("
					SELECT
						cd.categories_name
					FROM
						" . TABLE_CATEGORIES_DESCRIPTION . " cd
					WHERE
						cd.categories_id = '" . $this->_parent_category_id . "'
					AND
						cd.language_id = '" . (int) $_SESSION['languages_id'] . "';");
				
				if (!$category_info_result->EOF) {
					$category_name = $category_info_result->fields['categories_name'];
				}
			}
			
			$back_to = $category_name;
			
		} else if ($this->_object_type == 'product') {
			$category_info_result = $db->Execute("
				SELECT
					cd.categories_name
				FROM
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				WHERE
					cd.categories_id = '" . $this->_category_id . "'
				AND
					cd.language_id = '" . (int) $_SESSION['languages_id'] . "';");
			
			if (!$category_info_result->EOF) {
				$category_name = $category_info_result->fields['categories_name'];
			}
			
			$back_to = $category_name;
		}
		
		$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/ceon-umm-back-to.png') .
			' <a href="' . $back_link . '">' . sprintf(TEXT_BACK_TO, htmlentities($back_to, ENT_COMPAT, CHARSET)) .
			'</a></p>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _processMappingsFormSubmission()
	
	/**
	 * Handle the submission of mapping(s), checking that they are valid and if no errors are encountered, a new
	 * mapping is added and any old mapping made historical. Any deleted mappings are also made historical.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processMappingsFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		// Get the information about the page type so it can be used in the clash check and to actually save any
		// mapping
		if ($this->_object_type == 'category') {
			$main_page = FILENAME_DEFAULT;
			
		} else if ($this->_object_type == 'product') {
			$page_handler = zen_get_handler_from_type(zen_get_products_type($this->_product_id));
			
			$main_page = $page_handler . '_info';
		}
		
		for ($i = 0; $i < $num_languages; $i++) {
			$this->_prev_mappings[$languages[$i]['id']] =
				$this->_posted_data['prev-mappings'][$languages[$i]['id']];
			
			$this->_mappings[$languages[$i]['id']] =
				$this->_cleanUpURIMapping($this->_posted_data['mappings'][$languages[$i]['id']]);
			
			$prev_mapping = $this->_prev_mappings[$languages[$i]['id']];
			
			$mapping = $this->_mappings[$languages[$i]['id']];
			
			if (strlen($mapping) > 0 && $mapping != $prev_mapping) {
				// New/updated mapping has been specified
				
				// Check that the mapping(s) just entered doesn't/don't clash with any existing mapping(s), so the
				// user can be notified
				$mapping_clashed = false;
				
				$columns_to_retrieve = array(
					'main_page',
					'associated_db_id',
					'query_string_parameters'
					);
				
				$selections = array(
					'uri' => zen_db_prepare_input($mapping),
					'current_uri' => 1,
					'language_id' => $languages[$i]['id']
					);
				
				$order_by = null;
				
				$existing_uri_mapping_result =
					$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
				
				// If the existing mapping is simply having some capitalisation changed then a case insensitive
				// comparison might result in a false positive for a mapping clash, so prevent that by checking the
				// mapping's settings don't match.
				// @NOTE Support for query string parameters value not implemented
				$associated_db_id = ($this->_object_type == 'category' ? $this->_category_id : $this->_product_id);
				
				if (!$existing_uri_mapping_result->EOF &&
						!($existing_uri_mapping_result->fields['main_page'] == $main_page &&
						$existing_uri_mapping_result->fields['associated_db_id'] == $associated_db_id &&
						is_null($existing_uri_mapping_result->fields['query_string_parameters']))) {
					// This mapping clashes with an existing mapping
					if (!isset($this->_error_messages['mappings'])) {
						$this->_error_messages['mappings'] = array();
					}
					
					if (!is_array($this->_error_messages['mappings'][$languages[$i]['id']])) {
						$this->_error_messages['mappings'][$languages[$i]['id']] = array();
					}
					
					$this->_error_messages['mappings'][$languages[$i]['id']][] = TEXT_ERROR_URI_ENTERED_CLASHES;
				}
			}
		}
		
		if (!isset($this->_error_messages['mappings'])) {
			// All mappings valid! (Attempt to) save all the mappings
			
			// Build the information for the additional product pages that must have URIs managed
			if ($this->_object_type == 'product') {
				if (!isset($this->_products_admin_instance) || is_null($this->_products_admin_instance)) {
					require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminProducts.php');
					
					$this->_products_admin_instance = new CeonURIMappingAdminProducts();
				}
				
				$page_types = array(
					'product_reviews',
					'product_reviews_info',
					'product_reviews_write',
					'tell_a_friend',
					'ask_a_question'
					);
				
				$page_types_to_manage = array();
				
				foreach ($page_types as $page_type) {
					if ($this->_products_admin_instance->autoManageProductRelatedPageURI($page_type)) {
						$page_types_to_manage[] = $page_type;
					}
				}
			}
			
			for ($i = 0; $i < $num_languages; $i++) {
				$prev_mapping = $this->_prev_mappings[$languages[$i]['id']];
				
				$mapping = $this->_mappings[$languages[$i]['id']];
				
				$insert_mapping = false;
				$update_mapping = false;
				
				if ($mapping != '') {
					// Check if the URI mapping is being updated or does not yet exist
					if ($prev_mapping == '') {
						$insert_mapping = true;
					} else if ($prev_mapping != $mapping) {
						$update_mapping = true;
					}
				}
				
				if ($insert_mapping || $update_mapping) {
					if ($update_mapping) {
						// Consign previous mapping to the history, so old URI mapping isn't broken
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					}
					
					// Add the new URI mapping
					$uri = $mapping;
					
					if ($this->_object_type == 'category') {
						$mapping_added = $this->addURIMapping($uri, $languages[$i]['id'], $main_page, null,
							$this->_category_id);
						
					} else if ($this->_object_type == 'product') {
						$mapping_added =
							$this->addURIMapping($uri, $languages[$i]['id'], $main_page, null, $this->_product_id);
					}
					
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
						// New mapping must be used as "previous" mapping now
						$this->_prev_mappings[$languages[$i]['id']] = $this->_mappings[$languages[$i]['id']];
						
						if ($this->_object_type == 'category') {
							if ($insert_mapping) {
								$success_message = sprintf(TEXT_CATEGORY_MAPPING_ADDED,
									ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
									'" target="_blank">' . $uri . '</a>');
							} else {
								$success_message = sprintf(TEXT_CATEGORY_MAPPING_UPDATED,
									ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
									'" target="_blank">' . $uri . '</a>');
							}
						} else if ($this->_object_type == 'product') {
							if ($insert_mapping) {
								$success_message = sprintf(TEXT_PRODUCT_MAPPING_ADDED,
									ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
									'" target="_blank">' . $uri . '</a>');
							} else {
								$success_message = sprintf(TEXT_PRODUCT_MAPPING_UPDATED,
									ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
									'" target="_blank">' . $uri . '</a>');
							}
						}
						
						$messageStack->add($success_message, 'success');
						
					} else {
						if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS,
								ucwords($languages[$i]['name'])) . '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>';
							
						} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DATA, ucwords($languages[$i]['name'])) . $uri;
						} else {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DB, ucwords($languages[$i]['name'])) . $uri;
						}
						
						$messageStack->add($failure_message, 'error');
					}
					
					if ($this->_object_type == 'product') {
						// Now add the URI mappings for the review pages/tell-a-friend page for this product
						$base_uri = $uri . '/';
						
						// Get the language code for the mapping's language
						$language_code = strtolower($languages[$i]['code']);
						
						foreach ($page_types_to_manage as $page_type) {
							// Mark any existing URI mapping for this product related page as no longer being the
							// "primary" mapping, leaving it in the database so old  links aren't broken.
							$columns_to_retrieve = array(
								'uri'
								);
							
							$selections = array(
								'main_page' => $page_type,
								'associated_db_id' => (int) $this->_product_id,
								'language_id' => (int) $languages[$i]['id'],
								'current_uri' => 1,
								);
							
							$order_by = null;
							
							$current_uri_mapping_result =
								$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
							
							if (!$current_uri_mapping_result->EOF) {
								$this->makeURIMappingHistorical($current_uri_mapping_result->fields['uri'],
									$languages[$i]['id']);
							}
							
							$uri_part = $this->_products_admin_instance->getProductRelatedPageURIPart($page_type,
								$language_code);
							
							if ($uri_part == false) {
								// Unexpected database problem encountered
								continue;
							}
							
							$uri_part = $this->_convertStringForURI($uri_part, $language_code);
							
							$uri = $base_uri . $uri_part;
							
							$main_page = constant('FILENAME_' . strtoupper($page_type));
							
							$mapping_added = $this->addURIMapping($uri, $languages[$i]['id'], $main_page, null,
								$this->_product_id);
							
							if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
								$success_message = sprintf(TEXT_PRODUCT_MAPPING_ADDED,
									ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
									'" target="_blank">' . $uri . '</a>');
								
								$messageStack->add($success_message, 'success');
								
							} else {
								if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
									$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS,
										ucwords($languages[$i]['name'])) . ' <a href="' . HTTP_SERVER . $uri .
										'" target="_blank">' . $uri . '</a>';
										
								} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
									$failure_message = sprintf(CEON_URI_MAPPING_TEXT_ERROR_ADD_MAPPING_DATA,
										ucwords($languages[$i]['name'])) . $uri;
								} else {
									$failure_message = sprintf(CEON_URI_MAPPING_TEXT_ERROR_ADD_MAPPING_DB,
										ucwords($languages[$i]['name'])) . $uri;
								}
								
								$messageStack->add($failure_message, 'error');
							}
						}
					}
				} else if ($prev_mapping != '' && $mapping == '') {
					// No URI mapping, consign existing mapping to the history, so old URI mapping isn't broken
					if ($this->_object_type == 'category') {
						
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
						
						$success_message =
							sprintf(TEXT_CATEGORY_MAPPING_MADE_HISTORICAL, ucwords($languages[$i]['name']));
						
						$messageStack->add($success_message, 'caution');
						
					} else if ($this->_object_type == 'product') {
						
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
						
						$success_message =
							sprintf(TEXT_PRODUCT_MAPPING_MADE_HISTORICAL, ucwords($languages[$i]['name']));
						
						$messageStack->add($success_message, 'caution');
						
						
						foreach ($page_types_to_manage as $page_type) {
							$columns_to_retrieve = array(
								'uri'
								);
							
							$selections = array(
								'main_page' => $page_type,
								'associated_db_id' => (int) $this->_product_id,
								'language_id' => (int) $languages[$i]['id'],
								'current_uri' => 1,
								);
							
							$order_by = null;
							
							$current_uri_mapping_result =
								$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
							
							if (!$current_uri_mapping_result->EOF) {
								$prp_uri_mapping = $current_uri_mapping_result->fields['uri'];
								
								$this->makeURIMappingHistorical($prp_uri_mapping, $languages[$i]['id']);
								
								$success_message = sprintf(TEXT_PRODUCT_RELATED_PAGE_MAPPING_MADE_HISTORICAL,
									ucwords($languages[$i]['name']), $prp_uri_mapping);
								
								$messageStack->add($success_message, 'caution');
							}
						}
					}
				}
			}
		}
	}
	
	// }}}
}

// }}}
