<?php

/**
 * Ceon URI Mappings Manager Install/Upgrade Class - Installs/configures the module, upgrades from an earlier
 * version or finds and fixes and configuration/database issues.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMInstallOrUpgrade.php 1059 2012-10-01 16:43:34Z conor $
 */


// {{{ CeonUMMInstallOrUpgrade

/**
 * Installs or upgrades Ceon URI Mappings Manager. If a previous installation/upgrade attempt failed before it
 * completed, the class can be run again as it will attempt to find and fix any configuration/database issues.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMInstallOrUpgrade
{
	// {{{ properties
	
	/**
	 * The version of the module.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_version = null;
	
	/**
	 * The version of the module which is currently installed.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_installed_version = null;
	
	/**
	 * Tracks if the configs table has just been created.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_configs_table_created = false;
	
	/**
	 * Tracks if the category mapping templates table has just been created.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_category_mapping_templates_table_created = false;
	
	/**
	 * Tracks if the produc mapping templates table has just been created.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_product_mapping_templates_table_created = false;
	
	/**
	 * Tracks if the page mapping templates table has just been created.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_page_mapping_templates_table_created = false;
	
	/**
	 * Tracks if the other page types table has just been created.
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_other_pages_table_created = false;
	
	/**
	 * Maintains a list of any errors encountered in an installation or upgrade.
	 *
	 * @var     array
	 * @access  public
	 */
	public $error_messages = array();
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class. Installs/upgrades the database and adds or updates configuration
	 * options.
	 * 
	 * @access  public
	 * @param   string    $version   The version of the module.
	 * @param   string    $installed_version   The currently installed version of the module.
	 */
	public function __construct($version, $installed_version)
	{
		global $db;
		
		$this->_version = $version;
		
		$this->_installed_version = $installed_version;
		
		$this->_checkCreateDatabases();
		
		if (sizeof($this->error_messages) > 0) {
			// Can't progress without working databases
			return;
		}
		
		$this->_ensureDatabaseIsUpToDate();
		
		if (sizeof($this->error_messages) > 0) {
			// Can't progress without working databases
			return;
		}
		
		$this->_checkForDefaultConfig();
		
		$this->_checkForDefaultMappingTemplates();
		
		$this->_checkOtherPageTypesTablePopulated();
		
		$this->_checkZenCartConfigGroupAndOptions();
		
		// If updates were performed and all were successful, update version number 
		if (sizeof($this->error_messages) == 0 && $this->_installed_version != $this->_version) {
			// Only one config currently supported so the ID is hard-coded in the following SQL
			$update_db_version_number_sql = "
				UPDATE
					" . TABLE_CEON_UMM_CONFIGS . "
				SET
					version = '" . $this->_version . "'
				WHERE
					id = '1';";
			
			$update_db_version_number_result = $db->Execute($update_db_version_number_sql);
			
			// Reset the version check status as it may change when the software is upgraded
			if (isset($this->_ceon_base_model_code) && isset($_SESSION[$this->_ceon_base_model_code . '_vc_response'])) {
				unset($_SESSION[$this->_ceon_base_model_code . '_vc_response']);
			}
		}
	}
	
	// }}}
	
	
	// {{{ _checkCreateDatabases()
	
	/**
	 * Makes sure that the database tables exist. Creates any that don't.
	 *
	 * @access  protected
	 * @return  boolean   False if a problem occurred when creating a database table, true otherwise.
	 */
	protected function _checkCreateDatabases()
	{
		global $db, $messageStack;
		
		// Add the configuration table if it doesn't exist
		$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_CONFIGS . '";';
		$table_exists_result = $db->Execute($table_exists_query);
		
		if ($table_exists_result->EOF) {
			if (!$this->_DBUserHasPrivilege('CREATE')) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
			}
			
			$create_configs_table_sql = "
				CREATE TABLE IF NOT EXISTS
					`" . TABLE_CEON_UMM_CONFIGS . "`
					(
					`id` INT UNSIGNED NOT NULL,
					`version` VARCHAR(14) NOT NULL,
					`use_ajax_progress_indicators` INT(1) UNSIGNED NOT NULL,
					`ajax_progress_refresh_sec` INT(6) UNSIGNED NOT NULL,
					`standard_progress_refresh_sec` INT(8) UNSIGNED NOT NULL,
					`automatic_version_checking` INT(1) UNSIGNED DEFAULT 1,
					PRIMARY KEY (`id`)
					);";
			
			$create_configs_table_result = $db->Execute($create_configs_table_sql);
			
			// Check the table was created
			$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_CONFIGS . '";';
			
			$table_exists_result = $db->Execute($table_exists_query);
			
			if ($table_exists_result->EOF) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
				
			}
			
			$this->_configs_table_created = true;
			
			$messageStack->add('Configuration database table successfully created.', 'success');
		}
		
		
		// Add the category mapping templates table if it doesn't exist
		$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES . '";';
		
		$table_exists_result = $db->Execute($table_exists_query);
		
		if ($table_exists_result->EOF) {
			if (!$this->_DBUserHasPrivilege('CREATE')) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
			}
			
			$create_mapping_templates_table_sql = "
				CREATE TABLE
					`" . TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES . "`
					(
					`category_id` INT UNSIGNED NOT NULL,
					`language_id` INT(11) UNSIGNED DEFAULT '0',
					`mapping_template` VARCHAR(240) NOT NULL,
					`date_added_modified` DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
					INDEX `id_idx` (`category_id`, `language_id`)
					);";
			
			$create_mapping_templates_table_result = $db->Execute($create_mapping_templates_table_sql);
			
			// Check the table was created
			$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES . '";';
			
			$table_exists_result = $db->Execute($table_exists_query);
			
			if ($table_exists_result->EOF) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
			}
			
			$this->_category_mappings_template_table_created = true;
			
			$messageStack->add('Category Mapping Templates database table successfully created.', 'success');
		}
		
		
		// Add the product mapping templates table if it doesn't exist
		$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES . '";';
		
		$table_exists_result = $db->Execute($table_exists_query);
		
		if ($table_exists_result->EOF) {
			if (!$this->_DBUserHasPrivilege('CREATE')) {
				$this->error_messages[] = 'Database table can not be created! The database user may not have' .
					' CREATE TABLE privileges!';
				
				return false;
			}
			
			$create_mapping_templates_table_sql = "
				CREATE TABLE
					`" . TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES . "`
					(
					`category_id` INT UNSIGNED NOT NULL,
					`language_id` INT(11) UNSIGNED DEFAULT '0',
					`mapping_template` VARCHAR(240) NOT NULL,
					`date_added_modified` DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
					INDEX `id_idx` (`category_id`, `language_id`)
					);";
			
			$create_mapping_templates_table_result = $db->Execute($create_mapping_templates_table_sql);
			
			// Check the table was created
			$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES . '";';
			
			$table_exists_result = $db->Execute($table_exists_query);
			
			if ($table_exists_result->EOF) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
			}
			
			$this->_product_mappings_template_table_created = true;
			
			$messageStack->add('Product Mapping Templates database table successfully created.', 'success');
		}
		
		
		// Add the page mapping templates table if it doesn't exist
		$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . '";';
		
		$table_exists_result = $db->Execute($table_exists_query);
		
		if ($table_exists_result->EOF) {
			if (!$this->_DBUserHasPrivilege('CREATE')) {
				$this->error_messages[] = 'Database table can not be created! The database user may not have' .
					' CREATE TABLE privileges!';
				
				return false;
			}
			
			$create_mapping_templates_table_sql = "
				CREATE TABLE
					`" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "`
					(
					`page_type` VARCHAR(40) NOT NULL,
					`language_id` INT(11) UNSIGNED DEFAULT '0',
					`mapping_template` VARCHAR(240) NOT NULL,
					`date_added_modified` DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
					INDEX `id_idx` (`page_type`, `language_id`)
					);";
			
			$create_mapping_templates_table_result = $db->Execute($create_mapping_templates_table_sql);
			
			// Check the table was created
			$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . '";';
			
			$table_exists_result = $db->Execute($table_exists_query);
			
			if ($table_exists_result->EOF) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
			}
			
			$this->_page_mappings_template_table_created = true;
			
			$messageStack->add('Page Mapping Templates database table successfully created.', 'success');
		}
		
		
		// Add the other pages table if it doesn't exist
		$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_OTHER_PAGES . '";';
		
		$table_exists_result = $db->Execute($table_exists_query);
		
		if ($table_exists_result->EOF) {
			if (!$this->_DBUserHasPrivilege('CREATE')) {
				$this->error_messages[] = 'Database table can not be created! The database user' .
					' does not have CREATE TABLE privileges!';
				
				return false;
			}
			
			$create_other_pages_table_sql = "
				CREATE TABLE
					`" . TABLE_CEON_UMM_OTHER_PAGES . "`
					(
					`id`  INT(6) UNSIGNED NOT NULL AUTO_INCREMENT,
					`sort_order` INT(5) UNSIGNED NOT NULL,
					`main_page` VARCHAR(45) NOT NULL,
					`query_string_parameters` VARCHAR(255) DEFAULT NULL,
					`use_define_type`  INT(1) UNSIGNED NOT NULL DEFAULT 1,
					`box_heading_define_name` VARCHAR(140) DEFAULT NULL,
					`heading_link_title_define_name` VARCHAR(140) DEFAULT NULL,
					`breadcrumb_define_name` VARCHAR(140) DEFAULT NULL,
					`date_added` DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
					`date_modified` DATETIME NULL,
					PRIMARY KEY (`id`)
					);";
			
			$create_other_pages_table_result = $db->Execute($create_other_pages_table_sql);
			
			// Check the table was created
			$table_exists_query = 'SHOW TABLES LIKE "' . TABLE_CEON_UMM_OTHER_PAGES . '";';
			
			$table_exists_result = $db->Execute($table_exists_query);
			
			if ($table_exists_result->EOF) {
				$this->error_messages[] = 'Database table could not be created! The database user may not have' .
					' CREATE TABLE privileges?!';
				
				return false;
			}
			
			$this->_other_pages_table_created = true;
			
			$messageStack->add('Other Pages database table successfully created.', 'success');
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _ensureDatabaseIsUpToDate()
	
	/**
	 * Makes sure that the database tables are up to date.
	 *
	 * @access  protected
	 * @return  boolean   False if a problem occurred when updating a database table, true otherwise.
	 */
	protected function _ensureDatabaseIsUpToDate()
	{
		global $db;
		
		// If a problem occurs trying to update a table, take note but continue to try and update any other table,
		// so things are as good as can be in a bad situation
		$problem_updating_table = false;
		
		if (!$this->_other_pages_table_created) {
			if (!$this->_checkUpdateOtherPagesTable()) {
				$problem_updating_table = true;
			}
		}
		
		if ($problem_updating_table) {
			return false;
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _checkUpdateOtherPagesTable()
	
	/**
	 * Updates the length of particular columns in the Other Pages database table.
	 *
	 * @access  protected
	 * @return  boolean   Whether or not the database table is up to date.
	 */
	protected function _checkUpdateOtherPagesTable()
	{
		global $db, $messageStack;
		
		// Get the list of columns in the database table
		$columns = array();
		
		$columns_query = 'SHOW COLUMNS FROM ' . TABLE_CEON_UMM_OTHER_PAGES . ';';
		$columns_result = $db->Execute($columns_query);
		
		$column_length_out_of_date = false;
		
		while (!$columns_result->EOF) {
			$column_type = strtolower(preg_replace('/\s/', '', $columns_result->fields['Type']));
			
			if ($columns_result->fields['Field'] == 'box_heading_define_name' &&
					strpos($column_type, 'varchar(80)') !== false) {
				$column_length_out_of_date = true;
				
				break;
			}
			
			$columns_result->MoveNext();
		}
		
		if ($column_length_out_of_date) {
			// Updating old version beta1 database table
			if (!$this->_DBUserHasPrivilege('ALTER')) {
				$this->error_messages[] = 'Database table can not be updated! The database user may not have' .
					' ALTER TABLE privileges!';
				
				return false;
			}
			
			// Update the length of the column
			$update_column_sql = "
				ALTER TABLE
					" . TABLE_CEON_UMM_OTHER_PAGES . "
				CHANGE
					`box_heading_define_name`
					`box_heading_define_name` VARCHAR(140) DEFAULT NULL;";
			
			$update_column_result = $db->Execute($update_column_sql);
			
			
			// Verify that the column was modified
			$columns_query = 'SHOW COLUMNS FROM ' . TABLE_CEON_UMM_OTHER_PAGES . ';';
			$columns_result = $db->Execute($columns_query);
			
			$column_length_out_of_date = false;
			
			while (!$columns_result->EOF) {
				$column_type = strtolower(preg_replace('/\s/', '', $columns_result->fields['Type']));
				
				if ($columns_result->fields['Field'] == 'box_heading_define_name' &&
						strpos($column_type, 'varchar(80)') !== false) {
					$column_length_out_of_date = true;
					
					break;
				}
				
				$columns_result->MoveNext();
			}
			
			if ($column_length_out_of_date) {
				// Unable to modify column! The database user may not actually have ALTER TABLE privileges
				$this->error_messages[] = 'Unable to modify column! The database user may not have ALTER TABLE' .
					' privileges';
				
				return false;
			}
			
			// Assume all other ALTER operations will work fine if the first one did
			
			$update_column_sql = "
				ALTER TABLE
					" . TABLE_CEON_UMM_OTHER_PAGES . "
				CHANGE
					`heading_link_title_define_name`
					`heading_link_title_define_name` VARCHAR(140) DEFAULT NULL;";
			
			$update_column_result = $db->Execute($update_column_sql);
			
			$update_column_sql = "
				ALTER TABLE
					" . TABLE_CEON_UMM_OTHER_PAGES . "
				CHANGE
					`breadcrumb_define_name`
					`breadcrumb_define_name` VARCHAR(140) DEFAULT NULL;";
			
			$update_column_result = $db->Execute($update_column_sql);
			
			$messageStack->add('Other Pages database table successfully updated from older beta1 format.',
				'success');
		}
	}
	
	// }}}
	
	
	// {{{ _checkForDefaultConfig()
	
	/**
	 * Makes sure that the default configuration exists. If not, it is created.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _checkForDefaultConfig()
	{
		global $db;
		
		$check_config_exists_sql = "
			SELECT
				id
			FROM
				" . TABLE_CEON_UMM_CONFIGS . "
			WHERE
				1 = 1;";
		
		$check_config_exists_result = $db->Execute($check_config_exists_sql);
		
		if (!$check_config_exists_result->EOF) {
			// Config already exists
			return;
		}
		
		$create_default_config_sql = "
			INSERT INTO
				" . TABLE_CEON_UMM_CONFIGS . "
				(
				id,
				version,
				use_ajax_progress_indicators,
				ajax_progress_refresh_sec,
				standard_progress_refresh_sec,
				automatic_version_checking
				)
			VALUES
				(
				'1',
				'" . $this->_version . "',
				'1',
				'5',
				'10',
				'1'
				);";
		
		$create_default_config_result = $db->Execute($create_default_config_sql);
	}
	
	// }}}
	
	
	// {{{ _checkForDefaultMappingTemplates()
	
	/**
	 * Makes sure that there are default mapping templates. If there aren't, they are added.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _checkForDefaultMappingTemplates()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		for ($i = 0; $i < $num_languages; $i++) {
			// Check the default template for categories/subcategories, for the current language
			$check_mapping_template_exists_sql = "
				SELECT
					mapping_template
				FROM
					" . TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES . "
				WHERE
					category_id = '0'
				AND
					language_id = '" . (int) $languages[$i]['id'] . "';";
			
			$check_mapping_template_exists_result = $db->Execute($check_mapping_template_exists_sql);
			
			if ($check_mapping_template_exists_result->EOF) {
				$create_default_mapping_template_sql = "
					INSERT INTO
						" . TABLE_CEON_UMM_CATEGORY_MAPPING_TEMPLATES . "
						(
						category_id,
						language_id,
						mapping_template,
						date_added_modified
						)
					VALUES
						(
						'0',
						'" . (int) $languages[$i]['id'] . "',
						'/{dir-ws-catalog}/{category-path}/{category-name}',
						NOW()
						);";
				
				$create_default_mapping_template_result = $db->Execute($create_default_mapping_template_sql);
				
				$messageStack->add(sprintf('Default %s Mapping Template added for Categories/Subcategories',
					ucwords($languages[$i]['name'])), 'success');
			}
			
			// Check the default template for products in a category, for the current language
			$check_mapping_template_exists_sql = "
				SELECT
					mapping_template
				FROM
					" . TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES . "
				WHERE
					category_id = '0'
				AND
					language_id = '" . (int) $languages[$i]['id'] . "';";
			
			$check_mapping_template_exists_result =
				$db->Execute($check_mapping_template_exists_sql);
			
			if ($check_mapping_template_exists_result->EOF) {
				$create_default_mapping_template_sql = "
					INSERT INTO
						" . TABLE_CEON_UMM_PRODUCT_MAPPING_TEMPLATES . "
						(
						category_id,
						language_id,
						mapping_template,
						date_added_modified
						)
					VALUES
						(
						'0',
						'" . (int) $languages[$i]['id'] . "',
						'/{dir-ws-catalog}/{category-path}/{product-name}',
						NOW()
						);";
				
				$create_default_mapping_template_result = $db->Execute($create_default_mapping_template_sql);
				
				$messageStack->add(sprintf('Default %s Mapping Template added for Products',
					ucwords($languages[$i]['name'])), 'success');
			}
			
			// Check the default template for manufacturers, for the current language
			$check_mapping_template_exists_sql = "
				SELECT
					page_type
				FROM
					" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "
				WHERE
					page_type = 'manufacturer'
				AND
					language_id = '" . (int) $languages[$i]['id'] . "';";
			
			$check_mapping_template_exists_result = $db->Execute($check_mapping_template_exists_sql);
			
			if ($check_mapping_template_exists_result->EOF) {
				$create_default_mapping_template_sql = "
					INSERT INTO
						" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "
						(
						page_type,
						language_id,
						mapping_template,
						date_added_modified
						)
					VALUES
						(
						'manufacturer',
						'" . (int) $languages[$i]['id'] . "',
						'/{dir-ws-catalog}/{manufacturer-name}',
						NOW()
						);";
				
				$create_default_mapping_template_result = $db->Execute($create_default_mapping_template_sql);
				
				$messageStack->add(sprintf('Default %s Mapping Template added for Manufacturers',
					ucwords($languages[$i]['name'])), 'success');
			}
			
			// Check the default template for EZ-Pages, for the current language
			$check_mapping_template_exists_sql = "
				SELECT
					page_type
				FROM
					" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "
				WHERE
					page_type = 'ez-page'
				AND
					language_id = '" . (int) $languages[$i]['id'] . "';";
			
			$check_mapping_template_exists_result = $db->Execute($check_mapping_template_exists_sql);
			
			if ($check_mapping_template_exists_result->EOF) {
				$create_default_mapping_template_sql = "
					INSERT INTO
						" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "
						(
						page_type,
						language_id,
						mapping_template,
						date_added_modified
						)
					VALUES
						(
						'ez-page',
						'" . (int) $languages[$i]['id'] . "',
						'/{dir-ws-catalog}/{ez-page-name}',
						NOW()
						);";
				
				$create_default_mapping_template_result = $db->Execute($create_default_mapping_template_sql);
				
				$messageStack->add(sprintf('Default %s Mapping Template added for EZ-Pages',
					ucwords($languages[$i]['name'])), 'success');
			}
			
			// Check the default template for other Zen Cart pages, for the current language
			$check_mapping_template_exists_sql = "
				SELECT
					page_type
				FROM
					" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "
				WHERE
					page_type = 'other-page'
				AND
					language_id = '" . (int) $languages[$i]['id'] . "';";
			
			$check_mapping_template_exists_result =
				$db->Execute($check_mapping_template_exists_sql);
			
			if ($check_mapping_template_exists_result->EOF) {
				$create_default_mapping_template_sql = "
					INSERT INTO
						" . TABLE_CEON_UMM_PAGE_MAPPING_TEMPLATES . "
						(
						page_type,
						language_id,
						mapping_template,
						date_added_modified
						)
					VALUES
						(
						'other-page',
						'" . (int) $languages[$i]['id'] . "',
						'/{dir-ws-catalog}/{page-define-text}',
						NOW()
						);";
				
				$create_default_mapping_template_result = $db->Execute($create_default_mapping_template_sql);
				
				$messageStack->add(sprintf('Default %s Mapping Template added for Other Zen Cart' .
					' Pages', ucwords($languages[$i]['name'])), 'success');
			}
		}
	}
	
	// }}}
	
	
	// {{{ _checkOtherPageTypesTablePopulated()
	
	/**
	 * Makes sure that the other page types database is populated. If not, it populate it with some defaults.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _checkOtherPageTypesTablePopulated()
	{
		global $db, $messageStack;
		
		$check_table_populated_sql = "
			SELECT
				id
			FROM
				" . TABLE_CEON_UMM_OTHER_PAGES . "
			WHERE
				1 = 1
			LIMIT
				1;";
		
		$check_table_populated_result = $db->Execute($check_table_populated_sql);
		
		if (!$check_table_populated_result->EOF) {
			// Table populated
			return;
		}
		
		// Table empty so add some default values
		// Spaces are used within define names as an array delimiter
		$other_pages_info = array(
			array(
				'page' => 'account',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADER_TITLE_MY_ACCOUNT',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'account_back_in_stock_notifications',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'ACCOUNT_BACK_IN_STOCK_NOTIFICATIONS_NAVBAR_TITLE_1' .
					' ' . 'ACCOUNT_BACK_IN_STOCK_NOTIFICATIONS_NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'account_edit',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'account_history',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'account_history_info',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2' . ' ' .
					'HEADING_TITLE',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'account_newsletters',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'account_notifications',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'account_password',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'address_book',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'address_book_process',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2' . ' ' .
					'HEADING_TITLE',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'advanced_search',
				'box_heading_define_name' => 'BOX_SEARCH_ADVANCED_SEARCH',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'advanced_search_result',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'back_in_stock_notification_subscribe',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'BACK_IN_STOCK_NOTIFICATION_NAVBAR_TITLE',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'back_in_stock_notification_unsubscribe',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'BACK_IN_STOCK_NOTIFICATION_UNSUBSCRIBE_NAVBAR_TITLE',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'checkout_confirmation',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'checkout_payment',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'checkout_payment_address',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'checkout_shipping',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADER_TITLE_CHECKOUT',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'checkout_shipping_address',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_1' . ' ' . 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'checkout_success',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => 'NAVBAR_TITLE_2',
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'conditions',
				'box_heading_define_name' => 'BOX_INFORMATION_CONDITIONS',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'contact_us',
				'box_heading_define_name' => 'BOX_INFORMATION_CONTACT',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'create_account',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADER_TITLE_CREATE_ACCOUNT',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'create_account_success',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'discount_coupon',
				'box_heading_define_name' => 'BOX_INFORMATION_DISCOUNT_COUPONS',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'featured_products',
				'box_heading_define_name' => 'BOX_HEADING_FEATURED_PRODUCTS',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'gv_faq',
				'box_heading_define_name' => 'BOX_INFORMATION_GV',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'gv_redeem',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'gv_send',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'login',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADER_TITLE_LOGIN',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'logoff',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADER_TITLE_LOGOFF',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'page_2',
				'box_heading_define_name' => 'BOX_INFORMATION_PAGE_2',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'page_3',
				'box_heading_define_name' => 'BOX_INFORMATION_PAGE_3',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'page_4',
				'box_heading_define_name' => 'BOX_INFORMATION_PAGE_4',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'password_forgotten',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADING_TITLE',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'privacy',
				'box_heading_define_name' => 'BOX_INFORMATION_PRIVACY',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'products_all',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'products_new',
				'box_heading_define_name' => 'BOX_HEADING_WHATS_NEW',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'reviews',
				'box_heading_define_name' => 'BOX_HEADING_REVIEWS',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'shippinginfo',
				'box_heading_define_name' => 'BOX_INFORMATION_SHIPPING',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'shopping_cart',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => 'HEADER_TITLE_CART_CONTENTS',
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME
				),
			array(
				'page' => 'site_map',
				'box_heading_define_name' => 'BOX_INFORMATION_SITE_MAP',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'specials',
				'box_heading_define_name' => 'BOX_HEADING_SPECIALS',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				),
			array(
				'page' => 'subscribe',
				'box_heading_define_name' => null,
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME
				),
			array(
				'page' => 'unsubscribe',
				'box_heading_define_name' => 'BOX_INFORMATION_UNSUBSCRIBE',
				'heading_link_title_define_name' => null,
				'breadcrumb_define_name' => null,
				'use_define_type' => CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME
				)
			);
		
		// Since this is the first population, start the sort order from 1
		$sort_order = 1;
		
		foreach ($other_pages_info as $other_page_info) {
			$add_page_info_sql = "
				INSERT INTO
					" . TABLE_CEON_UMM_OTHER_PAGES . "
					(
					main_page,
					sort_order,
					use_define_type,
					box_heading_define_name,
					heading_link_title_define_name,
					breadcrumb_define_name,
					date_added
					)
				VALUES
					(
					'" . $other_page_info['page'] . "',
					'" . $sort_order . "',
					'" . $other_page_info['use_define_type'] . "',
					" . (!is_null($other_page_info['box_heading_define_name']) ?
						"'" . $other_page_info['box_heading_define_name'] . "'" : 'NULL') . ",
					" . (!is_null($other_page_info['heading_link_title_define_name']) ?
						"'" . $other_page_info['heading_link_title_define_name'] . "'" : 'NULL') . ",
					" . (!is_null($other_page_info['breadcrumb_define_name']) ?
						"'" . $other_page_info['breadcrumb_define_name'] . "'" : 'NULL') . ",
					NOW()
					);";
			
			$add_page_info_result = $db->Execute($add_page_info_sql);
			
			$sort_order++;
		}
		
		$messageStack->add('Default list of Other Zen Cart Page Types added to database table.', 'success');
	}
	
	// }}}
	
	
	// {{{ _checkZenCartConfigGroupAndOptions()
	
	/**
	 * Makes sure that the configuration group and options are present in the Zen Cart configuration table. If any
	 * are missing, they are created.
	 *
	 * @access  protected
	 * @return  boolean   Whether or not the configuration group and options are present and valid.
	 */
	protected function _checkZenCartConfigGroupAndOptions()
	{
		global $db, $messageStack;
		
		$check_config_group_exists_sql = "
			SELECT
				configuration_group_id
			FROM
				" . TABLE_CONFIGURATION_GROUP . "
			WHERE
				configuration_group_title = 'Ceon URI Mappings Manager';";
		
		$check_config_group_exists_result = $db->Execute($check_config_group_exists_sql);
		
		if (!$check_config_group_exists_result->EOF) {
			$configuration_group_id = $check_config_group_exists_result->fields['configuration_group_id'];
		} else {
			$add_config_group_options_sql = "
				INSERT INTO
					" . TABLE_CONFIGURATION_GROUP . "
					(
					configuration_group_title,
					configuration_group_description,
					sort_order,
					visible
					)
				VALUES
					(
					'Ceon URI Mappings Manager',
					'Set Ceon URI Mappings Manager Options',
					'1',
					'1'
					);";
			
			$add_config_group_options_result = $db->Execute($add_config_group_options_sql);
			
			$this->_actions_performed = true;
			
			$configuration_group_id_sql = "
				SELECT
					configuration_group_id  
				FROM
					" . TABLE_CONFIGURATION_GROUP . "
				WHERE
					configuration_group_title = 'Ceon URI Mappings Manager';";
			
			$configuration_group_id_result = $db->Execute($configuration_group_id_sql);
			
			if (!$configuration_group_id_result->EOF) {
				$configuration_group_id = $configuration_group_id_result->fields['configuration_group_id'];
			} else {
				// Problem getting ID!
				$this->error_messages[] = 'Couldn\'t get the ID of the configuration group!';
				
				return false;
			}
			
			$set_group_sort_order_sql = "
				UPDATE
					" . TABLE_CONFIGURATION_GROUP . "
				SET
					sort_order = '" . $configuration_group_id . "'
				WHERE
					configuration_group_id = '" . $configuration_group_id . "';";
			
			$set_group_sort_order_result = $db->Execute($set_group_sort_order_sql);
			
			$messageStack->add('Configuration group created.', 'success');
		}
		
		$check_config_option_exists_sql = "
			SELECT
				configuration_group_id
			FROM
				" . TABLE_CONFIGURATION . "
			WHERE
				configuration_key = 'CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES';";
		
		$check_config_option_exists_result = $db->Execute($check_config_option_exists_sql);
		
		if (!$check_config_option_exists_result->EOF) {
			// Make sure the option is assigned to the correct group
			if ($check_config_option_exists_result->fields['configuration_group_id'] != $configuration_group_id) {
				
				$set_group_id_sql = "
					UPDATE
						" . TABLE_CONFIGURATION . "
					SET
						configuration_group_id = '" . $configuration_group_id . "'
					WHERE
						configuration_key = 'CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES';";
				
				$set_group_id_result = $db->Execute($set_group_id_sql);
				
				$messageStack->add('Configuration option assigned to correct group.', 'success');
				
				$this->_actions_performed = true;
			}
		} else {
			$add_config_option_sql = "
				INSERT INTO
					" . TABLE_CONFIGURATION . "
					(
					`configuration_title`,
					`configuration_key`,
					`configuration_value`,
					`configuration_description`,
					`configuration_group_id`,
					`sort_order`,
					`set_function`,
					`date_added`
					)
				VALUES
					(
					'Auto-generate URIs for Unmapped Categories/Products/Manufacturers/EZ-Pages',
					'CEON_URI_MAPPINGS_MANAGER_AUTOGENERATE_URIS_FOR_UNMAPPED_PAGES',
					'0',
					'<br /><fieldset><legend><strong>Option Overview</strong></legend>If a user is browsing the site and they encounter any Categories/Products/Manufacturers/EZ-Pages which are still using the standard Zen Cart dynamically-built URIs, because no URI mapping has been generated/set for them in the admin, should the software attempt to create a static URI mapping there and then?<br /><br />If so, when a page is linked to that doesn''t yet have a static URI, the software will look up the template for URI mappings for the page type and attempt to generate the page''s URI mapping. This will then be used instead of the dynamic Zen Cart URI.</fieldset><br /><fieldset><legend><strong>Please Note:</strong></legend>This isn''t as comprehensive a process as using the URI Mappings Manager itself:<br /><br />If any errors are encountered in the auto-generation process, the software simply won''t generate the static URI for the page.<br /><br />Also, even if &ldquo;Auto-append&rdquo; is enabled for product URI mappings, no URI mapping will be generated if the first attempt to generate a product''s URI results in a URI mapping that would clash with an existing mapping (no further attempt to find a unique mapping for the product would be made).<br /><br />This &ldquo;auto-generation fallback&rdquo; functionality applies only for Categories/Products/Manufacturers/EZ-Pages, it does not apply for &ldquo;other&rdquo; Zen Cart pages, the URI Mappings Manager must be used for them.</fieldset><br /><br />0 = off <br />1 = on',
					'" . $configuration_group_id . "',
					'1',
					'zen_cfg_select_option(array(''0'', ''1''), ',
					NOW()
					);";
			
			$add_config_option_result = $db->Execute($add_config_option_sql);
			
			$messageStack->add('Auto-generate URIs configuration group option added.', 'success');
			
			$this->_actions_performed = true;
		}
		
		$check_config_option_exists_sql = "
			SELECT
				configuration_group_id
			FROM
				" . TABLE_CONFIGURATION . "
			WHERE
				configuration_key =
					'CEON_URI_MAPPINGS_MANAGER_ADD_HISTORICAL_URIS_FOR_UNMAPPED_SEO_MODULE_PAGES';";
		
		$check_config_option_exists_result = $db->Execute($check_config_option_exists_sql);
		
		if (!$check_config_option_exists_result->EOF) {
			// Make sure the option is assigned to the correct group
			if ($check_config_option_exists_result->fields['configuration_group_id'] != $configuration_group_id) {
				
				$set_group_id_sql = "
					UPDATE
						" . TABLE_CONFIGURATION . "
					SET
						configuration_group_id = '" . $configuration_group_id . "'
					WHERE
						configuration_key =
							'CEON_URI_MAPPINGS_MANAGER_ADD_HISTORICAL_URIS_FOR_UNMAPPED_SEO_MODULE_PAGES';";
				
				$set_group_id_result = $db->Execute($set_group_id_sql);
				
				$messageStack->add('Configuration option assigned to correct group.', 'success');
				
				$this->_actions_performed = true;
			}
		} else {
			$add_config_option_sql = "
				INSERT INTO
					" . TABLE_CONFIGURATION . "
					(
					`configuration_title`,
					`configuration_key`,
					`configuration_value`,
					`configuration_description`,
					`configuration_group_id`,
					`sort_order`,
					`set_function`,
					`date_added`
					)
				VALUES
					(
					'Identify and Convert Ultimate SEO URLs/Simple SEO URLs URIs to Historical URIs?',
					'CEON_URI_MAPPINGS_MANAGER_ADD_HISTORICAL_URIS_FOR_UNMAPPED_SEO_MODULE_PAGES',
					'None - this feature is disabled',
					'<br /><fieldset><legend><strong>Option Overview</strong></legend>When a static URI from Ultimate SEO URLS or Simple SEO URLs is identified as being used to try and access a page on the site, should the software create a historical URI mapping there and then?<br /><br />If neither of these other modules were ever used for the store, enabling this option wastes system resources unnecessarily, so it is disabled by default.</fieldset><br /><fieldset><legend><strong>WARNING:</strong></legend>In an ideal world this option would be enabled until all possible old URIs have their historical URIs created.<br /><br />However, sad people exist on this planet, and this option could potentially be hijacked to add useless records to the database table, if URIs matching any of these modules'' formats are used to access the store, as the URIs could just be made up (e.g. /this-uri-never-existed-but-no-way-to-tell-p-274.html), so it might not be the best idea to leave this on permanently.<br /><br />As a very basic defence against such malicious usage, the software will not add any more historical URI records to the database for any particular product/category etc. once an arbitrary limit of 10 historical URIs for each has been found to have been met.</fieldset><br /><br />Select which modules have been used on this store before, and for which the software should attempt to identify and convert the old URIs:',
					'" . $configuration_group_id . "',
					'1',
					'zen_cfg_select_option(array(''None - this feature is disabled'', ''Ultimate SEO URLs'', ''Simple SEO URLs'', ''Both Ultimate SEO URLs and Simple SEO URLs''), ',
					NOW()
					);";
			
			$add_config_option_result = $db->Execute($add_config_option_sql);
			
			$messageStack->add('Add Historical URIs configuration group option added.', 'success');
			
			$this->_actions_performed = true;
		}
		
		// Make sure configuration group can be displayed in admin menu and remove any old/duplicate menu item
		if (function_exists('zen_register_admin_page')) {
			if (!zen_page_key_exists('ceon_uri_mappings_manager_cg')) {
				// Add the link to the Ceon URI Mappings Manager Zen Cart configuration options to the admin menu
				zen_register_admin_page('ceon_uri_mappings_manager_cg', 'BOX_CEON_UMM_CONFIG_GROUP',
					'FILENAME_CONFIGURATION', 'gID=' . $configuration_group_id, 'configuration', 'Y',
					$configuration_group_id);
				
				$messageStack->add('Configuration group added to admin menu.', 'success');
			}
			
			if (zen_page_key_exists('ceon_uri_mappings_manager_config')) {
				zen_deregister_admin_pages('ceon_uri_mappings_manager_config');
				
				$messageStack->add('Old configuration group removed from admin menu.', 'success');
			}
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _DBUserHasPrivilege()
	
	/**
	 * Checks if the current database user has a particular privilege type for the store's database.
	 *
	 * @access  protected
	 * @param   string    $privilege_type   The type of privilege to be checked for.
	 * @return  boolean   Whether or not the current database user has the specified privilege.
	 */
	protected function _DBUserHasPrivilege($privilege_type)
	{
		global $db;
		
		if (isset($_GET['override-db-privileges-check'])) {
			// Unfortunately some servers have been found to not return the correct list of privileges! Make things
			// easier for the user by allowing an override flag to be used
			return true;
		}
		
		$privilege_type = strtoupper($privilege_type);
		
		$db_user_has_privilege = false;
		
		$check_user_privileges_sql = "SHOW GRANTS;";
		
		$check_user_privileges_result = $db->Execute($check_user_privileges_sql);
		
		while (!$check_user_privileges_result->EOF) {
			// Check each GRANT for the current user to see if it covers the current database and gives the
			// specified privilege
			foreach ($check_user_privileges_result->fields as $current_grant_string) {
				if (preg_match('/GRANT (.*) ON (.*) TO /i', $current_grant_string, $matches)) {
					// Get the privilege string
					$privilege_string = strtoupper($matches[1]);
					
					// Extract the database name for this grant string by removing all the characters/strings it
					// could be surrounded with or followed by
					$database_name = $matches[2];
					
					$database_name = str_replace('`', '', $database_name);
					$database_name = str_replace("'", '', $database_name);
					$database_name = str_replace('.*', '', $database_name);
					$database_name = preg_replace('/@.*$/', '', $database_name);
					$database_name = str_replace('\_', '_', $database_name);
					
					if (($database_name == '*' || $database_name == DB_DATABASE) &&
							(strpos($privilege_string, 'ALL PRIVILEGES') !== false ||
							strpos($privilege_string, $privilege_type) !== false)) {
						// This grant gives the specified privilege for the Zen Cart database, no need to examine
						// any others
						$db_user_has_privilege = true;
						
						break 2;
					}
				}
			}
			
			$check_user_privileges_result->MoveNext();
		}
		
		return $db_user_has_privilege;
	}
	
	// }}}
}

// }}}
