<?php

include 'includes/application_top.php';

function decodeError($errno): array {
    // error_log("Error $errno encountered.");
    $errorMessage = "";
    define("UNKNOWN_ERROR_OFFSET", 1971);

    // a list of ADN response codes can be found here: https://developer.authorize.net/api/reference/dist/json/responseCodes.json
    switch ($errno)
    {
        case "0":
            $errorMessage = "Unknown error occurred. Please contact us for assistance.";
            break;
        case "E00003":
            $errorMessage = "Could not parse POST body. Please contact us for assistance.";
            break;
        case "E00118":
        case "2":
        case "191":
        case "288":
        case "314":
            $errorMessage = "The transaction has been declined.";
            break;
        case "3": 
            $errorMessage = "Your order was declined with a referral to voice authorization.";
            break;
        case "5":
        case "74":
        case "75":
        case "76":
        case "125":
            $errorMessage = "An error occurred. There was an invalid amount sent to ADN.";
            break;
        case "6":
        case "7":
        case "8":
        case "37":
        case "44":
            $errorMessage = "Invalid card information.";
            break;
        case "17":
        case "28":
            $errorMessage = "We do not accept this card type.";
            break;
        case "9":
        case "10":
        case "107":
        case "110":
            $errorMessage = "Invalid ACH information";
            break;
        case "53":
            $errorMessage = "Transaction type invalid for ACH";
            break;
        case "18":
        case "90":
            $errorMessage = "We do not accept ACH transactions at this time.";
            break;
        case "11":
            $errorMessage = "A duplicate transaction was detected. Please contact us for assistance.";
            break;
        case "E00005":
        case "E00006":
        case "E00007":
        case "E00008":
        case "E00010":
        case "E00011":
        case "13":
        case "32":
            $errorMessage = "An error occurred. Merchant Credentials Invalid. Contact us if this problem persists.";
            break;
        case "E00064":
            $errorMessage = "Client authorization failed. Please contact us for assistance.";
            break;
        case "I00011":
        case "114":
        case "115":
        case "130":
        case "131":
        case "132":
        case "901":
            $errorMessage = "An error occurred. Merchant account is not active. Contact us if this problem persists.";
            break;
        case "E00001":
        case "E00053":
        case "E00062":
        case "E00063":
        case "E00104":
        case "E00133":
        case "19":
        case "20":
        case "21":
        case "22":
        case "23":
        case "25":
        case "26":
        case "35":
        case "57":
        case "58":
        case "59":
        case "60":
        case "61":
        case "62":
        case "63":
        case "120":
        case "121":
        case "122":
        case "192":
            $errorMessage = "An error occurred during processing. Please try again or contact us if this problem persists.";
            break;
        case "E00027":
            $errorMessage = "The transaction was unsuccessful.";
        case "40":
        case "66":
        case "195":
        case "250":
        case "251":
        case "261":
            $errorMessage = "Transaction was cancelled due to security concerns. Contact us if this problem persists.";
            break;
        case "27":
        case "45":
        case "65":
        case "78":
        case "165":
        case "127":
            $errorMessage = "AVS/Card Code filter triggered. Your billing address or card security code (CVV) failed to match what your financial institution has on file.";
            break;
        case "E00129":
        case "52":
        case "152":
            $errorMessage = "Client could not be notified. This transaction may be declined. Please contact us for confirmation.";
            break;
        case "19711":
            $errorMessage = "Session has expired. Please make note of the items in your cart and try checking out again.";
            break;
        case "19712":
            $errorMessage = "Invalid POST body, could not perform request. (2)";
            break;
        case "19713":
            $errorMessage = "Invalid POST body, could not perform request. (3)";
            break;
        default:
            $errno = UNKNOWN_ERROR_OFFSET . $errno;
            $errorMessage = "An Unknown Error Occurred. ($errno)";
            break;
    }
    
    return array(
        "code" => 0,
        "message" => $errorMessage
    );
}

function clearSessionVariables() {
    unset($_SESSION['shipping']);
    unset($_SESSION['taxes']);
    unset($_SESSION['coupon']);
    unset($_SESSION['cart_calculated']);
}

function resetCart($orderID) {
    $_SESSION['cart']->reset(true);
    $_SESSION['new_oID'] = $orderID;
}

function sendOrderConfirmationMailer($orderID, $adnPostData){
    global $db;

    // generate products table
    $productsTableHtml = "";
    foreach ($_SESSION['cart']->get_products() as $product)
    {
        $productQuantity = $product['quantity'];
        $productName = $product['name'];
        $productModel = $product['model'];
        $productPrice = $product['price'] * $productQuantity;

        $productsTableHtml .= "<tr>";
        $productsTableHtml .= "<td>$productQuantity</td>";
        $productsTableHtml .= "<td>$productName</td>";
        $productsTableHtml .= "<td>$productModel</td>";
        $productsTableHtml .= "<td>$productPrice</td>";
        $productsTableHtml .= "</tr>";
    }

    // generate order totals
    $orderTotalsTableHtml = "";
    $orderTotalsQuery = $db->Execute("SELECT title, value FROM zen_orders_total WHERE orders_id = $orderID ORDER BY orders_total_id ASC");
    while (!$orderTotalsQuery->EOF)
    {
        $orderTotalsTitle = $orderTotalsQuery->fields['title'];
        $orderTotalsValue = number_format($orderTotalsQuery->fields['value'], 2);

        $orderTotalsTableHtml .= "<tr>";
        $orderTotalsTableHtml .= "<td width=100%;></td>";
        $orderTotalsTableHtml .= "<td>$orderTotalsTitle</td>";
        $orderTotalsTableHtml .= "<td>$</td>";
        $orderTotalsTableHtml .= "<td width=0.1%;>$orderTotalsValue</td>";
        $orderTotalsTableHtml .= "</tr>";

        $orderTotalsQuery->MoveNext();
    }
    $orderTotalsQuery = null;
    unset($orderTotalsQuery);

    // generate order comments table
    $orderCommentsTableHtml = "";
    $orderCommentsQuery = $db->Execute("SELECT osh.date_added, osh.comments, os.orders_status_name FROM zen_orders_status_history osh JOIN zen_orders_status os ON osh.orders_status_id = os.orders_status_id WHERE osh.orders_id = $orderID AND osh.customer_notified > 0 ORDER BY osh.orders_status_history_id ASC");
    while (!$orderCommentsQuery->EOF)
    {
        $commentDate = date("jS F, Y", strtotime($orderCommentsQuery->fields['date_added']));
        $commentMessage = $orderCommentsQuery->fields['comments'];
        $commentStatus = $orderCommentsQuery->fields['orders_status_name'];

        $orderCommentsTableHtml .= "<tr>";
        $orderCommentsTableHtml .= "<td>$commentDate</td>";
        $orderCommentsTableHtml .= "<td>$commentMessage</td>";
        $orderCommentsTableHtml .= "<td>$commentStatus</td>";
        $orderCommentsTableHtml .= "</tr>";

        $orderCommentsQuery->MoveNext();
    }
    $orderCommentsQuery = null;
    unset($orderCommentsQuery);


    // generate shipping and billing addresses
    $orderDetailsQuery = $db->Execute("
    SELECT 
    `delivery_name`,`delivery_company`,`delivery_street_address`,`delivery_suburb`,`delivery_city`,`delivery_postcode`,`delivery_state`,`delivery_country`,
    `billing_name`,`billing_company`,`billing_street_address`,`billing_suburb`,`billing_city`,`billing_postcode`,`billing_state`,`billing_country`,
    `payment_method`,`shipping_method`,`date_purchased` FROM `zen_orders` WHERE orders_id = $orderID
    ");

    $shippingDetailHtml = "
    <tr><td>{$orderDetailsQuery->fields['delivery_name']}</td></tr>
    " . (!empty($orderDetailsQuery->fields['delivery_company'])? "<tr><td>{$orderDetailsQuery->fields['delivery_company']}</td></tr>" : "") . "
    <tr><td>{$orderDetailsQuery->fields['delivery_street_address']}</td></tr>
    " . (!empty($orderDetailsQuery->fields['delivery_suburb'])? "<tr><td>{$orderDetailsQuery->fields['delivery_suburb']}</td></tr>" : "") . "
    <tr><td>{$orderDetailsQuery->fields['delivery_city']}, {$orderDetailsQuery->fields['delivery_state']}</td></tr>
    <tr><td>{$orderDetailsQuery->fields['delivery_postcode']}, {$orderDetailsQuery->fields['delivery_country']}</td></tr>
    ";

    $billingDetailHtml = "<tr><td>{$orderDetailsQuery->fields['billing_name']}</td></tr>
    " . (!empty($orderDetailsQuery->fields['billing_company'])? "<tr><td>{$orderDetailsQuery->fields['billing_company']}</td></tr>" : "") . "
    <tr><td>{$orderDetailsQuery->fields['billing_street_address']}</td></tr>
    " . (!empty($orderDetailsQuery->fields['billing_suburb'])? "<tr><td>{$orderDetailsQuery->fields['billing_suburb']}</td></tr>" : "") . "
    <tr><td>{$orderDetailsQuery->fields['billing_city']}, {$orderDetailsQuery->fields['billing_state']}</td></tr>
    <tr><td>{$orderDetailsQuery->fields['billing_postcode']}, {$orderDetailsQuery->fields['billing_country']}</td></tr>
    ";
    $orderAddresseQuery = null;
    unset($orderAddresseQuery);

    // generate reset password token
    $resetPasswordToken = md5($_SESSION['customers_email_address'] . date("Ymdhis"));

    // prep mailer content to be sent
    $mailerContent = array(
        "BASE_HREF" => "https://westechequipment.com",
        "EMAIL_COMMON_CSS" => "FGC:./email_common.css",
        "EMAIL_LOGO_FILE" => "/images/westech-logo.png",
        "EMAIL_LOGO_ALT_TEXT" => "Westech Equipment",
        "EXTRA_HEADER_INFO" => "",
        "EMAIL_TEXT_HEADER" => "Thank you for your purchase.",
        "EMAIL_FIRST_NAME" => "",
        "EMAIL_LAST_NAME" => "{$orderDetailsQuery->fields['delivery_name']}",
        "EMAIL_THANKS_FOR_SHOPPING" => "Thank you for shopping at Westech Equipment.",
        "EMAIL_DETAILS_FOLLOW" => "",
        "INTRO_ORDER_NUM_TITLE" => "Order Number #",
        "INTRO_ORDER_NUMBER" => "$orderID",
        "INTRO_DATE_TITLE" => "Ordered On",
        "INTRO_DATE_ORDERED" => date("jS F, Y", strtotime($orderDetailsQuery->fields['date_purchased'])),
        "INTRO_URL_VALUE" => "?main_page=checkout_confirmation&oID=$orderID",
        "INTRO_URL_TEXT" => "View Your Order",
        "PRODUCTS_TITLE" => "Products Ordered",
        "PRODUCTS_DETAIL" => "
            <table>
            <tr>
            <th>QTY</th>
            <th>Name</th>
            <th>Model</th>
            <th>Price</th>
            </tr>

            $productsTableHtml
            </table>
            ",
            "ORDER_TOTALS" => "
            <table>
            $orderTotalsTableHtml
            </table>
            ",
            "ORDER_COMMENTS" => "
            <table>
            <tr>
            <th>Date</th>
            <th>Comment</th>
            <th>Status</th>
            </tr>

            $orderCommentsTableHtml
            </table>
            ",
            "HEADING_ADDRESS_INFORMATION" => "",
            "ADDRESS_DELIVERY_TITLE" => "Shipping Address",
            "ADDRESS_DELIVERY_DETAIL" => "
            <table>
            $shippingDetailHtml
            </table>",
        "SHIPPING_METHOD_TITLE" => "Shipping Method",
        "SHIPPING_METHOD_DETAIL" => $orderDetailsQuery->fields['shipping_method'],
        "ADDRESS_BILLING_TITLE" => "Billing Address",
        "ADDRESS_BILLING_DETAIL" => "
            <table>
            $billingDetailHtml
            </table>",
        "PAYMENT_METHOD_TITLE" => "Payment Method",
        "PAYMENT_METHOD_DETAIL" => $adnPostData['type'],
        "PAYMENT_METHOD_FOOTER" => ($adnPostData['type'] == "guest")? "To set your password and convert your guest account to a full account, you can <a href='?main_page=forgot_password&token=$resetPasswordToken'>visit this link</a>. A full customer account can view past orders, upload and apply tax exemptions, and receive greater technical support." : "",
        "EMAIL_FOOTER_COPYRIGHT" => "&#169; " . date("Y") . " Westech Equipment",
        "EMAIL_DISCLAIMER" => "By placing an order you agreed to our <a href='?main_page=terms_and_conditions'>terms and conditions</a>.",
        "EMAIL_SPAM_DISCLAIMER" => "We comply with the CAN-SPAM Act. You can unsubscribe from future emails <a href='?main_page=unsubscribe&token={$_SESSION['customers_email_address']}'>here</a>.",
        "EXTRA_INFO" => ""
    );
    $orderDetailsQuery = null;
    unset($orderDetailsQuery);
    
    if ($adnPostData['type'] == "guest")
    {
        $updateCustomerSecretQuery = $db->Execute("UPDATE zen_customers SET customers_secret = '$resetPasswordToken' WHERE customers_email_address = '{$_SESSION['customers_email_address']}'");
    }
    $updateCustomerSecretQuery = null;
    unset($updateCustomerSecretQuery);

    // send mailer to customer
    zen_mail(($_SESSION['customer_first_name'] . ' ' . $_SESSION['customer_last_name']), $_SESSION['customers_email_address'], "Order $orderID at Westech Equipment", "Thank you for your purchase.", "Westech Equipment", "contact@westechequipment.com", $mailerContent, "checkout");

    // send same mailer to everyone listed in the zen configuration "SEND_EXTRA_ORDER_EMAILS_TO" as well
    $ccSendExtraOrderEmailToQuery = $db->Execute("SELECT configuration_value FROM zen_configuration WHERE configuration_key = 'SEND_EXTRA_ORDER_EMAILS_TO'");
    foreach (explode(', ', $ccSendExtraOrderEmailToQuery->fields['configuration_value']) as $email)
    {
        zen_mail(($_SESSION['customer_first_name'] . ' ' . $_SESSION['customer_last_name']), $email, "Order $orderID at Westech Equipment", "Thank you for your purchase.", "Westech Equipment", "contact@westechequipment.com", $mailerContent, "checkout");
    }
    $ccSendExtraOrderEmailToQuery = null;
    unset($ccSendExtraOrderEmailToQuery);
}

function lockTotalsTable() {
    global $db;

    $db->Execute("LOCK TABLES zen_orders_total WRITE");
}

function lockHistoryTable() {
    global $db;

    $db->Execute("LOCK TABLES zen_orders_status_history WRITE");
}

function lockProductsTable() {
    global $db;

    $db->Execute("LOCK TABLES zen_orders_products WRITE");
}

function postOrder($cID, $shipping, $billing, $phoneNumber, $email, $cardData, $orderID, $ADNresult, $comments): bool {
    global $db;

    try 
    {
        // Update order setup
        $updateOrderQuery = "UPDATE `zen_orders`
        SET
        `customers_id` = :cID:, `customers_name` = :cName:, `customers_company` = :cCompany:, `customers_street_address` = :cAdd1:,
        `customers_suburb` = :cAdd2:, `customers_city` = :cCity:, `customers_postcode` = :cZip:, `customers_state` = :cState:,
        `customers_country` = :cCountry:, `customers_telephone` = :cPhone:, `customers_email_address` = :cEmail:, `delivery_name` = :cShipName:,
        `delivery_company` = :cShipCompany:, `delivery_street_address` = :cShipAdd1:, `delivery_suburb` = :cShipAdd2:, `delivery_city` = :cShipCity:,
        `delivery_postcode` = :cShipZip:, `delivery_state` = :cShipState:, `delivery_country` = :cShipCountry:, `billing_name` = :cBillName:,
        `billing_company` = :cBillCompany:, `billing_street_address` = :cBillAdd1:, `billing_suburb` = :cBillAdd2:, `billing_city` = :cBillCity:,
        `billing_postcode` = :cBillZip:, `billing_state` = :cBillState:, `billing_country` = :cBillCountry:, `payment_method` = :cPayMethod:,
        `payment_module_code` = 'ADN', `shipping_method` = :cShipMethod:, `shipping_module_code` = 'UPS_CUSTOM', `coupon_code` = :coupon_code:,
        `last_modified` = now(), `date_purchased` = now(), `orders_status` = 1, `currency` = 'USD',
        `currency_value` = '1', `order_total` = :order_total:, `order_tax` = :order_tax:, `paypal_ipn_id` = :adn_txid:,
        `ip_address` = '', `language_code` = 'en', `order_weight` = :oWeight:
        WHERE `orders_id` = $orderID";
        
        $updateOrderQueryValues = array(
            "cID" => $cID,
            "cName" => $shipping['firstName'] . ' ' . $shipping['lastName'],
            "cCompany" => $shipping['company'],
            "cAdd1" => $shipping['address'],
            "cAdd2" => $shipping['suburb'],
            "cCity" => $shipping['city'],
            "cZip" => $shipping['zip'],
            "cState" => $shipping['state'],
            "cCountry" => $shipping['country'],
            "cPhone" => $phoneNumber,
            "cEmail" => $email,
            "cShipName" => $shipping['firstName'] . ' ' . $shipping['lastName'],
            "cShipCompany" => $shipping['company'],
            "cShipAdd1" => $shipping['address'],
            "cShipAdd2" => $shipping['suburb'],
            "cShipCity" => $shipping['city'],
            "cShipZip" => $shipping['zip'],
            "cShipState" => $shipping['state'],
            "cShipCountry" => $shipping['country'],
            "cBillName" => $billing['firstName'] . ' ' . $billing['lastName'],
            "cBillCompany" => $billing['company'],
            "cBillAdd1" => $billing['address'],
            "cBillAdd2" => $billing['suburb'],
            "cBillCity" => $billing['city'],
            "cBillZip" => $billing['zip'],
            "cBillState" => $billing['state'],
            "cBillCountry" => $billing['country'],
            "cPayMethod" => $cardData['type'],
            "cShipMethod" => $_SESSION['shipping']['values']['method']?? 'Unknown',
            "coupon_code" => $_SESSION['coupon']['code']?? '',
            "order_total" => $_SESSION['cart_calculated']['total'],
            "order_tax" => $_SESSION['cart_calculated']['taxes'],
            "adn_txid" => $ADNresult['txID'],
            "oWeight" => $_SESSION['cart']->weight
        );
        $updateOrderQuery = foreachBindVars($updateOrderQueryValues, $updateOrderQuery);

        // update order
        $db->Execute($updateOrderQuery);
    
        // cleanup
        unset($updateOrderQuery, $updateOrderQueryValues);
    
        // insert order products
        $orderProducts = array();
    
        // insert order products preperation
        foreach ($_SESSION['cart']->get_products() as $orderProduct)
        {
            $productsIdPrepared = $orderProduct['id'];
            if (strpos($orderProduct['id'], ":") > -1)
            {
                $productsIdPrepared = strtok($orderProduct['id'], ":");
            }
    
            $orderProducts[$productsIdPrepared] = array(
                "pModel" => $orderProduct['model'],
                "pName" => $orderProduct['name'],
                "pPrice" => $orderProduct['price'],
                "pFinalPrice" => $orderProduct['final_price'],
                "pQuantity" => $orderProduct['quantity'],
                "pPrid" => $orderProduct['id'],
                "pWeight" => $orderProduct['weight'],
                "pFreeShip" => $orderProducts['product_is_always_free_shipping']
            );
        }
    
        // prepare order products insert statement
        foreach ($orderProducts as $prid => $pdata)
        {
            $insertOrderProductsQuery = "INSERT INTO `zen_orders_products`
            (`orders_id`, products_id, products_model, products_name,
            products_price, final_price, products_quantity, products_prid,
            products_weight, product_is_always_free_shipping)
            VALUES
            (:oID:, :pID:, :pModel:, :pName:,
            :pPrice:, :pFinalPrice:, :pQuantity:, :pPrid:,
            :pWeight:, :pFreeShip:)";
    
            $insertOrderProductsQueryValues = array(
                "oID" => $orderID,
                "pID" => $prid,
                "pModel" => $pdata['pModel'],
                "pName" => $pdata['pName'],
                "pPrice" => $pdata['pPrice'],
                "pFinalPrice" => $pdata['pFinalPrice'],
                "pQuantity" => $pdata['pQuantity'],
                "pPrid" => $pdata['pPrid'],
                "pWeight" => $pdata['pWeight'],
                "pFreeShip" => $pdata['pFreeShip']
            );
            $insertOrderProductsQuery = foreachBindVars($insertOrderProductsQueryValues, $insertOrderProductsQuery);
    
            lockProductsTable();
            // insert order products
            $db->Execute($insertOrderProductsQuery);

            $insertOrderProductsIDQuery = $db->Execute("SELECT orders_products_id FROM zen_orders_products ORDER BY orders_products_id DESC LIMIT 1");
            $insertOrderProductsID = $insertOrderProductsIDQuery->fields['orders_products_id'];
    
            unlockTables();
    
            unset($insertOrderProductsQuery, $insertOrderProductsQueryValues);
    
            // prepare product attributes
            if (is_array($pdata['attributes']))
            {
                // prepare product attributes query
                foreach ($pdata['attributes'] as $attribute => $value)
                {
                    $selectAttributesAndOptionsQuery = "SELECT 
                    a.options_values_price AS ovp, a.price_prefix AS pp, a.product_attribute_is_free AS paif, a.products_attributes_weight AS paw,
                    a.products_attributes_weight_prefix AS pawp, a.attributes_discounted AS ad, a.attributes_price_base_included AS apbi, a.attributes_price_onetime AS apo,
                    a.attributes_price_factor AS apf, a.attributes_price_factor_offset AS apfo, a.attributes_price_factor_onetime AS apfot, a.attributes_price_factor_onetime_offset AS apfoo,
                    o.products_options_name AS pon,
                    v.products_options_values_name AS povn
                    FROM zen_products_attributes AS a,
                    zen_products_options AS o, 
                    zen_products_options_values AS v 
                    WHERE a.products_id = '$prid' AND a.options_id = '$attribute' AND a.options_values_id = '$value' 
                    AND o.products_options_id = '$attribute' 
                    AND v.products_options_values_id = '$value'
                    ";
    
                    $selectAttributesAndOptionsQueryResult = $db->Execute($selectAttributesAndOptionsQuery);
    
                    $insertProductAttributesQuery = "INSERT INTO `zen_orders_products_attributes`
                    (`orders_id`, `orders_products_id`, `products_options`, `products_options_values`,
                    `options_values_price`, `price_prefix`, `product_attribute_is_free`, `products_attributes_weight`,
                    `products_attributes_weight_prefix`, `attributes_discounted`, `attributes_price_base_included`, `attributes_price_onetime`,
                    `attributes_price_factor`, `attributes_price_factor_offset`, `attributes_price_factor_onetime`, `attributes_price_factor_onetime_offset`,
                    `attributes_qty_prices`, `attributes_qty_prices_onetime`, `attributes_price_words`, `attributes_price_words_free`,
                    `attributes_price_letters`, `attributes_price_letters_free`, `products_options_id`, `products_options_values_id`,
                    `products_prid`) 
                    VALUES 
                    (:orders_id:, :orders_products_id:, :products_options:, :products_options_values:,
                    :options_values_price:, :price_prefix:, :product_attribute_is_free:, :products_attributes_weight:,
                    :products_attributes_weight_prefix:, :attributes_discounted:, :attributes_price_base_included:, :attributes_price_onetime:,
                    :attributes_price_factor:, :attributes_price_factor_offset:, :attributes_price_factor_onetime:, :attributes_price_factor_onetime_offset:,
                    :attributes_qty_prices:, :attributes_qty_prices_onetime:, :attributes_price_words:, :attributes_price_words_free:,
                    :attributes_price_letters:, :attributes_price_letters_free:, :products_options_id:, :products_options_values_id:,
                    :products_prid:);
                    ";
    
                    $insertProductAttributesQueryValues = array(
                        "orders_id" => $orderID,
                        "orders_products_id" => $insertOrderProductsID,
                        "products_options" => $selectAttributesAndOptionsQueryResult->fields['pon'],
                        "products_options_values" => $selectAttributesAndOptionsQueryResult->fields['povn'],
                        "options_values_price" => $selectAttributesAndOptionsQueryResult->fields['ovp'],
                        "price_prefix" => $selectAttributesAndOptionsQueryResult->fields['pp'],
                        "product_attribute_is_free" => $selectAttributesAndOptionsQueryResult->fields['paif'],
                        "products_attributes_weight" => $selectAttributesAndOptionsQueryResult->fields['paw'],
                        "products_attributes_weight_prefix" => $selectAttributesAndOptionsQueryResult->fields['pawp'],
                        "attributes_discounted" => $selectAttributesAndOptionsQueryResult->fields['ad'],
                        "attributes_price_base_included" => $selectAttributesAndOptionsQueryResult->fields['apbi'],
                        "attributes_price_onetime" => $selectAttributesAndOptionsQueryResult->fields['apo'],
                        "attributes_price_factor" => $selectAttributesAndOptionsQueryResult->fields['apf'],
                        "attributes_price_factor_offset" => $selectAttributesAndOptionsQueryResult->fields['apfo'],
                        "attributes_price_factor_onetime" => $selectAttributesAndOptionsQueryResult->fields['apfot'],
                        "attributes_price_factor_onetime_offset" => $selectAttributesAndOptionsQueryResult->fields['apfoo'],
                        "attributes_qty_prices" => '',
                        "attributes_qty_prices_onetime" => '',
                        "attributes_price_words" => 0.0,
                        "attributes_price_words_free" => 0,
                        "attributes_price_letters" => 0.0,
                        "attributes_price_letters_free" => 0, 
                        "products_options_id" => $attribute,
                        "products_options_values_id" => $value,
                        "products_prid" => $pdata['pPrid']
                    );
                    $insertProductAttributesQuery = foreachBindVars($insertProductAttributesQueryValues, $insertProductAttributesQuery);
                    
                    // insert product attributes
                    $db->Execute($insertProductAttributesQuery);
    
                    unset($selectAttributesAndOptionsQuery, $selectAttributesAndOptionsQueryResult, $insertProductAttributesQuery, $insertProductAttributesQueryValues);
                }
            }
        }
    
        // prepare history events to be inserted into comments
        $adnAuth = $ADNresult['authCode'];
        $adnTxID = $ADNresult['txID'];
        $adnMajor = $ADNresult['major'];
        $adnMinor = $ADNresult['minor'];
        $historyEvents = array(
            "($orderID, 1, now(), -1, 'Transaction: [ Auth: $adnAuth ID: $adnTxID C: $adnMajor:$adnMinor ]', 'ADN System')"
        );
    
        // prepare comments to be inserted into order status history
        if (!empty($comments))
        {
            $historyPrepare = ",
            ($orderID, 1, now(), -1, :comments:, 'User')";
            $historyPrepare = $db->bindVars($historyPrepare, ":comments:", $comments, 'string');
            
            // concatenate user comments as historyPrepare to historyEvents
            $historyEvents[] = $historyPrepare;
        }

        // prepare insert history statement
        $insertHistoryQuery = "INSERT INTO `zen_orders_status_history`
        (orders_id, orders_status_id, date_added, customer_notified, comments, updated_by)
        VALUES
        ";

        foreach ($historyEvents as $h)
        {
            $insertHistoryQuery .= "$h
            ";
        }

        lockHistoryTable();
        
        // insert history events
        $db->Execute($insertHistoryQuery);
    
        unlockTables();

        unset($insertHistoryQuery);
        
        // generate totals
        $subtotal = $_SESSION['cart_calculated']['subtotal'];
        $shipping = $_SESSION['cart_calculated']['shipping'];
        $taxes = $_SESSION['cart_calculated']['taxes'];
        $total = $_SESSION['cart_calculated']['total'];
        
        $subtotalFromatted = number_format($subtotal, 2);
        $shippingFormatted = number_format($shipping, 2);
        $taxesFormatted = number_format($taxes, 2);
        $totalFormatted = number_format($total, 2);
    
        // prepare totals insertion
        $insertOrderTotalsQuery = "INSERT INTO `zen_orders_total`
        (`orders_id`, `title`, `text`, `value`, `class`, `sort_order`)
        VALUES
        ('$orderID', 'Sub-Total:', '$$subtotalFromatted', '$subtotal', 'ot_subtotal', '100'),
        ('$orderID', 'Shipping: ', '$$shippingFormatted', '$shipping', 'ot_upsc', '300'),
        ";

        $lowOrderThreshold = 50.00;
        $lowOrderFee = 20.00;
    
        // generate low-order-fee line item
        if ($subtotal < $lowOrderThreshold)
        {
            $insertOrderTotalsQuery .= "('$orderID', 'Low Order Fee:', '$$lowOrderFee', '$lowOrderFee', 'ot_loworderfee', '400'),
            ";
        }
    
        // generate coupon line item
        if (isset($_SESSION['coupon']['code']))
        {
            $code = $_SESSION['coupon']['code'];
            $amount = $_SESSION['coupon']['amount'];
            $type = $_SESSION['coupon']['type'];
    
            $couponFormatted = number_format($amount, 2);
            
            if ($type == "P")
            {
                $amount *= $subtotal;
                $couponFormatted = number_format(($amount), 2);
            }
            
            $insertOrderTotalsQuery .= "
            ('$orderID', 'Coupon: $code', '-$$couponFormatted', '-$amount', 'ot_coupon', '600'),
            ";
        }
    
        // generate tax line item
        $insertOrderTotalsQuery .= "
        ('$orderID', 'Tax:', '$$taxesFormatted', '$taxes', 'ot_avatax', '700'),";
    
        // generate total line item
        $insertOrderTotalsQuery .= "
        ($orderID, 'Total:', '$$totalFormatted', '$total', 'ot_total', '999')
        ";
    
        lockTotalsTable();
        
        // insert totals line items
        $db->Execute($insertOrderTotalsQuery);
    
        unlockTables();

        unset($subtotal, $shipping, $taxes, $total, $subtotalFromatted, $shippingFormatted, $taxesFormatted, $totalFormatted, $insertOrderTotalsQuery);
    }
    catch (Exception $e)
    {
        return false;
    }

    return true;
}

function lockCustomerTable() {
    global $db;

    $db->Execute("LOCK TABLES zen_customers WRITE");
}

function createPsuedoAccount($guestDetails): int {
    global $db;

    $checkForExistingPseudoAccountQuery = "SELECT `customers_id` FROM `zen_customers` WHERE `customers_email_address` = :cEA:";

    $checkForExistingPseudoAccountQuery = $db->bindVars($checkForExistingPseudoAccountQuery, ":cEA:", $guestDetails['customers_email_address'], 'string');
    $checkForExistingPseudoAccountQueryResult = $db->Execute($checkForExistingPseudoAccountQuery);

    if ($checkForExistingPseudoAccountQueryResult->RecordCount() >= 1)
    {
        $secret = md5($guestDetails['customers_email_address'] . date("Ymdhis"));

        $cID = $checkForExistingPseudoAccountQueryResult->fields['customers_id'];

        $updatePsuedoAccountSecretQuery = "UPDATE `zen_customers` SET `customers_secret` = :cS: WHERE `customers_id` = :cID:";

        $updatePsuedoAccountSecretQueryValues = array(
            "cS" => $secret,
            "cID" => $cID
        );

        $updatePsuedoAccountSecretQuery = foreachBindVars($updatePsuedoAccountSecretQueryValues, $updatePsuedoAccountSecretQuery);

        $db->Execute($updatePsuedoAccountSecretQuery);

        return ($cID);
    }

    $hashpass = password_hash(date("Ymdhis"), PASSWORD_DEFAULT);
    $secret = md5($_SESSION['customers_email_address'] . date("Ymdhis"));

    lockCustomerTable();

    $insertCustomerQuery = "INSERT INTO `zen_customers` (`customers_firstname`, `customers_lastname`, `customers_email_address`, `customers_telephone`,
    `customers_password`, `customers_secret`)
    VALUES
    (:cfn:, :cln:, :cea:, :ctp:, :cpw:, :cs:)";

    $insertCustomerQueryValues = array(
        "cfn" => $guestDetails['customers_firstname'],
        "cln" => $guestDetails['customers_lastname'],
        "cea" => $guestDetails['customers_email_address'],
        "ctp" => $guestDetails['customers_telephone'],
        "cpw" => $hashpass,
        "cs" => $secret
    );

    $insertCustomerQuery = foreachBindVars($insertCustomerQueryValues, $insertCustomerQuery);

    $db->Execute($insertCustomerQuery);

    $cIDQuery = $db->Execute("SELECT customers_id FROM zen_customers ORDER BY customers_id DESC LIMIT 1");
    $cID = $cIDQuery->fields['customers_id'];

    unlockTables();

    $insertCustomerInfoQuery = "INSERT INTO `zen_customers_info` 
    (`customers_info_id`, `customers_info_date_account_created`)
    VALUES 
    (:cID:, now())";

    $insertCustomerInfoQuery = $db->bindVars($insertCustomerInfoQuery, ":cID:", $cID, 'integer');

    $db->Execute($insertCustomerInfoQuery);

    $_SESSION['customers_email_address'] = $guestDetails['customers_email_address'];

    return $cID;
}

function prepareCustomerID($email, $oID, $ADNresult, $guestDetails): int {
    global $db;

    // if customer id is not set, set it
    $cID = ((empty($guestDetails))? $_SESSION['customer_id'] : createPsuedoAccount($guestDetails));

    return $cID;
}

function processOrderThroughADN($customerID, $orderID, $cardData, $save, $shipping, $billing): array {
    global $db;
    $auth = adn_authorize_card($orderID, $cardData, $save, $shipping, $billing);

    $authReq = $auth['debug'];
    $authResp = $auth['auth'];

    $adnSent = print_r($authReq, true);
    $adnReceived = print_r($authResp, true);

    $adnInsertQuery = "INSERT INTO `zen_apid_authorizenet`
    (`customer_id`, `order_id`, `transaction_id`, `sent`,
    `received`, `time`, `session_id`)
    VALUES
    (:cID:, :oID:, :adnTXID:, :adnSent:,
    :adnReceived:, now(), :sessionID:)";

    $adnInsertQueryValues = array(
        "cID" => $customerID,
        "oID" => $orderID,
        "adnTXID" => $authResp['transactionResponse']['transId'],
        "adnSent" => $adnSent,
        "adnReceived" => $adnReceived,
        "sessionID" => session_id()
    );

    $adnInsertQuery = foreachBindVars($adnInsertQueryValues, $adnInsertQuery);

    $db->Execute($adnInsertQuery);

    if (!is_null(findValueInArray($auth, "responseCode")))
    {
        $responseCode = findValueInArray($auth, "responseCode");

        if (in_array($responseCode, array("1", "4")))
        {
            $auth = $authResp['transactionResponse'];

            return array(
                "code" => 1,
                "sent" => $authReq,
                "received" => $authResp,
                "authCode" => $auth['authCode'],
                "txID" => $auth['transId'],
                "major" => $responseCode,
                "minor" => findValueInArray($auth, "code")
            );
        }
    }

    if (!is_null(findValueInArray($auth, "errors")) || !is_null(findValueInArray($auth, "error")) || !is_null(findValueInArray($auth, "errorCode")) || !is_null(findValueInArray($auth, "errorText")))
    {
        if (!is_null(findValueInArray($auth, "errorCode")))
        {
            return decodeError(findValueInArray($auth, "errorCode"));
        }

        if (!is_null(findValueInArray($auth, "error")))
        {
            return array(
                "code" => 0,
                "message" => "An error occurred but no error code was found. Please contact us for assistance."
            );
        }

        if (!is_null(findValueInArray($auth, "errors")))
        {
            return array(
                "code" => 0,
                "message" => "One or more unknown errors occurred. Please contact us for assistance."
            );
        }

        return array(
            "code" => 0,
            "message" => "An unknown error occurred. Please contact us immediately for assistance. (0)"
        );
    }
    if (findValueInArray($auth, "resultCode") == "Error")
    {
        return decodeError(findValueInArray($auth, "code"));
    }
    $codeKeys = findAllValuesInArray($auth, "code");
    foreach ($codeKeys as $code)
    {
        if (strpos($code, "E") !== FALSE)
        {
            return decodeError($code);
        }
    }

    return array(
        "code" => 0,
        "message" => "An unknown error occurred. Please contact us immediately for assistance. (1)"
    );
}

function unlockTables() {
    global $db;

    $db->Execute("UNLOCK TABLES");
}

function reserveOrderId(): int {
    global $db;

    // this guarantees that each order has a unique order ID, regardless of collissions.
    $db->Execute("INSERT INTO zen_orders VALUES()");
    $lid = $db->Execute("SELECT orders_id FROM zen_orders ORDER BY orders_id DESC LIMIT 1");

    return $lid->fields['orders_id'];
}

function lockOrderTable() {
    global $db;

    $db->Execute("LOCK TABLES zen_orders WRITE");
}

function checkSession(): bool {
    // checks session variables to see if they were unset, if so, the session is bad
    return count($_SESSION['cart']->get_products()) > 0 && isset($_SESSION['cart'], $_SESSION['shipping'], $_SESSION['taxes']);
}

// default value of status, in case the order fails to process it can be returned, upon success, it shouldn't ever be this, since it's set elsewhere.
$status = array(
    "code" => 0,
    "message" => "We couldn't successfully process your order. Please contact us for assistance."
);

$cID = "";
$orderID = "";
$ADNresult = array();

try
{
    if (!checkSession()) throw new Exception(19711);

    //sanitize input
    $email = isset($_SESSION['customers_email_address'])? $_SESSION['customers_email_address'] : filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
    $cardData = $_POST['cardData'];

    $cardData['type'] = htmlspecialchars($cardData['type']);

    if (!in_array($cardData['type'], array("guest", "new", "adn-saved")))
    {
        throw new Exception(19712);
    }

    if (in_array($cardData['type'], array("guest", "new")))
    {
        if (preg_match("/[^0-9]/", $cardData['exp_month'])) throw new Exception(19712);
        if (preg_match("/[^0-9]/", $cardData['exp_year'])) throw new Exception(19712);
        if (preg_match("/[^0-9]/", $cardData['cardCode'])) throw new Exception(19712);
        $cardData['expiration'] = "{$cardData['exp_year']}-{$cardData['exp_month']}";
        //remove all characters from card number that aren't digits
        $cardData['cardNumber'] = preg_replace("/[^0-9]/", "", $cardData['cardNumber']);
        $cardData['cardCode'] = preg_replace("/[^0-9]/", "", $cardData['cardCode']);
        $cardData['name'] = htmlspecialchars($cardData['name']);
    }
    else if ($cardData['type'] == "adn-saved")
    {
        $cardData['id'] = htmlspecialchars($cardData['id']);
        $adnProfile = adn_get_profile($_SESSION['customer_id']);
        $paymentProfileIDs = findAllValuesInArray($adnProfile, "customerPaymentProfileId");
        if (is_null($paymentProfileIDs)) throw new Exception(19712);
        if (!in_array($cardData['id'], $paymentProfileIDs))
        {
            error_log(print_r($adnProfile, true));
            error_log($cardData['id']);
            error_log(print_r($paymentProfileIDs, true));

            throw new Exception(19713);
        }
    }

    $shippingCompany = htmlspecialchars($_POST['shippingCompany']);
    $shippingFirstName = htmlspecialchars($_POST['shippingFirstName']);
    $shippingLastName = htmlspecialchars($_POST['shippingLastName']);
    $shippingStreetAddressOne = htmlspecialchars($_POST['shippingStreetAddressOne']);
    $shippingStreetAddressTwo = htmlspecialchars($_POST['shippingStreetAddressTwo']);
    $shippingStreetAddressThree = htmlspecialchars($_POST['shippingStreetAddressThree']);
    $shippingCity = htmlspecialchars($_POST['shippingCity']);
    $shippingState = htmlspecialchars($_POST['shippingState']);
    $shippingZip = htmlspecialchars($_POST['shippingZip']);
    $shippingCountry = htmlspecialchars($_POST['shippingCountry']);
    $phoneNumber = htmlspecialchars($_POST['phoneNumber']);
    $billingCompany = htmlspecialchars($_POST['billingCompany']);
    $billingFirstName = htmlspecialchars($_POST['billingFirstName']);
    $billingLastName = htmlspecialchars($_POST['billingLastName']);
    $billingStreetAddressOne = htmlspecialchars($_POST['billingStreetAddressOne']);
    $billingStreetAddressTwo = htmlspecialchars($_POST['billingStreetAddressTwo']);
    $billingStreetAddressThree = htmlspecialchars($_POST['billingStreetAddressThree']);
    $billingCity = htmlspecialchars($_POST['billingCity']);
    $billingState = htmlspecialchars($_POST['billingState']);
    $billingZip = htmlspecialchars($_POST['billingZip']);
    $billingCountry = htmlspecialchars($_POST['billingCountry']);
    $comments = htmlspecialchars($_POST['comments']);

    $shipping = array(
        "company" => $shippingCompany,
        "firstName" => $shippingFirstName,
        "lastName" => $shippingLastName,
        "address" => $shippingStreetAddressOne,
        "suburb" => $shippingStreetAddressTwo,
        "city" => $shippingCity,
        "state" => $shippingState,
        "zip" => $shippingZip,
        "country" => $shippingCountry
    );

    $billing = array(
        "company" => $billingCompany,
        "firstName" => $billingFirstName,
        "lastName" => $billingLastName,
        "address" => $billingStreetAddressOne,
        "suburb" => $billingStreetAddressTwo,
        "city" => $billingCity,
        "state" => $billingState,
        "zip" => $billingZip,
        "country" => $billingCountry
    );

    $guestDetails = array();

    if ($cardData['type'] == "guest")
    {
        $guestDetails = array(
            "customers_firstname" => $shippingFirstName,
            "customers_lastname" => $shippingLastName,
            "customers_email_address" => $email,
            "customers_telephone" => $phoneNumber,
        );
    }

    //set customer id depending on logged state
    // $cID = prepareCustomerID($email, $orderID, $ADNresult, $guestDetails);

    // if (!empty($guestDetails))
    // {
    //     $_SESSION['temp_customer_id'] = $cID;
    // }

    $cID = $_SESSION['customer_id'] ?? $_SESSION['temp_customer_id'];

    // lock table for critical section
    lockOrderTable();
    
    //start transaction so we can rollback if an error occurs
    $db->Execute("START TRANSACTION");

    //reserve order id
    $orderID = reserveOrderId();

    // release lock so other orders can process 
    unlockTables();

    //send processed order vars to ADN API
    $ADNresult = processOrderThroughADN($cID, $orderID, $cardData, $cardData['save'], $shipping, $billing);

    if ($ADNresult['code'] == 0)
    {
        $status = $ADNresult;
        throw new Exception("ignoreThrowable");
    }

    //update order WHERE order_id = reserved order id
    $processOrder = postOrder($cID, $shipping, $billing, $phoneNumber, $email, $cardData, $orderID, $ADNresult, $comments);

    //commit successful process
    $db->Execute("COMMIT");

    // send order confirmation mailer to customer and anyone else necessary
    sendOrderConfirmationMailer($orderID, $cardData);

    $status = array(
        "code" => 1,
        "majorCode" => $ADNresult['major'],
        "minorCode" => $ADNresult['minor'],
        "message" => "This transaction was successful.",
    );
}
catch (Exception $e)
{
    if ($e->getMessage() !== "ignoreThrowable") $status = decodeError($e->getMessage());
}
finally
{
    // cleanup
    $db->Execute("ROLLBACK");

    if ($status['code'] == 0 && !empty($orderID))
    {
        $db->Execute("DELETE FROM `zen_orders` WHERE `orders_id` = $orderID");
    }
    else
    {
        resetCart($orderID);
        clearSessionVariables();
    }

    unlockTables();
}

echo json_encode($status);