<?php

/**
 * Ceon URI Mapping - UMM Edition - Category URI Mappings Admin Functionality.
 *
 * This file contains a class with the methods necessary to handle URI mappings for categories.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingAdminCategories.php 1027 2012-07-17 20:31:10Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminCategoriesProducts.php');


// {{{ constants

define('CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_WITH_NO_NAME', -2);

// }}}


// {{{ CeonURIMappingAdminCategories

/**
 * Handles the URI mappings for categories.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2008 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingAdminCategories extends CeonURIMappingAdminCategoriesProducts
{
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingAdminCategories class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ autogenCategoryURIMapping()
	
	/**
	 * Generates a URI for a category, for the specified language. Doesn't check the format of the mapping, only
	 * the format of the individual placement tags it replaces in the template, so the actual URI should be
	 * cleaned/validated after this method has returned a URI.
	 *
	 * @access  public
	 * @param   integer   $id              The ID of the category.
	 * @param   integer   $parent_category_id   The ID of the parent category (used if the details in the database
	 *                                          could be out of date as new information is being submitted when the
	 *                                          URI is being generated).
	 * @param   string    $name            The name of category (used if new information is being submitted when
	 *                                     the URI is being generated).
	 * @param   string    $language_code   The ISO 639 language code of the language.
	 * @param   integer   $language_id     The Zen Cart language ID for the language.
	 * @param   string    $mapping_template   The mapping template for this category.
	 * @return  string    The auto-generated URI for the category and language.
	 */
	public function autogenCategoryURIMapping($id, $parent_category_id, $name, $language_code, $language_id,
		$mapping_template = null)
	{
		global $db;
		
		// Minimise the number of SQL queries by loading all usable info for the category at once, if any of it is
		// requested but not supplied
		if (is_null($mapping_template) ||
				(strpos($mapping_template, '{category-path}') !== false && is_null($parent_category_id)) ||
				(strpos($mapping_template, '{category-name}') !== false && is_null($name))) {
			$category_info_result = $db->Execute("
				SELECT
					c.parent_id,
					cd.categories_name
				FROM
					" . TABLE_CATEGORIES . " c
				LEFT JOIN
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				ON
					cd.categories_id = c.categories_id
				WHERE
					c.categories_id = '" . (int) $id . "'
				AND
					cd.language_id = '" . (int) $language_id . "';");
			
			if ($category_info_result->EOF) {
				// Need the info!
				return ''; 
			}
			
			if (is_null($parent_category_id)) {
				$parent_category_id = $category_info_result->fields['parent_id'];
			}
			
			if (is_null($name)) {
				$name = $category_info_result->fields['categories_name'];
			}
		}
		
		// Has a mapping template been supplied or should the template for this category be looked up?
		if (is_null($mapping_template)) {
			// No template specified so must get the template for this category
			$mapping_template =
				$this->getCategoriesOrProductsURIMappingTemplate($parent_category_id, 'categories', $language_id);
		}
		
		$mapping = $mapping_template;
		
		// Add the category path if requested
		if (preg_match('/\{category\-path(\-max\-)*(last|first)*(\-[0-9]+)*\}/', $mapping_template, $matches)) {
			
			$placement_marker = $matches[0];
			
			if ($parent_category_id == 0) {
				$category_path = '';
			} else {
				$category_path_array =
					$this->getCategoryOrProductPath($parent_category_id, 'category', $language_id);
				
				$category_path_array = array_reverse($category_path_array);
				
				$category_path_strings = array();
				
				$num_categories = sizeof($category_path_array);
				
				// Limt the number of categories to be used in the URI, if specified by the placement tag format
				$starting_category_i = 0;
				
				$finishing_category_i = $num_categories - 1;
				
				if (isset($matches[1]) && $matches[1] == '-max-') {
					if (isset($matches[2]) && $matches[2] == 'last') {
						if (!isset($matches[3])) {
							// Limiting number hasn't been specified!
						} else {
							// Remove dash to the numerical value
							$max_categories = substr($matches[3], 1, strlen($matches[3]) - 1);
							
							$starting_category_i = $finishing_category_i - ($max_categories - 1);
							
							if ($starting_category_i < 0) {
								$starting_category_i = 0;
							}
						}
					}
				}
				
				for ($i = $starting_category_i; $i <= $finishing_category_i; $i++) {
					$category_path_part = $this->_convertStringForURI($category_path_array[$i], $language_code);
					
					if (strlen($category_path_part) == 0 || $category_path_part == '/') {
						// Must not generate URIs for any category which has no name
						return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME;
					}
					
					$category_path_strings[] = $category_path_part;
				}
				
				// Implode the category path strings into the category's URI part
				$category_path = implode('/', $category_path_strings);
			}
			
			$mapping = str_replace($placement_marker, $category_path, $mapping);
		}
		
		if (strpos($mapping_template, '{category-name}') !== false) {
			$name = $this->_convertStringForURI($name, $language_code);
			
			if (strlen($name) == 0 || $name == '/') {
				// Must not generate URIs for any category which has no name
				return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_WITH_NO_NAME;
			}
			
			$mapping = str_replace('{category-name}', $name, $mapping);
		}
		
		if (strpos($mapping_template, '{category-id}') !== false) {
			$mapping = str_replace('{category-id}', $id, $mapping);
		}
		
		if (strpos($mapping_template, '{language-name}') !== false) {
			if (!class_exists('language')) {
				include_once( DIR_WS_CLASSES . 'language.php');
			}
			
			$lang = new language();
			
			$mapping = str_replace('{language-name}', $lang->catalog_languages[$language_code]['name'], $mapping);
			unset($lang);
		}
		
		if (strpos($mapping_template, '{language-code}') !== false) {
			$mapping = str_replace('{language-code}', $language_code, $mapping);
		}
		
		if (strpos($mapping_template, '{language-directory}') !== false) {
			if (!class_exists('language')) {
				include_once( DIR_WS_CLASSES . 'language.php');
			}
			
			$lang = new language();
			
			$mapping = str_replace('{language-directory}', $lang->catalog_languages[$language_code]['directory'], $mapping);
			unset($lang);
		}
		
		if (strpos($mapping_template, '{dir-ws-catalog}') !== false) {
			$mapping = str_replace('{dir-ws-catalog}', DIR_WS_CATALOG, $mapping);
		}
		
		return $mapping;
	}
	
	// }}}
}

// }}}
