<?php

/**
 * Ceon URI Mapping - UMM Edition - Admin Shared Categories/Products Class.
 *
 * This file contains a class with shared methods necessary to handle the Ceon URI Mapping admin functionality for
 * categories or products.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingAdminCategoriesProducts.php 1054 2012-09-22 15:45:15Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdmin.php');


// {{{ constants

// Other constants defined in page-type admin class files
define('CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME', -1);

// }}}


// {{{ CeonURIMappingAdminCategoriesProducts

/**
 * Provides shared functionality for the Ceon URI Mapping admin functionality for categories and products.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2008 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingAdminCategoriesProducts extends CeonURIMappingAdmin
{
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingAdminCategoriesProducts class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ getCategoryOrProductPath()
	
	/**
	 * Looks up the hierarchy of the parent categories for the given category/product.
	 *
	 * @access  public
	 * @param   integer   $id            The ID of the category/product.
	 * @param   string    $for           Either 'category' or 'product'.
	 * @param   integer   $language_id   The ID of the language of the category names to lookup.
	 * @param   array     $categories_array   Used internally as part of recursion process.
	 * @return  array     A hierarchical array of the parent categories (if any) for the given category/product,
	 *                    from the "leaf" product/category back to the "root/top" category.
	 */
	public function getCategoryOrProductPath($id, $for, $language_id, $categories_array = '')
	{
		global $db;
		
		if (!is_array($categories_array)) {
			$categories_array = array();
		}
		
		if ($id == 0) {
			return $categories_array;
		}
		
		if ($for == 'product') {
			$master_category = $db->Execute("
				SELECT
					pd.products_name,
					p.master_categories_id
				FROM
					" . TABLE_PRODUCTS . " p
				LEFT JOIN
					" . TABLE_PRODUCTS_DESCRIPTION . " pd
				ON
					pd.products_id = p.products_id
				WHERE
					p.products_id = '" . (int) $id . "'
				AND
					pd.language_id = '" . (int) $language_id . "';");
			
			if (!$master_category->EOF) {
				$categories_array[] = $master_category->fields['products_name'];
				
				if ($master_category->fields['master_categories_id'] == '0') {
					// Product uses root/top category
				} else {
					$category = $db->Execute("
						SELECT
							cd.categories_name,
							c.parent_id
						FROM
							" . TABLE_CATEGORIES . " c,
							" . TABLE_CATEGORIES_DESCRIPTION . " cd
						WHERE
							c.categories_id = '" .
								(int) $master_category->fields['master_categories_id'] . "'
						AND
							c.categories_id = cd.categories_id
						AND
							cd.language_id = '" . (int) $language_id . "'");
					
					$categories_array[] = $category->fields['categories_name'];
					
					if ((zen_not_null($category->fields['parent_id'])) &&
							($category->fields['parent_id'] != '0')) {
						$categories_array = $this->getCategoryOrProductPath($category->fields['parent_id'],
							'category', $language_id, $categories_array);
					}
				}
			}
		} else if ($for == 'category') {
			$category = $db->Execute("
				SELECT
					cd.categories_name,
					c.parent_id
				FROM
					" . TABLE_CATEGORIES . " c,
					" . TABLE_CATEGORIES_DESCRIPTION . " cd
				WHERE
					c.categories_id = '" . (int) $id . "'
				AND
					c.categories_id = cd.categories_id
				AND
					cd.language_id = '" . (int) $language_id . "'");
			
			$categories_array[] = $category->fields['categories_name'];
			
			if ((zen_not_null($category->fields['parent_id'])) && ($category->fields['parent_id'] != '0')) {
				$categories_array = $this->getCategoryOrProductPath($category->fields['parent_id'], 'category',
					$language_id, $categories_array);
			}
		}
		
		return $categories_array;
	}
	
	// }}}
	
	
	// {{{ getCategoriesOrProductsURIMappingTemplate()
	
	/**
	 * Gets the most specific mapping template for the subcategories of, or products within, the specified
	 * category, for the specified language.
	 *
	 * @access  public
	 * @param   integer   $id            The ID of the category.
	 * @param   string    $for           Either 'categories' or 'products'.
	 * @param   integer   $language_id   The ID of the language for the template.
	 * @param   boolean   $use_cache     Whether or not the cache should be used.
	 * @return  string    The template for categories/products within the specified category.
	 */
	public function getCategoriesOrProductsURIMappingTemplate($category_id, $for, $language_id, $use_cache = true)
	{
		global $db;
		
		if ($use_cache) {
			if (!isset($GLOBALS['ceon-uri-mapping-templates-' . $for])) {
				// Cache not initialised yet, initialise it now
				$GLOBALS['ceon-uri-mapping-templates-' . $for] = array();
			} else if (isset($GLOBALS['ceon-uri-mapping-templates-' . $for][$category_id]) &&
					isset($GLOBALS['ceon-uri-mapping-templates-' . $for][$category_id][$language_id])) {
				// Mapping is already in cache, return it!
				return $GLOBALS['ceon-uri-mapping-templates-' . $for][$category_id][$language_id];
			}
		}
		
		$template = '';
		
		// Progress back through the list of parent categories for the specified category, getting the most
		// uniquely specified template
		if ($for == 'categories') {
			$template_type = 'category';
		} else {
			$template_type = 'product';
		}
		
		// Get the mapping template(s) for this category
		$columns_to_retrieve = array(
			'mapping_template'
			);
		
		$selections = array(
			'category_id' => $category_id,
			'language_id' => $language_id
			);
		
		$mapping_template_result =
			$this->getURIMappingTemplatesResultset($template_type, $columns_to_retrieve, $selections, null, 1);
		
		if (!$mapping_template_result->EOF) {
			// Have found a specific template for this category
			if ($use_cache) {
				// Add this template to the cache
				if (!isset($GLOBALS['ceon-uri-mapping-templates-' . $for][$category_id])) {
					$GLOBALS['ceon-uri-mapping-templates-' . $for][$category_id] = array();
				}
				
				$GLOBALS['ceon-uri-mapping-templates-' . $for][$category_id][$language_id] =
					$mapping_template_result->fields['mapping_template'];
			}
			
			return $mapping_template_result->fields['mapping_template'];
		}
		
		// No specific template found for this category, check parent's category
		if ($category_id == 0) {
			// Can't move any further up, at the top!
			// Really shouldn't get here
			return false;
		}
		
		$parent_category_id_result = $db->Execute("
			SELECT
				parent_id
			FROM
				" . TABLE_CATEGORIES . "
			WHERE
				categories_id = '" . (int) $category_id . "';");
		
		if ($parent_category_id_result->EOF) {
			// Should never happen
			return false;
		}
		
		return $this->getCategoriesOrProductsURIMappingTemplate(
			$parent_category_id_result->fields['parent_id'], $for, $language_id, $use_cache);
	}
	
	// }}}
}

// }}}
