<?php

/**
 * Ceon URI Mapping - UMM Edition - EZ-Page URI Mappings Admin Functionality.
 *
 * This file contains a class with the methods necessary to handle URI mappings for EZ-pages.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingAdminEZPages.php 1027 2012-07-17 20:31:10Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdmin.php');


// {{{ constants

define('CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_EZ_PAGE_WITH_NO_NAME', -6);

// }}}


// {{{ CeonURIMappingAdminEZPages

/**
 * Handles the URI mappings for EZ-pages.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingAdminEZPages extends CeonURIMappingAdmin
{
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingAdminEZPages class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ autogenEZPageURIMapping()
	
	/**
	 * Generates a URI mapping for an EZ-Page, for the specified language. Doesn't check the format of the mapping,
	 * only the format of the individual placement tags it replaces in the template, so the actual URI should be
	 * cleaned/validated after this method has returned a URI.
	 *
	 * @access  public
	 * @param   integer   $id              The ID of the EZ-Page.
	 * @param   string    $name            The name of EZ-Page (used if new information is being
	 *                                     submitted when the URI is being generated).
	 * @param   string    $language_code   The ISO 639 language code of the language.
	 * @param   integer   $language_id     The Zen Cart language ID for the language.
	 * @param   string    $mapping_template   The mapping template for EZ-Pages.
	 * @return  string    The auto-generated URI for the EZ-Page and language.
	 */
	public function autogenEZPageURIMapping($id, $name, $language_code, $language_id, $mapping_template = null)
	{
		global $db, $sniffer;
		
		if (is_null($name)) {
			// Assume name will be used.. load name from database
			$ez_page_table = (defined('TABLE_EZPAGES_CONTENT') && $sniffer->table_exists(TABLE_EZPAGES_CONTENT)) ? TABLE_EZPAGES_CONTENT : TABLE_EZPAGES;
			
			$ez_page_name_result = $db->Execute("
				SELECT
					pages_title
				FROM
					" . $ez_page_table . "
				WHERE
					pages_id = " . (int) $id . "
				AND
					languages_id = " . (int) $language_id);
			
			$name = $ez_page_name_result->fields['pages_title'];
		}
		
		// Has a mapping template been supplied or should the template for EZ-Pages be looked up?
		if (is_null($mapping_template)) {
			$mapping_template = $this->getEZPagesURIMappingTemplate($language_id);
		}
		
		$mapping = $mapping_template;
		
		if (strpos($mapping_template, '{ez-page-name}') !== false) {
			$name = $this->_convertStringForURI($name, $language_code);
			
			if (strlen($name) == 0 || $name == '/') {
				// Must not generate URIs for any ez-page which has no name
				return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_EZ_PAGE_WITH_NO_NAME;
			}
			
			$mapping = str_replace('{ez-page-name}', $name, $mapping);
		}
		
		if (strpos($mapping_template, '{ez-page-id}') !== false) {
			$mapping = str_replace('{ez-page-id}', $id, $mapping);
		}
		
		if (strpos($mapping_template, '{language-name}') !== false) {
			if (!class_exists('language')) {
				include_once( DIR_WS_CLASSES . 'language.php');
			}
			
			$lang = new language();
			
			$mapping = str_replace('{language-name}', $lang->catalog_languages[$language_code]['name'], $mapping);
			unset($lang);
		}
		
		if (strpos($mapping_template, '{language-code}') !== false) {
			$mapping = str_replace('{language-code}', $language_code, $mapping);
		}
		
		if (strpos($mapping_template, '{language-directory}') !== false) {
			if (!class_exists('language')) {
				include_once( DIR_WS_CLASSES . 'language.php');
			}
			
			$lang = new language();
			
			$mapping = str_replace('{language-directory}', $lang->catalog_languages[$language_code]['directory'], $mapping);
			unset($lang);
		}
		
		if (strpos($mapping_template, '{dir-ws-catalog}') !== false) {
			$mapping = str_replace('{dir-ws-catalog}', DIR_WS_CATALOG, $mapping);
		}
		
		return $mapping;
	}
	
	// }}}
	
	
	// {{{ getEZPagesURIMappingTemplate()
	
	/**
	 * Looks up the mapping template for the specified language.
	 *
	 * @access  public
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping template.
	 * @param   boolean   $use_cache     Whether or not the cache should be used.
	 * @return  string    The mapping template.
	 */
	public function getEZPagesURIMappingTemplate($language_id, $use_cache = true)
	{
		return $this->getObjectURIMappingTemplate('ez-page', $language_id, $use_cache);
	}
	
	// }}}
}

// }}}
