<?php

/**
 * Ceon URI Mapping - UMM Edition - Manufacturer URI Mappings Admin Functionality.
 *
 * This file contains a class with the methods necessary to handle URI mappings for manufacturers.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingAdminManufacturers.php 1027 2012-07-17 20:31:10Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdmin.php');


// {{{ constants

define('CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_MANUFACTURER_WITH_NO_NAME', -5);

// }}}


// {{{ CeonURIMappingAdminManufacturers

/**
 * Handles the URI mappings for Manufacturers.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingAdminManufacturers extends CeonURIMappingAdmin
{
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingAdminManufacturers class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ autogenManufacturerURIMapping()
	
	/**
	 * Generates a URI mapping for a manufacturer, for the specified language. Doesn't check the format of the
	 * mapping, only the format of the individual placement tags it replaces in the  template, so the actual URI
	 * should be cleaned/validated after this method has returned a URI.
	 *
	 * @access  public
	 * @param   integer   $id              The ID of the manufacturer.
	 * @param   string    $name            The name of manufacturer (used if new information is
	 *                                     being submitted when the URI is being generated).
	 * @param   string    $language_code   The ISO 639 language code of the language.
	 * @param   integer   $language_id     The Zen Cart language ID for the language.
	 * @param   string    $mapping_template   The mapping template for manufacturers.
	 * @return  string    The auto-generated URI for the manufacturer and language.
	 */
	public function autogenManufacturerURIMapping($id, $name, $language_code, $language_id, $mapping_template = null)
	{
		global $db;
		
		if (is_null($name)) {
			// Assume name will be used.. load name from database
			$manufacturer_name_result = $db->Execute("
				SELECT
					manufacturers_name
				FROM
					" . TABLE_MANUFACTURERS . "
				WHERE
					manufacturers_id = " . (int) $id . ";");
			
			$name = $manufacturer_name_result->fields['manufacturers_name'];
		}
		
		// Has a mapping template been supplied or should the template for manufacturers be looked up?
		if (is_null($mapping_template)) {
			$mapping_template = $this->getManufacturersURIMappingTemplate($language_id);
		}
		
		$mapping = $mapping_template;
		
		if (strpos($mapping_template, '{manufacturer-name}') !== false) {
			$name = $this->_convertStringForURI($name, $language_code);
			
			if (strlen($name) == 0 || $name == '/' /*|| $name == '\\' @TODO?*/) {
				// Must not generate URIs for any manufacturer which has no name
				return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_MANUFACTURER_WITH_NO_NAME;
			}
			
			$mapping = str_replace('{manufacturer-name}', $name, $mapping);
		}
		
		if (strpos($mapping_template, '{manufacturer-id}') !== false) {
			$mapping = str_replace('{manufacturer-id}', $id, $mapping);
		}
		
		if (strpos($mapping_template, '{dir-ws-catalog}') !== false) {
			if (strlen(DIR_WS_CATALOG) > 0) {
				$mapping = str_replace('{dir-ws-catalog}', DIR_WS_CATALOG, $mapping);
			} else {
				$mapping = str_replace('{dir-ws-catalog}', '/', $mapping);
			}
		}
		
		return $mapping;
	}
	
	// }}}
	
	
	// {{{ getManufacturersURIMappingTemplate()
	
	/**
	 * Looks up the mapping template for the specified language.
	 *
	 * @access  public
	 * @param   integer   $language_id   The Zen Cart language ID for the mapping template.
	 * @param   boolean   $use_cache     Whether or not the cache should be used.
	 * @return  string    The mapping template.
	 */
	public function getManufacturersURIMappingTemplate($language_id, $use_cache = true)
	{
		return $this->getObjectURIMappingTemplate('manufacturer', $language_id, $use_cache);
	}
	
	// }}}
}

// }}}
