<?php

/**
 * Ceon URI Mapping - UMM Edition - Product URI Mappings Admin Functionality.
 *
 * This file contains a class with the methods necessary to handle URI mappings for products.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonURIMappingAdminProducts.php 1054 2012-09-22 15:45:15Z conor $
 */

if (!defined('IS_ADMIN_FLAG')) {
	die('Illegal Access');
}

/**
 * Load in the parent class if not already loaded
 */
require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminCategoriesProducts.php');


// {{{ constants

define('CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_NAME', -3);
define('CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_MODEL', -4);

// }}}


// {{{ CeonURIMappingAdminProducts

/**
 * Handles the URI mappings for products.
 *
 * @package     ceon_uri_mapping
 * @author      Conor Kerr <zen-cart.uri-mapping@ceon.net>
 * @copyright   Copyright 2008-2019 Ceon
 * @copyright   Copyright 2003-2019 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mapping
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonURIMappingAdminProducts extends CeonURIMappingAdminCategoriesProducts
{
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the CeonURIMappingAdminProducts class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		parent::__construct();
	}
	
	// }}}
	
	
	// {{{ autoManageProductRelatedPageURI()
	
	/**
	 * Checks whether auto-managing of the specified product related page URI is enabled or disabled.
	 *
	 * @access  public
	 * @param   string    $page_type   The type of the page.
	 * @return  boolean   Whether auto-managing of the page's URI is enabled or disabled.
	 */
	public function autoManageProductRelatedPageURI($page_type)
	{
		global $db;
		
		if (!isset($automanage_enabled)) {
			static $automanage_enabled = array();
			
			// Only one config currently supported so ID is hard-coded in following SQL
			$automanage_enabled_sql = "
				SELECT
					manage_product_reviews_mappings,
					manage_product_reviews_info_mappings,
					manage_product_reviews_write_mappings,
					manage_tell_a_friend_mappings,
					manage_ask_a_question_mappings
				FROM
					" . TABLE_CEON_URI_MAPPING_CONFIGS . "
				WHERE
					id ='1';";
			
			$automanage_enabled_result = $db->Execute($automanage_enabled_sql);
			
			if (!$automanage_enabled_result->EOF) {
				$automanage_enabled = array(
					'product_reviews' =>
						$automanage_enabled_result->fields['manage_product_reviews_mappings'],
					'product_reviews_info' =>
						$automanage_enabled_result->fields['manage_product_reviews_info_mappings'],
					'product_reviews_write' =>
						$automanage_enabled_result->fields['manage_product_reviews_write_mappings'],
					'tell_a_friend' =>
						$automanage_enabled_result->fields['manage_tell_a_friend_mappings'],
					'ask_a_question' =>
						$automanage_enabled_result->fields['manage_ask_a_question_mappings']
					);
			}
		}
		
		if ($automanage_enabled[$page_type] == 1) {
			return true;
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ getProductRelatedPageURIPart()
	
	/**
	 * Looks up the product related page URI part for the page type and language specified.
	 *
	 * @access  public
	 * @param   string    $page_type   The type of the page.
	 * @param   string    $language_code   The language code the URI part should be returned for.
	 * @return  string|false   The product related page's URI part or false if there is no URI part for the
	 *                         specified page type and language code.
	 */
	public function getProductRelatedPageURIPart($page_type, $language_code)
	{
		global $db;
		
		if (!isset($uri_parts) || !isset($uri_parts[$language_code]) ||
				!isset($uri_parts[$language_code][$page_type])) {
			// URI part hasn't been cached for this page type for the specified language, cache all now
			static $uri_parts = array();
			
			// Only hold the information in memory for URI parts that are being auto-managed
			$page_types = array(
				'product_reviews',
				'product_reviews_info',
				'product_reviews_write',
				'tell_a_friend',
				'ask_a_question'
				);
			
			$page_types_sql_string = '';
			
			foreach ($page_types as $current_page_type) {
				if ($this->autoManageProductRelatedPageURI($current_page_type)) {
					if (strlen($page_types_sql_string) > 0) {
						$page_types_sql_string .= ' OR ';
					}
					
					$page_types_sql_string .= "page_type = '" . $current_page_type . "'\n";
				}
			}
			
			// Only one config currently supported so ID is hard-coded in following SQL
			$uri_parts_sql = "
				SELECT
					page_type,
					language_code,
					uri_part
				FROM
					" . TABLE_CEON_URI_MAPPING_PRODUCT_RELATED_PAGES_URI_PARTS . "
				WHERE
					" . $page_types_sql_string . ";";
			
			$uri_parts_result = $db->Execute($uri_parts_sql);
			
			while (!$uri_parts_result->EOF) {
				if (!isset($uri_parts[$uri_parts_result->fields['language_code']])) {
					$uri_parts[$uri_parts_result->fields['language_code']] = array();
				}
				
				$uri_parts[$uri_parts_result->fields['language_code']][$uri_parts_result->fields['page_type']] =
					$uri_parts_result->fields['uri_part'];
				
				$uri_parts_result->MoveNext();
			}
		}
		
		if (isset($uri_parts[$language_code]) && isset($uri_parts[$language_code][$page_type]) &&
				strlen($uri_parts[$language_code][$page_type]) > 0) {
			
			return $uri_parts[$language_code][$page_type];
		}
		
		return false;
	}
	
	// }}}
	
	
	// {{{ autogenProductURIMapping()
	
	/**
	 * Generates a URI mapping for a product, for the specified language. Doesn't check the format of the mapping, 
	 * only the format of the individual placement tags it replaces in the template, so the actual URI should be
	 * cleaned/validated after this method has returned a URI. If no template is specified, the template for the
	 * product's category will be looked up.
	 *
	 * @access  public
	 * @param   integer   $id              The ID of the product.
	 * @param   integer   $parent_category_id   The ID of the product's master category (will be looked up if not
	 *                                          supplied).
	 * @param   string    $name               The name of product (used if new information is being submitted when
	 *                                        the URI is being generated).
	 * @param   string    $language_code      The ISO 639 language code of the language.
	 * @param   integer   $language_id        The Zen Cart language ID for the language.
	 * @param   string    $model              The product's model code (will be looked up, if necessary, if not
	 *                                        supplied).
	 * @param   string    $mapping_template   The mapping template for this product.
	 * @return  string|integer    The auto-generated URI for the product and language or an error constant if there
	 *                            was a problem with the auto-generation.
	 */
	public function autogenProductURIMapping($id, $parent_category_id, $name, $language_code, $language_id, $model = null,
		$mapping_template = null)
	{
		global $db;
		
		$model = (!is_null($model) && strlen($model) > 0 ? $model : null);
		
		// Minimise the number of SQL queries by loading all usable info for the product at once, if any of it is
		// requested but not supplied
		if (!is_null($id) && (is_null($mapping_template) ||
				(strpos($mapping_template, '{category-path}') !== false && is_null($parent_category_id)) ||
				(strpos($mapping_template, '{manufacturer-name}') !== false) ||
				(strpos($mapping_template, '{product-name}') !== false && is_null($name)) ||
				(strpos($mapping_template, '{product-model}') !== false && is_null($model)))) {
			$product_info_result = $db->Execute("
				SELECT
					p.master_categories_id,
					m.manufacturers_name,
					pd.products_name,
					p.products_model
				FROM
					" . TABLE_PRODUCTS . " p
				LEFT JOIN
					" . TABLE_PRODUCTS_DESCRIPTION . " pd
				ON
					pd.products_id = p.products_id
				LEFT JOIN
					" . TABLE_MANUFACTURERS . " m
				ON
					m.manufacturers_id = p.manufacturers_id
				WHERE
					p.products_id = '" . (int) $id . "'
				AND
					pd.language_id = '" . (int) $language_id . "';");
			
			if ($product_info_result->EOF) {
				// Need the info!
				return ''; 
			}
			
			if (is_null($parent_category_id)) {
				$parent_category_id = $product_info_result->fields['master_categories_id'];
			}
			
			$manufacturer_name = $product_info_result->fields['manufacturers_name'];
			
			if (is_null($name)) {
				$name = $product_info_result->fields['products_name'];
			}
			
			if (is_null($model)) {
				$model = $product_info_result->fields['products_model'];
			}
		}
		
		// Has a mapping template been supplied or should the template for this product be looked up?
		if (is_null($mapping_template)) {
			// No template specified so must get the template for products in this product's category
			$mapping_template = $this->getCategoriesOrProductsURIMappingTemplate(
				$parent_category_id, 'products', $language_id);
		}
		
		$mapping = $mapping_template;
		
		// Add the category path if requested
		if (strpos($mapping_template, '{category-path}') !== false) {
			// Handle special (bad) case of product in the top category
			if ($parent_category_id == 0) {
				$category_path = '';
			} else {
				$category_path_array =
					$this->getCategoryOrProductPath($parent_category_id, 'category', $language_id);
				
				$category_path_array = array_reverse($category_path_array);
				
				for ($i = 0, $n = sizeof($category_path_array); $i < $n; $i++) {
					$category_path_part = $this->_convertStringForURI($category_path_array[$i], $language_code);
					
					if (strlen($category_path_part) == 0 || $category_path_part == '/') {
						// Must not generate URIs for any category which has no name
						return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_CATEGORY_PATH_PART_WITH_NO_NAME;
					}
					
					$category_path_array[$i] = $category_path_part;
				}
				
				// Implode the category path array into its URI part
				$category_path = implode('/', $category_path_array);
			}
			
			$mapping = str_replace('{category-path}', $category_path, $mapping);
		}
		
		if (strpos($mapping_template, '{manufacturer-name}') !== false) {
			$manufacturer_name = $this->_convertStringForURI($manufacturer_name, $language_code);
			
			$mapping = str_replace('{manufacturer-name}', $manufacturer_name, $mapping);
		}
		
		if (strpos($mapping_template, '{product-name}') !== false) {
			$name = $this->_convertStringForURI($name, $language_code);
			
			if (strlen($name) == 0 || $name == '/') {
				// Must not generate URIs for any product which has no name
				return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_NAME;
			}
			
			$mapping = str_replace('{product-name}', $name, $mapping);
		}
		
		if (strpos($mapping_template, '{product-model}') !== false) {
			$model = $this->_convertStringForURI($model, $language_code);
			
			if ((strlen($model) == 0 || $model == '/') &&
					strpos($mapping_template, '{product-name}') == false &&
					strpos($mapping_template, '{product-id}') === false) {
				// Must not generate URIs for any product which has no model when the model is the only unique tag
				// in the template!
				return CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_PRODUCT_WITH_NO_MODEL;
			}
			
			$mapping = str_replace('{product-model}', $model, $mapping);
		}
		
		if (strpos($mapping_template, '{product-id}') !== false) {
			$mapping = str_replace('{product-id}', $id, $mapping);
		}
		
		if (strpos($mapping_template, '{language-name}') !== false) {
			if (!class_exists('language')) {
				include_once( DIR_WS_CLASSES . 'language.php');
			}
			
			$lang = new language();
			
			$mapping = str_replace('{language-name}', $lang->catalog_languages[$language_code]['name'], $mapping);
			unset($lang);
		}
		
		if (strpos($mapping_template, '{language-code}') !== false) {
			$mapping = str_replace('{language-code}', $language_code, $mapping);
		}
		
		if (strpos($mapping_template, '{language-directory}') !== false) {
			if (!class_exists('language')) {
				include_once( DIR_WS_CLASSES . 'language.php');
			}
			
			$lang = new language();
			
			$mapping = str_replace('{language-directory}', $lang->catalog_languages[$language_code]['directory'], $mapping);
			unset($lang);
		}
		
		if (strpos($mapping_template, '{dir-ws-catalog}') !== false) {
			$mapping = str_replace('{dir-ws-catalog}', DIR_WS_CATALOG, $mapping);
		}
		
		return $mapping;
	}
	
	// }}}
}

// }}}
