<?php

/**
 * Ceon Advanced Shipper Upgrade Script - Upgrades database from version 2.x onwards.
 *
 * @package     ceon_advanced_shipper
 * @author      Conor Kerr <zen-cart.advanced-shipper@dev.ceon.net>
 * @copyright   Copyright 2007-2018 Ceon
 * @copyright   Portions Copyright 2003-2006 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://dev.ceon.net/web/zen-cart/advanced-shipper
 * @license     http://www.gnu.org/copyleft/gpl.html   GNU Public License V2.0
 * @version     $Id: advshipper_auto_upgrade.php 982 2018-11-21 16:47:00Z conor updated v6.0.5 $
 */

// Find out what the latest version number is
require_once(DIR_FS_CATALOG . DIR_WS_MODULES . 'shipping/advshipper.php');

// {{{ functions

// {{{ advshipperUpdateSurchargeColumn()

/*
 * Updates the surcharge column of the region configs table and update the format of any existing
 * surcharges.
 *
 * @return  none
 */
function advshipperUpdateSurchargeColumn()
{
	global $db;
	
	$update_surcharge_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		CHANGE
			`surcharge` `surcharge` TEXT NULL;";
	
	$update_surcharge_column_result = $db->Execute($update_surcharge_column_sql);
	
	// Now modify all existing surcharges to use the new format
	$surcharges_query = "
		SELECT
			config_id,
			method,
			region,
			surcharge
		FROM
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		WHERE
			surcharge IS NOT NULL;";
	
	$surcharges_result = $db->Execute($surcharges_query);
	
	while (!$surcharges_result->EOF) {
		$surcharge = $surcharges_result->fields['surcharge'];
		
		$pattern = '|^(\<[^\>]+\>)|iU';
		
		if (preg_match($pattern, $surcharge, $matches)) {
			$opening_tag = $matches[1];
			
			$surcharge = str_replace($opening_tag, $opening_tag . '*: ', $surcharge);
			
			$update_record_format_query = "
				UPDATE
					" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
				SET
					surcharge = '" . zen_db_input($surcharge) . "'
				WHERE
					config_id = '" . $surcharges_result->fields['config_id'] . "'
				AND
					method = '" . $surcharges_result->fields['method'] . "'
				AND
					region = '" . $surcharges_result->fields['region'] . "';";
			
			$update_record_format_result = $db->Execute($update_record_format_query);
		}
		
		$surcharges_result->MoveNext();
	}
}

// }}}


// {{{ advshipperUpdateUSPSConfig3_8_3()

/*
 * Update the USPS configuration options in the region USPS configs table.
 *
 * @return  none
 */
function advshipperUpdateUSPSConfig3_8_3()
{
	global $db;
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_gee` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_gendnr`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmisfrb` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_pmifrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmimfrb` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_pmisfrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmilfrb` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_pmimfrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$delete_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		DROP
			`international_fcmif`;";
	
	$delete_column_result = $db->Execute($delete_column_sql);
	
	
	$delete_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		DROP
			`international_fcmipar`;";
	
	$delete_column_result = $db->Execute($delete_column_sql);
}

// }}}


// {{{ advshipperUpdateUSPSConfig4_2_0()

/*
 * Update the USPS configuration options in the region USPS configs table.
 *
 * @return  none
 */
function advshipperUpdateUSPSConfig4_2_0()
{
	global $db;
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_ge` `international_geg` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_geg_d` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_geg`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_gendr` `international_geg_ndr` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_gendnr` `international_geg_ndnr` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_gee` `international_geg_e` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_emifre` `international_emi_fre` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_emi_lfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_emi_fre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmisfrb` `international_pmi_sfrb` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmimfrb` `international_pmi_mfrb` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmilfrb` `international_pmi_lfrb` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_dfrb` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_lfrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_lvfrb` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_dfrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmifre` `international_pmi_fre` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_sfre` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_pmi_fre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_pfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_sfre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_gcfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_pfre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_lfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_gcfre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_fcmile` `international_fcmi_le` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_fcmip` `international_fcmi_p` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_fcmil` `international_fcmi_l` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$delete_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		DROP
			`international_pmifrb`;";
	
	$delete_column_result = $db->Execute($delete_column_sql);
}

	
// {{{ advshipperUpdateUSPSConfig4_2_0()

/*
 * Update the USPS configuration options in the region USPS configs table.
 *
 * @return  none
 */
function advshipperUpdateUSPSConfigCEON6_0_0_CEON()
{
	global $db;
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`domestic_parcel` `domestic_retail_ground` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
/*	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_geg_d` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_geg`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_gendr` `international_geg_ndr` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_gendnr` `international_geg_ndnr` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_gee` `international_geg_e` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_emifre` `international_emi_fre` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_emi_lfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_emi_fre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmisfrb` `international_pmi_sfrb` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmimfrb` `international_pmi_mfrb` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmilfrb` `international_pmi_lfrb` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_dfrb` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_lfrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_lvfrb` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_dfrb`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_pmifre` `international_pmi_fre` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_sfre` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`international_pmi_fre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_pfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_sfre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_gcfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_pfre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		ADD
			`international_pmi_lfre` TINYINT NOT NULL DEFAULT '0'
		AFTER
			`international_pmi_gcfre`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_fcmile` `international_fcmi_le` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_fcmip` `international_fcmi_p` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		CHANGE
			`international_fcmil` `international_fcmi_l` TINYINT NOT NULL DEFAULT '1';";
	
	$rename_column_result = $db->Execute($rename_column_sql);
	
	
	$delete_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "
		DROP
			`international_pmifrb`;";
	
	$delete_column_result = $db->Execute($delete_column_sql);*/
}



// }}}


// {{{ advshipperAddCustomProductFieldSupport4_2_0()

/*
 * Adds the column to the config for the new custom product field key and add the table for mapping
 * the value of custom product field values to method numbers.
 *
 * @return  none
 */
function advshipperAddCustomProductFieldSupport4_2_0()
{ 
	global $db;
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_CONFIGS . "
		ADD
			`custom_product_field_name` VARCHAR(40) DEFAULT NULL
		AFTER
			`default_method`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the Custom Product Field Selections table ///////////////////////////////////////////////
	$add_custom_product_field_values_table_sql = "
		CREATE TABLE IF NOT EXISTS
			`" . TABLE_ADVANCED_SHIPPER_METHOD_CUSTOM_PRODUCT_FIELD_VALUES . "`
		(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`field_value_order` INT UNSIGNED NOT NULL,
			`field_value` VARCHAR(80) DEFAULT '',
			PRIMARY KEY ( 
				`config_id`,
				`method`,
				`field_value_order`,
				`field_value`
			)
		);";
	
	$add_custom_product_field_values_table_result =
		$db->Execute($add_custom_product_field_values_table_sql);
}

// }}}


// {{{ advshipperUpdateMethodSortOrderOptions4_2_0()

/*
 * Updates the available options for the ordering of methods.
 *
 * @return  none
 */
function advshipperUpdateMethodSortOrderOptions4_2_0()
{
	global $db;
	
	$sort_order_value_query = "
		SELECT
			configuration_value
		FROM
			" . TABLE_CONFIGURATION . "
		WHERE
			configuration_key = 'MODULE_ADVANCED_SHIPPER_METHOD_SORT_ORDER';";
	
	$sort_order_value_result = $db->Execute($sort_order_value_query);
	
	if ($sort_order_value_result->EOF) {
		// Error looking up value of sort order!
		$sort_order_value = '';
	} else {
		$sort_order_value = $sort_order_value_result->fields['configuration_value'];
	}
	
	if (strpos(strtolower($sort_order_value), 'admin method order') !== false) {
		$sort_order_value = 'By Shipping Date (for methods which have a shipping date selected),' .
			' then by the Order of the methods, as defined in the Admin.';
	} else if (strpos(strtolower($sort_order_value), 'cost - highest to lowest') !== false) {
		$sort_order_value = 'By Shipping Date (for methods which have a shipping date selected),' .
			' then by Cost - highest to lowest.';
	} else {
		$sort_order_value = 'By Shipping Date (for methods which have a shipping date selected),' .
			' then by Cost - lowest to highest.';
	}
	
	$update_sort_order_options_query = "
		UPDATE
			" . TABLE_CONFIGURATION . "
		SET
			configuration_value = '" . zen_db_input($sort_order_value) . "',
			configuration_description = 'How should the methods be sorted before being displayed to the customer?',
			set_function = 'zen_cfg_select_option(array(\'By Shipping Date (for methods which have a shipping date selected), then by Cost - lowest to highest.\', \'By Shipping Date (for methods which have a shipping date selected), then by Cost - highest to lowest.\', \'By Shipping Date (for methods which have a shipping date selected), then by the Order of the methods, as defined in the Admin.\', \'By Cost - lowest to highest. (Ordering will be done based on Cost only, any Shipping Date selected will be ignored when ordering the methods).\', \'By Cost - highest to lowest. (Ordering will be done based on Cost only, any Shipping Date selected will be ignored when ordering the methods).\', \'By the Order of the methods, as defined in the Admin. (Ordering will be done based on the Admin Order only, any Shipping Date selected will be ignored when ordering the methods).\'), '
		WHERE
			configuration_key = 'MODULE_ADVANCED_SHIPPER_METHOD_SORT_ORDER';";
	
	$update_sort_order_options_result = $db->Execute($update_sort_order_options_query);
}

// }}}


// {{{ advshipperAddVersionCheckingOptions4_2_0()

/*
 * Adds the options for the new automatic version checking functionality.
 *
 * @return  none
 */
function advshipperAddVersionCheckingOptions4_2_0()
{
	global $db;
	
	$db->Execute("insert into " . TABLE_CONFIGURATION . " (configuration_title, configuration_key, configuration_value, configuration_description, configuration_group_id, sort_order, set_function, date_added) values ('Version Check', 'MODULE_ADVANCED_SHIPPER_VERSION_CHECK', 'Automatic', 'Use Automatic or Manual Version Checking?', '6', '0', 'zen_cfg_select_option(array(\'Automatic\', \'Manual\'), ', now())");
}

// }}}


// {{{ advshipperAddMethodEnabledColumn4_2_0()

/*
 * Adds a column to the method config which is used to enable/disable methods.
 *
 * @return  none
 */
function advshipperAddMethodEnabledColumn4_2_0()
{ 
	global $db;
	
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_METHOD_CONFIGS . "
		ADD
			`enabled` TINYINT NOT NULL DEFAULT '1'
		AFTER
			`method`;";
	
	$add_column_result = $db->Execute($add_column_sql);
}

// }}}


// {{{ advshipperUpdateCustomProductFieldSupport5_0_0()

/*
 * Renames the custom product field column in the config table.
 *
 * @return  none
 */
function advshipperUpdateCustomProductFieldSupport5_0_0()
{ 
	global $db;
	
	$rename_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_CONFIGS . "
		CHANGE
			`custom_product_field_key` `custom_product_field_name` VARCHAR(40) DEFAULT NULL;";
	
	$rename_column_result = $db->Execute($rename_column_sql);
}

// }}}


// {{{ advshipperAddFedExSupport5_0_0()

/*
 * Adds the table for FedEx support to the database.
 *
 * @return  none
 */
function advshipperAddFedExSupport5_0_0()
{ 
	global $db;
	
	$add_fedex_configs_table_sql = "
		CREATE TABLE IF NOT EXISTS
			`" . TABLE_ADVANCED_SHIPPER_REGION_FEDEX_CONFIGS . "`
		(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`region` INT UNSIGNED NOT NULL,
			`source_country` INT UNSIGNED NOT NULL,
			`source_postcode` VARCHAR(15) NOT NULL,
			`fedex_key` VARCHAR(16) NOT NULL,
			`fedex_password` VARCHAR(25) NOT NULL,
			`fedex_account` VARCHAR(9) NOT NULL,
			`fedex_meter` VARCHAR(9) NOT NULL,
			`server` CHAR(1) NOT NULL,
			`weight_units` CHAR(2) NOT NULL,
			`rate_request_types` TINYINT NOT NULL DEFAULT '1',
			`drop_off_type` VARCHAR(23) NOT NULL,
			`packaging_type` VARCHAR(14) NOT NULL,
			`shipping_saturday` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_1_day_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_2_day` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_2_day_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_3_day_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_express_saver` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_ground` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_ground_home_delivery` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_standard_overnight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_first_overnight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_priority_overnight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_smart_post` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_fedex_national_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_international_ground` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_international_economy` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_international_economy_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_international_first` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_international_priority` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_international_priority_freight` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_europe_first_international_priority` TINYINT NOT NULL DEFAULT '1',
			
			PRIMARY KEY (
				`config_id`,
				`method`,
				`region`
			)
		);";
	
	$add_fedex_configs_table_result = $db->Execute($add_fedex_configs_table_sql);
}

// }}}

// }}}


// Handle latest releases //////////////////////////////////////////////////////////////////////////
if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 3) == '3.0') {
	// Update the surcharge column of the region configs table /////////////////////////////////////
	advshipperUpdateSurchargeColumn();
	
	// Add the Manufacturer Selections table ///////////////////////////////////////////////////////
	$add_method_manufacturers_table_sql = "
		CREATE TABLE IF NOT EXISTS
			`" . TABLE_ADVANCED_SHIPPER_METHOD_MANUFACTURERS . "`
		(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`manufacturer_order` INT UNSIGNED NOT NULL,
			`manufacturer_id` INT UNSIGNED NOT NULL,
			PRIMARY KEY ( 
				`config_id`,
				`method`,
				`manufacturer_order`,
				`manufacturer_id`
			)
		);";
	
	$add_method_manufacturers_table_result = $db->Execute($add_method_manufacturers_table_sql);
	
	// Add the new maximum weight per package column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`max_weight_per_package` FLOAT UNSIGNED NULL
		AFTER
			`table_of_rates`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the new packaging weights column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`packaging_weights` TEXT NULL
		AFTER
			`max_weight_per_package`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the new total up price inc tax column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`total_up_price_inc_tax` TINYINT NOT NULL DEFAULT '2'
		AFTER
			`rate_limits_inc`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 3) == '3.2') {
	// Update the surcharge column of the region configs table /////////////////////////////////////
	advshipperUpdateSurchargeColumn();
	
	// Add the new maximum weight per package column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`max_weight_per_package` FLOAT UNSIGNED NULL
		AFTER
			`table_of_rates`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the new packaging weights column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`packaging_weights` TEXT NULL
		AFTER
			`max_weight_per_package`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the new total up price inc tax column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`total_up_price_inc_tax` TINYINT NOT NULL DEFAULT '2'
		AFTER
			`rate_limits_inc`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 3) == '3.4') {
	// Update the surcharge column of the region configs table /////////////////////////////////////
	advshipperUpdateSurchargeColumn();
	
	// Add the new maximum weight per package column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`max_weight_per_package` FLOAT UNSIGNED NULL
		AFTER
			`table_of_rates`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Must ensure that packaging weights column has the correct null setting
	$update_packaging_weights_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		CHANGE
			`packaging_weights` `packaging_weights` TEXT NULL;";
	
	$update_packaging_weights_column_result = $db->Execute($update_packaging_weights_column_sql);
	
	// Add the new total up price inc tax column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`total_up_price_inc_tax` TINYINT NOT NULL DEFAULT '2'
		AFTER
			`rate_limits_inc`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.6.0') {
	// Update the surcharge column of the region configs table /////////////////////////////////////
	advshipperUpdateSurchargeColumn();
	
	// Must ensure that packaging weights column has the correct null setting
	$update_packaging_weights_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		CHANGE
			`packaging_weights` `packaging_weights` TEXT NULL;";
	
	$update_packaging_weights_column_result = $db->Execute($update_packaging_weights_column_sql);
	
	// Add the new total up price inc tax column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`total_up_price_inc_tax` TINYINT NOT NULL DEFAULT '2'
		AFTER
			`rate_limits_inc`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.6.1' ||
		MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.6.2') {
	// Update the surcharge column of the region configs table /////////////////////////////////////
	advshipperUpdateSurchargeColumn();
	
	// Add the new total up price inc tax column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`total_up_price_inc_tax` TINYINT NOT NULL DEFAULT '2'
		AFTER
			`rate_limits_inc`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 3) == '3.6') {
	// Update the surcharge column of the region configs table /////////////////////////////////////
	advshipperUpdateSurchargeColumn();
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.8.0' ||
		MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.8.1' ||
		MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.8.2') {
	
	advshipperUpdateUSPSConfig3_8_3();
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (MODULE_ADVANCED_SHIPPER_MADE_BY_CEON == '3.8.3' ||
		substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 3) == '4.0') {
	
	advshipperUpdateUSPSConfig4_2_0();
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperUpdateMethodSortOrderOptions4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 1) == '4') {
	advshipperUpdateCustomProductFieldSupport5_0_0();
	
	advshipperAddFedExSupport5_0_0();
	
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 1) == '5') {
	advshipperUpdateUSPSConfigCEON6_0_0_CEON();
	
} else if (substr(MODULE_ADVANCED_SHIPPER_MADE_BY_CEON, 0, 1) == '6') {
	
	// Database is up to date!
} else {
	// Check if the database should be upgraded from 2.0.x to 2.2.6 first //////////////////////////
	$columns_exist_query = 'SHOW COLUMNS FROM ' . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . ';';
	$columns_exist_result = $db->Execute($columns_exist_query);
	
	$columns = array();
	while (!$columns_exist_result->EOF) {
		$columns[] = $columns_exist_result->fields['Field'];
		$columns_exist_result->MoveNext();
	}
	
	if (!in_array('countries_zones', $columns)) {
		$add_column_sql = "
			ALTER TABLE
				" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
			ADD
				`countries_zones` TEXT NULL
			AFTER
				`countries_postcodes`;";
		
		$add_column_result = $db->Execute($add_column_sql);
	}
	

	$columns_exist_query = 'SHOW COLUMNS FROM ' . TABLE_ADVANCED_SHIPPER_METHOD_CONFIGS . ';';
	$columns_exist_result = $db->Execute($columns_exist_query);
	
	$columns = array();
	while (!$columns_exist_result->EOF) {
		$columns[] = $columns_exist_result->fields['Field'];
		$columns_exist_result->MoveNext();
	}

	
	// Update the format of all the tables of rates ////////////////////////////////////////////////
	if (!count(array_diff(array('method', 'calc_method'), $columns))) {
	$load_method_config_sql = "
		SELECT
			asmc.method,
			asmc.calc_method
		FROM
			" . TABLE_ADVANCED_SHIPPER_METHOD_CONFIGS . " asmc
		WHERE
			asmc.config_id = '" . $config_id . "'
		ORDER BY
			asmc.method;";
	
	$load_method_config_result = $db->Execute($load_method_config_sql);
	
	while (!$load_method_config_result->EOF) {
		
		$method = $load_method_config_result->fields['method'];
		$calc_method = $load_method_config_result->fields['calc_method'];
		
		switch ($calc_method) {
			case 1:
				$calc_method_string = ADVSHIPPER_CALC_METHOD_WEIGHT;
				break;
			case 2:
				$calc_method_string = ADVSHIPPER_CALC_METHOD_PRICE;
				break;
			case 3:
				$calc_method_string = ADVSHIPPER_CALC_METHOD_NUM_ITEMS;
				break;
		}
		
		$load_region_config_sql = "
			SELECT
				asrc.region,
				asrc.table_of_rates
			FROM
				" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . " asrc
			WHERE
				asrc.config_id = '" . $config_id . "'
			AND
				asrc.method = '" . $method . "'
			ORDER BY
				asrc.region;";
		
		$load_region_config_result = $db->Execute($load_region_config_sql);
		
		while (!$load_region_config_result->EOF) {
			
			$region = $load_region_config_result->fields['region'];
			$table_of_rates = $load_region_config_result->fields['table_of_rates'];
			
			// Wrap table of rates with calculation method tags
			$table_of_rates = '<' . $calc_method_string . '>' . $table_of_rates . '</' .
				$calc_method_string . '>';
			
			$update_table_of_rates_sql = "
				UPDATE
					" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
				SET
					table_of_rates = '" . $table_of_rates . "'
				WHERE
					config_id = '" . $config_id . "'
				AND
					method = '" . $method . "'
				AND
					region = '" . $region . "';";
			
			$update_table_of_rates_result = $db->Execute($update_table_of_rates_sql);
			
			$load_region_config_result->MoveNext();
		}
		
		$load_method_config_result->MoveNext();
	}
	
	
	// Remove deprecated calculation method column /////////////////////////////////////////////////
	$remove_calc_method_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_METHOD_CONFIGS . "
		DROP
			calc_method";
	
	$remove_calc_method_column_result = $db->Execute($remove_calc_method_column_sql);
	
	
	// Add region admin titles table ///////////////////////////////////////////////////////////////
	$add_region_admin_titles_table_sql = "
		CREATE TABLE
			`" . TABLE_ADVANCED_SHIPPER_REGION_ADMIN_TITLES . "`
			(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`region` INT UNSIGNED NOT NULL,
			`language_id` INT UNSIGNED NOT NULL,
			`title` TEXT DEFAULT NULL,
			PRIMARY KEY ( 
				`config_id`, 
				`method`,
				`region`,
				`language_id` 
			)
		);";
	
	$add_region_admin_titles_table_result = $db->Execute($add_region_admin_titles_table_sql);
	
	
	// Copy existing region titles to region admin titles table ////////////////////////////////////
	$load_region_titles_sql = "
		SELECT
			asrt.config_id,
			asrt.method,
			asrt.region,
			asrt.language_id,
			asrt.title
		FROM
			" . TABLE_ADVANCED_SHIPPER_REGION_TITLES . " asrt;";
	
	$load_region_titles_result = $db->Execute($load_region_titles_sql);
	
	while (!$load_region_titles_result->EOF) {
	
		$insert_region_admin_title_sql = "
			INSERT INTO
				" . TABLE_ADVANCED_SHIPPER_REGION_ADMIN_TITLES . "
				(
				config_id,
				method,
				region,
				language_id,
				title
				)
			VALUES
				(
				'" . $load_region_titles_result->fields['config_id'] . "',
				'" . $load_region_titles_result->fields['method'] . "',
				'" . $load_region_titles_result->fields['region'] . "',
				'" . $load_region_titles_result->fields['language_id'] . "',
				'" . $db->prepare_input($load_region_titles_result->fields['title']) . "'
				);";
		
		$insert_region_admin_title_result = $db->Execute($insert_region_admin_title_sql);
		
		$load_region_titles_result->MoveNext();
	}
	
	
	// Add the UPS config table ////////////////////////////////////////////////////////////////////
	$add_region_ups_configs_table_sql = "
		CREATE TABLE IF NOT EXISTS
			`" . TABLE_ADVANCED_SHIPPER_REGION_UPS_CONFIGS . "`
			(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`region` INT UNSIGNED NOT NULL,
			`source_country` INT UNSIGNED NOT NULL,
			`source_postcode` VARCHAR(15) NOT NULL,
			`pickup_method` VARCHAR(4) NOT NULL,
			`packaging` VARCHAR(4) NOT NULL,
			`delivery_type` VARCHAR(3) NOT NULL,
			`shipping_service_1dm` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_1dml` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_1da` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_1dal` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_1dapi` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_1dp` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_1dpl` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_2dm` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_2dml` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_2da` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_2dal` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_3ds` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_gnd` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_std` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_xpr` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_xprl` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_xdm` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_xdml` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_xpd` TINYINT NOT NULL DEFAULT '1',
			`shipping_service_wxs` TINYINT NOT NULL DEFAULT '1',
			PRIMARY KEY ( 
				`config_id`,
				`method`,
				`region`
			)
		);";
	
	$add_region_ups_configs_table_result = $db->Execute($add_region_ups_configs_table_sql);
	
	
	// Add the USPS config table ///////////////////////////////////////////////////////////////////
	$add_region_usps_configs_table_sql = "
		CREATE TABLE IF NOT EXISTS
			`" . TABLE_ADVANCED_SHIPPER_REGION_USPS_CONFIGS . "`
			(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`region` INT UNSIGNED NOT NULL,
			`user_id` VARCHAR(20) NOT NULL,
			`server` VARCHAR(1) NOT NULL DEFAULT 't',
			`source_country` INT UNSIGNED NOT NULL,
			`source_postcode` VARCHAR(15) NOT NULL,
			`machinable` TINYINT NOT NULL DEFAULT '1',
			`display_transit_time` TINYINT NOT NULL DEFAULT '1',
			`domestic_express` TINYINT NOT NULL DEFAULT '1',
			`domestic_priority` TINYINT NOT NULL DEFAULT '1',
			`domestic_first_class` TINYINT NOT NULL DEFAULT '1',
			`domestic_retail_ground` TINYINT NOT NULL DEFAULT '1',
			`domestic_media` TINYINT NOT NULL DEFAULT '1',
			`domestic_bpm` TINYINT NOT NULL DEFAULT '1',
			`domestic_library` TINYINT NOT NULL DEFAULT '1',
			`international_geg` TINYINT NOT NULL DEFAULT '1',
			`international_geg_d` TINYINT NOT NULL DEFAULT '0',
			`international_geg_ndr` TINYINT NOT NULL DEFAULT '1',
			`international_geg_ndnr` TINYINT NOT NULL DEFAULT '1',
			`international_geg_e` TINYINT NOT NULL DEFAULT '1',
			`international_emi` TINYINT NOT NULL DEFAULT '1',
			`international_emi_fre` TINYINT NOT NULL DEFAULT '1',
			`international_emi_lfre` TINYINT NOT NULL DEFAULT '0',
			`international_pmi` TINYINT NOT NULL DEFAULT '1',
			`international_pmi_sfrb` TINYINT NOT NULL DEFAULT '1',
			`international_pmi_mfrb` TINYINT NOT NULL DEFAULT '1',
			`international_pmi_lfrb` TINYINT NOT NULL DEFAULT '1',
			`international_pmi_dfrb` TINYINT NOT NULL DEFAULT '0',
			`international_pmi_lvfrb` TINYINT NOT NULL DEFAULT '0',
			`international_pmi_fre` TINYINT NOT NULL DEFAULT '1',
			`international_pmi_sfre` TINYINT NOT NULL DEFAULT '1',
			`international_pmi_pfre` TINYINT NOT NULL DEFAULT '0',
			`international_pmi_gcfre` TINYINT NOT NULL DEFAULT '0',
			`international_pmi_lfre` TINYINT NOT NULL DEFAULT '0',
			`international_fcmi_le` TINYINT NOT NULL DEFAULT '1',
			`international_fcmi_p` TINYINT NOT NULL DEFAULT '1',
			`international_fcmi_l` TINYINT NOT NULL DEFAULT '1',
			PRIMARY KEY ( 
				`config_id`,
				`method`,
				`region`
			)
		);";
	
	$add_region_usps_configs_table_result = $db->Execute($add_region_usps_configs_table_sql);
	
	
	// Update the handling fee column of the region configs table //////////////////////////////////
	$update_handling_fee_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		CHANGE
			`handling_fee` `surcharge` TEXT NULL;";
	
	$update_handling_fee_column_result = $db->Execute($update_handling_fee_column_sql);
	
	
	// Remove the show rate calculations column of the method configs table ////////////////////////
	$drop_show_rate_calc_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_METHOD_CONFIGS . "
		DROP 
			`show_rate_calc`;";
	
	$drop_show_rate_calc_column_result = $db->Execute($drop_show_rate_calc_column_sql);
	
	
	// Add region surcharge titles table ///////////////////////////////////////////////////////////
	$add_region_surcharge_titles_table_sql = "
		CREATE TABLE
			`" . TABLE_ADVANCED_SHIPPER_REGION_SURCHARGE_TITLES . "`
			(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`region` INT UNSIGNED NOT NULL,
			`language_id` INT UNSIGNED NOT NULL,
			`title` TEXT DEFAULT NULL,
			PRIMARY KEY ( 
				`config_id`, 
				`method`,
				`region`,
				`language_id` 
			)
		);";
	
	$add_region_surcharge_titles_table_result = $db->Execute($add_region_surcharge_titles_table_sql);
	
	
	// Create default surcharge titles /////////////////////////////////////////////////////////////
	$load_regions_sql = "
		SELECT
			asrt.config_id,
			asrt.method,
			asrt.region,
			asrt.language_id,
			asrt.title
		FROM
			" . TABLE_ADVANCED_SHIPPER_REGION_TITLES . " asrt;";
	
	$load_regions_result = $db->Execute($load_regions_sql);
	
	while (!$load_regions_result->EOF) {
	
		$insert_region_surcharge_title_sql = "
			INSERT INTO
				" . TABLE_ADVANCED_SHIPPER_REGION_SURCHARGE_TITLES . "
				(
				config_id,
				method,
				region,
				language_id,
				title
				)
			VALUES
				(
				'" . $load_regions_result->fields['config_id'] . "',
				'" . $load_regions_result->fields['method'] . "',
				'" . $load_regions_result->fields['region'] . "',
				'" . $load_regions_result->fields['language_id'] . "',
				NULL
				);";
		
		$insert_region_surcharge_title_result = $db->Execute($insert_region_surcharge_title_sql);
		
		$load_regions_result->MoveNext();
	}
	
	
	// Add option for method sorting ///////////////////////////////////////////////////////////////
	$add_method_sorting_setting_sql = "
		INSERT INTO
			" . TABLE_CONFIGURATION . "
			(
			configuration_title,
			configuration_key,
			configuration_value,
			configuration_description,
			configuration_group_id,
			sort_order,
			set_function,
			date_added
			)
		VALUES
			(
			'Method Sorting',
			'MODULE_ADVANCED_SHIPPER_METHOD_SORT_ORDER',
			'By Shipping Date (for methods which have a shipping date selected), then by Cost - lowest to highest.',
			'How should the methods be sorted before being displayed to the customer?',
			'6',
			'0',
			'zen_cfg_select_option(array(\'By Shipping Date (for methods which have a shipping date selected), then by Cost - lowest to highest.\', \'By Shipping Date (for methods which have a shipping date selected), then by Cost - highest to lowest.\', \'By Shipping Date (for methods which have a shipping date selected), then by the Order of the methods, as defined in the Admin.\', \'By Cost - lowest to highest. (Ordering will be done based on Cost only, any Shipping Date selected will be ignored when ordering the methods).\', \'By Cost - highest to lowest. (Ordering will be done based on Cost only, any Shipping Date selected will be ignored when ordering the methods).\', \'By the Order of the methods, as defined in the Admin. (Ordering will be done based on the Admin Order only, any Shipping Date selected will be ignored when ordering the methods).\'), ',
			now()
			);";
	
	$add_method_sorting_setting_result = $db->Execute($add_method_sorting_setting_sql);
	
	// Add the Manufacturer Selections table ///////////////////////////////////////////////////////
	$add_method_manufacturers_table_sql = "
		CREATE TABLE IF NOT EXISTS
			`" . TABLE_ADVANCED_SHIPPER_METHOD_MANUFACTURERS . "`
		(
			`config_id` INT UNSIGNED NOT NULL,
			`method` INT UNSIGNED NOT NULL,
			`manufacturer_order` INT UNSIGNED NOT NULL,
			`manufacturer_id` INT UNSIGNED NOT NULL,
			PRIMARY KEY ( 
				`config_id`,
				`method`,
				`manufacturer_order`,
				`manufacturer_id`
			)
		);";
	
	$add_method_manufacturers_table_result = $db->Execute($add_method_manufacturers_table_sql);
	
	// Add the new maximum weight per package column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`max_weight_per_package` FLOAT UNSIGNED NULL
		AFTER
			`table_of_rates`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the new packaging weights column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`packaging_weights` TEXT NULL
		AFTER
			`max_weight_per_package`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	// Add the new total up price inc tax column to the region configs table
	$add_column_sql = "
		ALTER TABLE
			" . TABLE_ADVANCED_SHIPPER_REGION_CONFIGS . "
		ADD
			`total_up_price_inc_tax` TINYINT NOT NULL DEFAULT '2'
		AFTER
			`rate_limits_inc`;";
	
	$add_column_result = $db->Execute($add_column_sql);
	
	advshipperAddCustomProductFieldSupport4_2_0();
	
	advshipperAddVersionCheckingOptions4_2_0();
	
	advshipperAddMethodEnabledColumn4_2_0();
	
	advshipperAddFedExSupport5_0_0();
	}
}

// Update version number of database tables ////////////////////////////////////////////////////////
$update_db_version_number_sql = "
	UPDATE
		" . TABLE_CONFIGURATION . "
	SET
		configuration_title = '</b></fieldset><a href=\"http://dev.ceon.net\" target=\"_blank\"><img src=\"" .
			DIR_WS_IMAGES . "ceon-button-logo.png\" alt=\"Made by Ceon. &copy; 2007-" . (date('Y') > 2012 ? date('Y') : 2012) . " Ceon\" align=\"right\" style=\"margin: 1em 0.2em; border: none;\" /></a><br />Module &copy; 2007-" . (date('Y') > 2012 ? date('Y') : 2012) . " Ceon<p style=\"display: none\">',
		configuration_value = '" . MODULE_ADVANCED_SHIPPER_VERSION_NO . "',
		last_modified = NOW()
	WHERE
		configuration_key = 'MODULE_ADVANCED_SHIPPER_MADE_BY_CEON';";

$update_db_version_number_result = $db->Execute($update_db_version_number_sql);

// Finished!
$advshipper_upgraded = true;

// Reset version check status now that the module has been updated!
if (isset($_SESSION['ceon_advanced_shipper_version_checker_response'])) {
	unset($_SESSION['ceon_advanced_shipper_version_checker_response']);
}

?>