<?php

/**
 * Ceon URI Mappings Manager Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMM.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Version class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMVersion.php');


// {{{ CeonUMM

/**
 * Main Ceon URI Mappings Manager class. Installs/upgrades the software, performs configuration checks, runs the
 * appropriate action and builds the interface output.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMM extends CeonUMMVersion
{
	// {{{ properties
	
	/**
	 * Holds the information about the panels this software uses. The active panel will also have its output
	 * content stored.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_panels_info = array();
	
	/**
	 * The ID of the active panel.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_active_panel_id = array();
	
	/**
	 * Maintains a list of any errors encountered in the current configuration, installation or upgrade or for the
	 * executed action.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_error_messages = array();
	
	/**
	 * The version of Ceon URI Mapping currently installed.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_ceon_uri_mapping_version = null;
	
	/**
	 * Should AJAX be used for progress indicators?
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_use_ajax_progress_indicators = null;
	
	/**
	 * How often (in milliseconds) should AJAX progress indicators update their status? 
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_ajax_progress_refresh_sec = null;
	
	/**
	 * How often (in seconds) should standard progress indicators "update their status" (Resubmit)? 
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_standard_progress_refresh_sec = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class. Instigates the installation/upgrade functionality if necessary, checks
	 * the configuration and runs the appropriate action.
	 * 
	 * @access  public
	 */
	public function __construct($load_config = false)
	{
		// Set the flag so that the configuration isn't autoloaded by the admin superclass - it may not exist or
		// yet or it may be out of date!
		parent::__construct($load_config);
		
		if (!$this->_lookUpInstalledVersion()) {
			// Error occurred when attempting to get the version number
		}
		
		// Must make sure a compatible edition of Ceon URI Mapping is installed as this software depends upon UMM
		// edition functionality
		$compatible_uri_mapping_edition_installed = $this->_checkCeonURIMappingEdition();
		
		if (!$compatible_uri_mapping_edition_installed) {
			$this->_buildVersionErrorOutput();
			
			$this->_buildFooter();
			
			return;
		}
		
		// If the configuration options are being submitted then the configuration has previously been loaded and
		// tested. If not, it must be loaded and checked now
		if (!isset($_POST['ajax_progress_refresh_sec'])) {
			$installed_and_up_to_date = $this->_checkInstalledAndUpToDate();
			
			if (!$installed_and_up_to_date) {
				$this->_buildVersionErrorOutput();
				
				$this->_buildFooter();
				
				return;
			}
			
			$this->_loadConfig();
		}
		
		$this->_run();
	}
	
	// }}}
	
	
	// {{{ _run()
	
	/**
	 * Main method which performs the required action(s) and builds the output.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _run()
	{
		// Determine the action to be undertaken, by checking which panel is the active panel
		$panel = (isset($_GET['tab']) ? $_GET['tab'] : '');
		
		switch ($panel) {
			case 'create-manage':
				$class = 'CeonUMMCreateManage';
				$this->_active_panel_id = 'create-manage-panel';
				break;
			case 'import-old-uris':
				$class = 'CeonUMMImportOldURIs';
				$this->_active_panel_id = 'import-old-uris-panel';
				break;
			case 'historical-uris':
				$class = 'CeonUMMHistoricalURIs';
				$this->_active_panel_id = 'historical-uris-panel';
				break;
			case 'alternate-uris':
				$class = 'CeonUMMAlternateURIs';
				$this->_active_panel_id = 'alternate-uris-panel';
				break;
			default:
				$class = 'CeonUMMDashboard';
				$this->_active_panel_id = 'dashboard-panel';
				break;
		}
		
		/**
		 * Load in the appropriate Ceon URI Mappings Manager class for the action
		 */
		require_once(DIR_WS_CLASSES . 'class.' . $class . '.php');
		
		// Perform the required action now and get its output, to be added to its panel //////
		$mappings_manager = new $class($this->_use_ajax_progress_indicators, $this->_ajax_progress_refresh_sec,
			$this->_standard_progress_refresh_sec);
		
		$panel_content = $mappings_manager->getOutput();
		
		$errors = $mappings_manager->getErrorMessages();
		
		foreach ($errors as $error_message) {
			$this->_error_messages[] = $error_message;
		}
		
		
		// Build the information required to build the software's interface //////
		$this->_setPanelInfo();
		
		$this->_buildAndAddPanels($panel_content);
		
		
		// Actually output the software's interface now //////
		
		// Display a count of the number of errors (if any) at the top of the page
		$num_errors = 0;
		
		foreach ($this->_error_messages as $error_message) {
			if (is_array($error_message)) {
				$num_errors += sizeof($error_message);
			} else {
				$num_errors++;
			}
		}
		
		if ($num_errors > 0) {
			if ($num_errors == 1) {
				$this->_output .= '<p class="ErrorIntro">' . TEXT_ERROR_IN_CONFIG . '</p>';
			} else if ($num_errors > 0 ) {
				$this->_output .= sprintf('<p class="ErrorIntro">' . TEXT_ERRORS_IN_CONFIG, $num_errors) . '</p>';
			}
		}
		
		$this->_buildTabbedPanelMenu();
		
		$this->_buildPanels();
		
		$this->_buildFooter();
	}
	
	// }}}
	
	
	// {{{ _setPanelInfo()
	
	/**
	 * Simply sets the info for the panels for the software, to be used to build the panels in the tabbed
	 * interface.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _setPanelInfo()
	{
		$panel_page_uri = zen_href_link(FILENAME_CEON_UMM, '', 'NONSSL');
		
		$this->_panels_info[] = array(
			'id' => 'dashboard-panel',
			'title' => TEXT_TAB_DASHBOARD,
			'uri' => $panel_page_uri,
			'content' => ''
			);
		
		
		$panel_page_uri = zen_href_link(FILENAME_CEON_UMM, 'tab=create-manage', 'NONSSL');
		
		$this->_panels_info[] = array(
			'id' => 'create-manage-panel',
			'title' => TEXT_TAB_CREATE_MANAGE,
			'uri' => $panel_page_uri,
			'content' => ''
			);
		
		
		$panel_page_uri = zen_href_link(FILENAME_CEON_UMM, 'tab=import-old-uris', 'NONSSL');
		
		$this->_panels_info[] = array(
			'id' => 'import-old-uris-panel',
			'title' => TEXT_TAB_IMPORT_OLD_URIS,
			'uri' => $panel_page_uri,
			'content' => ''
			);
		
		
		$panel_page_uri = zen_href_link(FILENAME_CEON_UMM, 'tab=historical-uris', 'NONSSL');
		
		$this->_panels_info[] = array(
			'id' => 'historical-uris-panel',
			'title' => TEXT_TAB_HISTORICAL_URIS,
			'uri' => $panel_page_uri,
			'content' => ''
			);
		
		
		$panel_page_uri = zen_href_link(FILENAME_CEON_UMM, 'tab=alternate-uris', 'NONSSL');
		
		$this->_panels_info[] = array(
			'id' => 'alternate-uris-panel',
			'title' => TEXT_TAB_ALTERNATE_URIS,
			'uri' => $panel_page_uri,
			'content' => ''
			);
	}
	
	// }}}
	
	
	// {{{ _buildAndAddPanels()
	
	/**
	 * Builds the panels this software uses, adding each to the list of panels in the tabbed interface and setting
	 * the content for the active panel.
	 *
	 * @access  protected
	 * @param   string     $active_panel_content   The content to be assigned to the active panel.
	 * @return  none
	 */
	protected function _buildAndAddPanels($active_panel_content)
	{
		foreach ($this->_panels_info as $panel) {
			$panel_content = '';
			
			if ($panel['id'] == $this->_active_panel_id) {
				$panel_content =
					$this->_buildConfigPanel($panel['id'], $panel['title'], '', $active_panel_content);
			}
			
			$this->_addPanel($panel['id'], $panel['title'], $panel['uri'], $panel_content);
		}
	}
	
	// }}}
	
	
	// {{{ _getSelectedPanelID()
	
	/**
	 * Simply returns the ID of the active panel.
	 *
	 * @access  protected
	 * @return  string     The ID of the selected panel.
	 */
	protected function _getSelectedPanelID()
	{
		return $this->_active_panel_id;
	}
	
	// }}}
	
	
	// {{{ _checkInstalledAndUpToDate()
	
	/**
	 * Checks that the module is installed and that the database and configuration are up to date.
	 *
	 * @access  protected
	 * @return  boolean   True if the module is installed and the database and configuration are up to date, false
	 *                    otherwise.
	 */
	protected function _checkInstalledAndUpToDate()
	{
		global $messageStack;
		
		if ((is_null($this->_installed_version) || $this->_installed_version != $this->_version) ||
				isset($_GET['check-config'])) {
			// Main variable holds list of installation/upgrade errors (if any)
			$version_errors = array();
			
			// Instantiate and run the installation/upgrade class
			require_once('class.CeonUMMInstallOrUpgrade.php');
			
			$install_or_upgrade = new CeonUMMInstallOrUpgrade($this->_version, $this->_installed_version);
			
			if (sizeof($install_or_upgrade->error_messages) > 0) {
				$this->_error_messages = $install_or_upgrade->error_messages;
				
				return false;
				
			} else {
				// Version has been brought up to date
				if (is_null($this->_installed_version)) {
					$messageStack->add(sprintf(SUCCESS_MODULE_INSTALLED, $this->_version), 'success');
				} else {
					$messageStack->add(sprintf(SUCCESS_DATABASE_AND_CONFIG_UPDATED, $this->_version), 'success');
				}
				
				$this->_installed_version = $this->_version;
			}
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _checkCeonURIMappingEdition()
	
	/**
	 * Checks that the edition of Ceon URI Mapping installed is compatible with this software.
	 *
	 * @access  protected
	 * @return  boolean   True if a compatible edition of Ceon URI Mapping is installed, false otherwise.
	 */
	protected function _checkCeonURIMappingEdition()
	{
		global $db;
		
		// Just use a simple method of instantiating Ceon URI Mapping in a temporary object and checking its
		// edition property
		$version_file = DIR_WS_CLASSES . 'class.CeonURIMappingVersion.php';
		
		if (!file_exists($version_file)) {
			// Must be old version of Ceon URI Mapping
			$this->_error_messages[] = TEXT_CEON_URI_MAPPING_UMM_EDITION_REQUIRED_INSTALL;
			
			return false;
		}
		
		require_once($version_file);
		
		$temp = new CeonURIMappingVersion();
		
		if (!method_exists($temp, 'getVersion')) {
			$this->_error_messages[] = TEXT_CEON_URI_MAPPING_UMM_EDITION_LATEST_VERSION_REQUIRED;
			
			return false;
		}
		
		$this->_ceon_uri_mapping_version = $temp->getVersion();
		
		// Need version 4.2.0 or above
		if ((int) substr($this->_ceon_uri_mapping_version, 0, 1) < 4 || 
				substr($this->_ceon_uri_mapping_version, 0, 3) == '4.0') {
			$this->_error_messages[] = TEXT_CEON_URI_MAPPING_UMM_EDITION_LATEST_VERSION_REQUIRED;
			
			return false;
		}
		
		if (strpos($temp->getEditionCode(), 'UMM') === false) {
			$this->_error_messages[] = sprintf(TEXT_CEON_URI_MAPPING_UMM_EDITION_REQUIRED_UPGRADE,
				TEXT_CEON_URI_MAPPING_STANDARD_EDITION, $this->_ceon_uri_mapping_version);
			
			return false;
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _buildVersionErrorOutput()
	
	/**
	 * Builds the HTML for the error messages to be displayed to the user when a problem occurs in the installation
	 * or upgrade process.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildVersionErrorOutput()
	{
		foreach ($this->_error_messages as $version_error) {
			$this->_output .=  '<p class="ErrorIntro">' . $version_error . '</p>' . "\n";
		}
	}
	
	// }}}
	
	
	// {{{ _loadConfig()
	
	/**
	 * Loads the configuration for the software from the database.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _loadConfig()
	{
		global $db;
		
		// Only one config currently supported so its ID is hard-coded in the following SQL
		$load_config_sql = "
			SELECT
				version,
				use_ajax_progress_indicators,
				ajax_progress_refresh_sec,
				standard_progress_refresh_sec,
				automatic_version_checking
			FROM
				" . TABLE_CEON_UMM_CONFIGS . " 
			WHERE
				id = '1';";
		
		$load_config_result = $db->Execute($load_config_sql);
		
		if ($load_config_result->EOF) {
			
		} else {
			$this->_installed_version = $load_config_result->fields['version'];
			
			$this->_use_ajax_progress_indicators =
				($load_config_result->fields['use_ajax_progress_indicators'] == 1 ? true : false);
			
			$this->_ajax_progress_refresh_sec = $load_config_result->fields['ajax_progress_refresh_sec'];
			
			$this->_standard_progress_refresh_sec = $load_config_result->fields['standard_progress_refresh_sec'];
			
			$this->_automatic_version_checking =
				($load_config_result->fields['automatic_version_checking'] == 1 ? true : false);
		}
	}
	
	// }}}
}

// }}}
