<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate EZ-Page URIs Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2019 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageEZPages.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Listing class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMListing.php');


// {{{ CeonUMMCreateManageEZPages

/**
 * Handles URI management/auto-generation functionality for EZ-Pages.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2019 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageEZPages extends CeonUMMListing
{
	// {{{ properties
	
	/**
	 * The mapping templates being used by EZ-Pages for this instance at runtime. Indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_page_mapping_templates = null;
	
	/**
	 * The information about the EZ-Pages.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_pages = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' . 'ceon_umm_create_manage_ez_pages.php');
		
		if (!defined('TEXT_EZ_PAGES_MAPPINGS_TITLE') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' . 'ceon_umm_create_manage_ez_pages.php');
		}
		
		parent::__construct();
		
		// Run the actual functionality and build the output
		$this->_buildEZPagesListing();
	}
	
	// }}}
	
	
	// {{{ _buildEZPagesListing()
	
	/**
	 * Builds a list of all the EZ-Pages for the store, textfields to enter/change any mappings (en masse), and a
	 * gadget to autogenerate URIs for all the selected pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildEZPagesListing()
	{
		global $db, $languages, $num_languages, $ceon_uri_mapping_demo, $messageStack;
		
		$content = '';
		
		// Get the mapping template(s) for EZ-Pages pages
		$this->_page_mapping_templates = array();
		
		for ($i = 0; $i < $num_languages; $i++) {
			$this->_page_mapping_templates[$languages[$i]['id']] =
				$this->getObjectURIMappingTemplate('ez-page', $languages[$i]['id']);
		}
		
		// Initialise the main environment/data variables
		$this->_selected_items = array();
		$this->_mappings = array();
		
		// Get the information for the EZ-Pages
		$this->_pages = array();
		
		$ez_pages_result = $db->Execute("
			SELECT
				e.pages_id,
				languages_id,
				e.pages_title
            FROM
				" . TABLE_EZPAGES . " e
            WHERE
				1 = 1
            ORDER BY
				e.pages_title");
		
		while (!$ez_pages_result->EOF) {
			if (!isset($this->_pages[$ez_pages_result->fields['pages_id']])) {
				$this->_pages[$ez_pages_result->fields['pages_id']] = array();
			}
			
			$this->_pages[$ez_pages_result->fields['pages_id']][$ez_pages_result->fields['languages_id']] =
				$ez_pages_result->fields['pages_title'];
			
			$ez_pages_result->MoveNext();
		}
		
		// Has the page been submitted, either for auto-generation or to have the auto-generation settings set or
		// the URIs manually updated?
		if (isset($this->_posted_data['auto-generate-selected-pages'])) {
			$this->_autogenerateEZPagesMappings();
		} else if (isset($this->_posted_data['save-uri-mappings'])) {
			$this->_processEZPagesListingFormSubmission();
		} else {
			// Page being loaded for first time, load information and set up defaults
			
			// Get the current URI mapping(s) for each page, if any
			foreach ($this->_pages as $page_id => $page_titles) {
				$this->_mappings[$page_id] = array();
				
				$columns_to_retrieve = array(
					'uri',
					'language_id'
					);
				
				$selections = array(
					'current_uri' => 1,
					'main_page' => FILENAME_EZPAGES,
					'associated_db_id' => $page_id
					);
				
				$page_mappings_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections);
				
				while (!$page_mappings_result->EOF) {
					$this->_mappings[$page_id][$page_mappings_result->fields['language_id']] =
						$page_mappings_result->fields['uri'];
					
					$page_mappings_result->MoveNext();
				}
			}
			
			// Default to all listed pages being selected
			$this->_select_all = true;
			
			foreach ($this->_pages as $page_id => $page_titles) {
				$this->_selected_items[] = $page_id;
			}
		}
		
		// All data has been gathered, build the page //////
		$panel_title = TEXT_EZ_PAGES_MAPPINGS_TITLE;
		
		$content .= '<h1>' . $panel_title . '</h1>' . "\n";
		
		$content .= '<p>' . TEXT_EZ_PAGES_MAPPINGS_INTRO . '</p>' . "\n";
		
		$content .= '<ul>' . "\n";
		
		$content .= '<li>' . TEXT_EZ_PAGES_MAPPINGS_AUTOGENERATION_INTRO . '</li>' . "\n";
		
		$content .= '<li>' . TEXT_EZ_PAGES_MAPPINGS_MANUAL_EDITING_INTRO . '</li>' . "\n";
		
		$content .= '<li>' . TEXT_EZ_PAGES_MAPPINGS_MANUAL_EDITING_DELETE . '</li>' . "\n";
		
		$content .= '</ul>' . "\n";
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action .
			'" method="post" name="page-listing-form" id="page-listing-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Add the default button for the form, but align it offscreen. This means using the return key (pressing
		// enter) will submit the entered URIs rather than run the auto-generation
		$save_mappings_button_title = TEXT_SAVE_URI_MAPPINGS_BUTTON;
		$save_mappings_button_desc = TEXT_SAVE_URI_MAPPINGS_BUTTON_HOVER_INFO;
		
		$content .= '<input type="submit" class="SaveURIMappingsButton" name="save-uri-mappings"' .
			' id="save-uri-mappings" value="' . $save_mappings_button_title .
			'" style="position: absolute; left: -100%;" />' . "\n";
		
		
		// Add the autogenerate controls for EZ-Pages //////
		$content .= '<div class="AutogenerateControlsWrapper EZPages" id="autogenerate-controls-top">' . "\n";
		
		$content .= '<fieldset class="AutogenerateControls"><legend>' . TEXT_AUTOGENERATE_MAPPINGS . '</legend>' .
			"\n";
		
		$content .= '<p class="AutogenerationOptions">' . "\n";
		
		$content .= '<input type="radio" name="autogenerate-page-mappings" value="exc-products"' .
			' id="autogenerate-page-mappings"' . (1 == 1 ? ' checked="checked"' : '') . ' />' . "\n"; 
		
		$content .= '<label for="autogenerate-page-mappings">' . TEXT_SELECTED_PAGES . '</label>' . '</p>' . "\n";
		
		// Main button to generate URI mappings for all selected pages
		$button_title = TEXT_AUTOGENERATE_URI_MAPPINGS_BUTTON;
		$button_desc = TEXT_AUTOGENERATE_EZ_PAGES_URI_MAPPINGS_BUTTON_HOVER_INFO;
		
		$content .= '<input type="submit" class="AutogenerateURIMappingsButton"' .
			' name="auto-generate-selected-pages" id="auto-generate-selected-pages"' .
			' value="' . $button_title . '" title="' . $button_desc . '" />' . "\n"; 
		
		$content .= '</fieldset>' . "\n";
		
		$content .= '</div>' . "\n";
		
		
		// Build the info for the EZ-Pages //////
		
		// Build the list of mapping templates for the other pages
		if ($num_languages == 1) {
			$content .= '<p class="MainMappingTemplatesIntro EZPages"><label>' .
				TEXT_MAPPING_TEMPLATE_FOR_EZ_PAGES . '&nbsp;' . '</label>' . '</p>' . "\n";
		} else {
			$content .= '<p class="MainMappingTemplatesIntro EZPages"><label>' .
				TEXT_MAPPING_TEMPLATES_FOR_EZ_PAGES . '&nbsp;' . '</label>' . '</p>' . "\n";
		}
		
		$content .= '<ul class="MainMappingTemplates">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li>';
			
			$content .= '<span class="MappingLanguageIcon">' . zen_image(DIR_WS_CATALOG_LANGUAGES .
				$languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) .
				'</span>';
			
			$content .= $this->_addTooltipDescriptionsOfPlacementTags('ez-page',
				$this->_page_mapping_templates[$languages[$i]['id']]) . '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		// Build the output for all of the EZ-Pages
		$params = zen_get_all_get_params(array('listing', 'cat-id'));
		
		$back_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p class="BackToSection TopLink">' . zen_image(DIR_WS_IMAGES . 'icons/ceon-umm-back-to.png') .
			' <a href="' . $back_link . '">' . TEXT_BACK_TO_CREATE_MANAGE_URIS . '</a></p>' . "\n";
		

		$select_all_text = addslashes(TEXT_TICK_TO_SELECT_ALL_PAGES);
		$deselect_all_text = addslashes(TEXT_TICK_TO_DESELECT_ALL_PAGES);
		
		$content .= <<<TABLE_JS
<script type="text/javascript">
<!--
function SelectAllPages(toggle_el, el_id)
{
	select_or_deselect = toggle_el.checked
	
	form_el = document.getElementById('page-listing-form');
	
	for (var t, i = 0; t = form_el.elements[el_id][i++]; t.checked=select_or_deselect);
	
	if (select_or_deselect) {
		toggle_el.title = '$deselect_all_text';
	} else {
		toggle_el.title = '$select_all_text';
	}
}
// -->
</script>
TABLE_JS;
		
		$content .= '<table cellpadding="0" cellspacing="0" id="page-listing" class="Listing">' .
			"\n";
		
		// Build header row
		$content .= '<tr>' . "\n";
		
		$content .= '<th id="checkboxes-col">' .
			'<input type="checkbox" name="select-all" value="1"' .
			' onclick="javascript:SelectAllPages(this, \'page[]\');"' .
			($this->_select_all ? ' checked="checked"' : '') . ' title="' . TEXT_TICK_TO_DESELECT_ALL_PAGES .
			'" />' . '</th>' . "\n";
		
		$content .= '<th>' . TEXT_EZ_PAGE . '</th>' . "\n";
		
		$content .= '</tr>' . "\n";
		
		
		// Build page rows
		$num_pages = sizeof($this->_pages);
		
		$page_i = 0;
		
		foreach ($this->_pages as $page_id => $page_titles) {
			$content .= '<tr class="' . (($page_i % 2 == 0) ? 'EvenRow' : 'OddRow') . '">' . "\n";
			
			$content .= '<td>' . "\n";
			
			$content .= '<input type="checkbox" name="page[]" value="' . $page_id . '"' .
				(in_array($page_id, $this->_selected_items) ? ' checked="checked"' : '') .
				' title="' . TEXT_TICK_TO_SELECT_EZ_PAGE . '" class="Textfield" />' . "\n";
			
			$content .= '</td>' . "\n";
			
			
			// Build page type name and current URI mapping(s) for this page
			$content .= '<td>' . "\n";
			
			if (isset($page_titles[$_SESSION['languages_id']])) {
				$ez_page_title = $page_titles[$_SESSION['languages_id']];
			} else {
				// No EZ-Page title for admin's language. Use first page title defined
				foreach ($page_titles as $page_title) {
					$ez_page_title = $page_title;
					
					break;
				}
			}
			
			$content .= '<h3>' . htmlentities($ez_page_title, ENT_COMPAT, CHARSET) . '</h3>' . "\n";
			
			
			// Build the textfield(s) to enter/update/delete the mapping(s) for this page
			$content .= '<ul class="PageMappings">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				$content .= '<input type="text" name="mappings[' . $page_id. '][' .
					$languages[$i]['id'] . ']" value="' . (!empty($this->_mappings[$page_id][$languages[$i]['id']]) ? $this->_mappings[$page_id][$languages[$i]['id']] : '') . '"' .
					' class="Textfield" />' . "\n";
				
				if (isset($this->_error_messages['mappings']) &&
						isset($this->_error_messages['mappings'][$page_id]) &&
						isset($this->_error_messages['mappings'][$page_id][$languages[$i]['id']])) {
					
					$error_message = $this->_error_messages['mappings'][$page_id][$languages[$i]['id']];
					
					$content .= '<p class="MappingError">' . $error_message['message'];
					
					if (isset($error_message['link'])) {
						$content .= ' <a href="' . $error_message['link'] . '" target="_blank">' .
							$error_message['link_title'] . '</a>';
					}
					
					if (isset($error_message['edit_link'])) {
						$content .= ' - <a href="' . $error_message['edit_link'] . '" target="_blank">' .
							$error_message['edit_link_title'] . '</a>';
					}
					
					$content .= '</p>' . "\n";
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			$content .= '</td>' . "\n";
			
			
			$content .= '</tr>' . "\n";
			
			$page_i++;
		}
		
		$content .= '</table>' . "\n";
		
		
		// Main button to save any entered/modified URIs
		$content .= '<p class="SaveURIMappingsButton">' . "\n";
		
		$content .= '<input type="submit" class="SaveURIMappingsButton" name="save-uri-mappings"' .
			' id="save-uri-mappings" value="' . $save_mappings_button_title . '" title="' .
			$save_mappings_button_desc . '" />' . "\n"; 
		
		$content .= '</p>';
		
		$content .= '<div class="SpacerSmall"></div>' . "\n";
		
		
		// Build the back to top link
		$content .= '<p class="BackToTop"><a href="#top">' . TEXT_BACK_TO_TOP .
			zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to-top.png') . '</a></p>' . "\n";
		
		
		// Build the "back to create/manage URIs" link
		$content .= '<p class="BackToSection">' . zen_image(DIR_WS_IMAGES . 'icons/ceon-umm-back-to.png') .
			' <a href="' . $back_link . '">' . TEXT_BACK_TO_CREATE_MANAGE_URIS . '</a></p>' . "\n";
		
		
		$content .= '</form>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _autogenerateEZPagesMappings()
	
	/**
	 * Auto-generate the mappings for all the selected EZ-Pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _autogenerateEZPagesMappings()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		$this->_handleItemSelections('page');
		
		// Get the current URI mapping(s) for each EZ-Page, if any
		// Time is taken to load the information here instead of using data stored in a form, mainly because a
		// "refresh" of a posted page can include out of date data/relationships
		$this->_setPrevURIMappingsForEZPages();
		
		if (!isset($this->_ez_pages_admin_instance) || is_null($this->_ez_pages_admin_instance)) {
			require_once(DIR_FS_CATALOG . DIR_WS_CLASSES . 'class.CeonURIMappingAdminEZPages.php');
			
			$this->_ez_pages_admin_instance = new CeonURIMappingAdminEZPages();
		}
		
		foreach ($this->_pages as $page_id => $page_titles) {
			if (!in_array($page_id, $this->_selected_items)) {
				// Not selected for auto-generation, simply store values for display
				$this->_mappings[$page_id] = $this->_prev_mappings[$page_id];
				
				continue;
			}
			
			// URI mapping is to be auto-generated
			
			for ($i = 0; $i < $num_languages; $i++) {
				$prev_mapping = $this->_prev_mappings[$page_id][$languages[$i]['id']];
				
				// Get some sort of value for the EZ-Page's title
				if (isset($page_titles[$languages[$i]['id']])) {
					$ez_page_title = $page_titles[$languages[$i]['id']];
				} else {
					// No EZ-Page title for current language. Use first page title defined
					foreach ($page_titles as $page_title) {
						$ez_page_title = $page_title;
						
						break;
					}
				}
				
				$mapping = $this->_ez_pages_admin_instance->autogenEZPageURIMapping($page_id, $ez_page_title,
					$languages[$i]['code'], $languages[$i]['id'],
					$this->_page_mapping_templates[$languages[$i]['id']]);
				
				if ($mapping == CEON_URI_MAPPING_GENERATION_ATTEMPT_FOR_EZ_PAGE_WITH_NO_NAME) {
					// Can't generate the URI because of missing "uniqueness" data
					
					$message = TEXT_ERROR_AUTOGENERATION_EZ_PAGE_HAS_NO_NAME;
					
					$failure_message = sprintf($message, ucwords($languages[$i]['name']));
					
					// Build a link to the EZ-Page's edit page so the user can fix the problem immediately
					$edit_link = zen_href_link(FILENAME_EZPAGES, 'ezID=' . $page_id . '&action=new', 'NONSSL',
						true, true, false, false);
					
					$edit_link_title = TEXT_EDIT_EZ_PAGE . ' ' . $page_id;
					
					if (!isset($this->_error_messages['mappings'])) {
						$this->_error_messages['mappings'] = array();
					}
					
					if (!is_array($this->_error_messages['mappings'][$page_id])) {
						$this->_error_messages['mappings'][$page_id] = array();
					}
					
					$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
						array(
							'message' => $failure_message,
							'edit_link' => $edit_link,
							'edit_link_title' => $edit_link_title
							);
					
					// No new mapping generated so restore mapping's value
					$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
					
					continue;
				}
				
				// Make sure URI mapping is relative to root of site and does not have a trailing slash or any
				// illegal characters
				$mapping = $this->_cleanUpURIMapping($mapping);
				
				if (strlen($mapping) > 0 && $mapping != $prev_mapping) {
					// New mapping has been generated
					
					// Check that the mapping just generated doesn't clash with any existing mappings, so the user
					// can be notified
					$columns_to_retrieve = array(
						'main_page',
						'associated_db_id',
						'query_string_parameters'
						);
					
					$selections = array(
						'uri' => zen_db_prepare_input($mapping),
						'current_uri' => 1,
						'language_id' => $languages[$i]['id']
						);
					
					$existing_uri_mapping_result =
						$this->getURIMappingsResultset($columns_to_retrieve, $selections, null, 1);
					
					// If the existing mapping is simply having some capitalisation changed then a case insensitive
					// comparison might result in a false positive for a mapping clash, so prevent that by checking
					// the mapping's settings don't match
					if (!$existing_uri_mapping_result->EOF &&
							!($existing_uri_mapping_result->fields['main_page'] == FILENAME_EZPAGES &&
							$existing_uri_mapping_result->fields['associated_db_id'] == $page_id &&
							is_null($existing_uri_mapping_result->fields['query_string_parameters']))) {
						// This mapping clashes with an existing mapping
						$failure_message = TEXT_ERROR_AUTOGENERATED_EZ_PAGE_URI_CLASHES;
						
						// Build a link to the EZ-Page's edit page so the user can fix the problem immediately
						$edit_link = zen_href_link(FILENAME_EZPAGES, 'ezID=' . $page_id . '&action=new', 'NONSSL',
							true, true, false, false);
						
						$edit_link_title = TEXT_EDIT_EZ_PAGE . ' ' . $ez_page_title;
						
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!is_array($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
							array(
								'message' => $failure_message,
								'link' => HTTP_SERVER . $mapping,
								'link_title' => $mapping,
								'edit_link' => $edit_link,
								'edit_link_title' => $edit_link_title
								);
						
						// No new mapping generated so restore mapping's value
						$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
						
						// Can't create/save mapping
						continue;
					}
				}
				
				$insert_mapping = false;
				$update_mapping = false;
				
				if ($mapping != '') {
					// Check if the URI mapping is being updated or does not yet exist
					if ($prev_mapping == '') {
						$insert_mapping = true;
					} else if ($prev_mapping != $mapping) {
						$update_mapping = true;
					}
				}
				
				if ($insert_mapping || $update_mapping) {
					if ($update_mapping) {
						// Consign previous mapping to the history, so old URI mapping isn't broken
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					}
					
					// Add the new URI mapping
					$uri = $mapping;
					
					$main_page = FILENAME_EZPAGES;
					
					$associated_db_id = $page_id;
					
					$mapping_added =
						$this->addURIMapping($uri, $languages[$i]['id'], $main_page, null, $associated_db_id);
					
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
						if ($insert_mapping) {
							$success_message = sprintf(TEXT_EZ_PAGE_MAPPING_ADDED, ucwords($languages[$i]['name']),
								'<a href="' . HTTP_SERVER . $uri . '" target="_blank">' . $uri . '</a>');
						} else {
							$success_message = sprintf(TEXT_EZ_PAGE_MAPPING_UPDATED,
								ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>');
						}
						
						$messageStack->add($success_message, 'success');
						
					} else {
						if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS, ucwords($languages[$i]['name']));
							
						} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DATA, ucwords($languages[$i]['name']));
						} else {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DB, ucwords($languages[$i]['name']));
						}
						
						// Build a link to the EZ-Page's edit page so the user can fix the problem immediately
						$edit_link = zen_href_link(FILENAME_EZPAGES, 'ezID=' . $page_id . '&action=new', 'NONSSL',
							true, true, false, false);
						
						$edit_link_title = TEXT_EDIT_EZ_PAGE . ' ' . $ez_page_title;
						
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!is_array($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
							array(
								'message' => $failure_message,
								'link' => HTTP_SERVER . $mapping,
								'link_title' => $mapping,
								'edit_link' => $edit_link,
								'edit_link_title' => $edit_link_title
								);
						
						// No new mapping generated so restore mapping's value
						$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
						
						continue;
					}
				} else if ($prev_mapping != '' && $mapping == '') {
					// No URI mapping, consign existing mapping to the history, so old URI mapping isn't broken
					$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					
					$success_message = sprintf(TEXT_EZ_PAGE_MAPPING_MADE_HISTORICAL,
						ucwords($languages[$i]['name']), $ez_page_title);
					
					$messageStack->add($success_message, 'caution');
				}
				
				// Mapping's value has been updated
				$this->_mappings[$page_id][$languages[$i]['id']] = $mapping;
			}
		}
	}
	
	// }}}
	
	
	// {{{ _processEZPagesListingFormSubmission()
	
	/**
	 * Handle the submission of the EZ-Pages listing. Creates/updates any mappings and makes any cleared mappings
	 * historical.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processEZPagesListingFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		$this->_handleItemSelections('page');
		
		// Get the current URI mapping(s) for each EZ-Page, if any
		// Time is taken to load the information here instead of using data stored in a form, mainly because a
		// "refresh" of a posted page can include out of date data/relationships
		$this->_setPrevURIMappingsForEZPages();
		
		// Store the values in the textfields for the page mappings, then check each one to see if a mapping should
		// be created, changed or made historical
		foreach ($this->_pages as $page_id => $page_titles) {
			$this->_mappings[$page_id] = array();
			
			for ($i = 0; $i < $num_languages; $i++) {
				$this->_mappings[$page_id][$languages[$i]['id']] = $this->_cleanUpURIMapping(
					$this->_posted_data['mappings'][$page_id][$languages[$i]['id']]);
				
				$prev_mapping = $this->_prev_mappings[$page_id][$languages[$i]['id']];
				
				$mapping = $this->_mappings[$page_id][$languages[$i]['id']];
				
				// Get some sort of value for the EZ-Page's title
				if (isset($page_titles[$languages[$i]['id']])) {
					$ez_page_title = $page_titles[$languages[$i]['id']];
				} else {
					// No EZ-Page title for current language. Use first page title defined
					foreach ($page_titles as $page_title) {
						$ez_page_title = $page_title;
						
						break;
					}
				}
				
				if (strlen($mapping) > 0 && $mapping != $prev_mapping) {
					// New/updated mapping has been specified
					
					// Check that the mapping(s) just entered doesn't/don't clash with any existing mapping(s), so
					// the user can be notified
					$mapping_clashed = false;
					
					$columns_to_retrieve = array(
						'main_page',
						'associated_db_id',
						'query_string_parameters'
						);
					
					$selections = array(
						'uri' => zen_db_prepare_input($mapping),
						'current_uri' => 1,
						'language_id' => $languages[$i]['id']
						);
					
					$order_by = null;
					
					$existing_uri_mapping_result = $this->getURIMappingsResultset(
						$columns_to_retrieve, $selections, $order_by, 1);
					
					// If the existing mapping is simply having some capitalisation changed then a case insensitive
					// comparison might result in a false positive for a mapping clash, so prevent that by checking
					// the mapping's settings don't match
					if (!$existing_uri_mapping_result->EOF &&
							!($existing_uri_mapping_result->fields['main_page'] == FILENAME_EZPAGES &&
							$existing_uri_mapping_result->fields['associated_db_id'] == $page_id &&
							is_null($existing_uri_mapping_result->fields['query_string_parameters']))) {
						// This mapping clashes with an existing mapping
						
						// Build a link to the EZ-Page's edit page so the user can fix the problem immediately
						$edit_link = zen_href_link(FILENAME_EZPAGES, 'ezID=' . $page_id . '&action=new', 'NONSSL',
						true, true, false, false);
						
						$edit_link_title = TEXT_EDIT_EZ_PAGE . ' ' . $ez_page_title;
						
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!is_array($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
							array(
								'message' => TEXT_ERROR_PAGE_URI_ENTERED_CLASHES,
								'link' => HTTP_SERVER . $mapping,
								'link_title' => $mapping,
								'edit_link' => $edit_link,
								'edit_link_title' => $edit_link_title
								);
						
						continue;
					}
				}
				
				$insert_mapping = false;
				$update_mapping = false;
				
				if ($mapping != '') {
					// Check if the URI mapping is being updated or does not yet exist
					if ($prev_mapping == '') {
						$insert_mapping = true;
					} else if ($prev_mapping != $mapping) {
						$update_mapping = true;
					}
				}
				
				if ($insert_mapping || $update_mapping) {
					if ($update_mapping) {
						// Consign previous mapping to the history, so old URI mapping isn't broken
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					}
					
					// Add the new URI mapping
					$uri = $mapping;
					
					$main_page = FILENAME_EZPAGES;
					
					$associated_db_id = $page_id;
					
					$mapping_added =
						$this->addURIMapping($uri, $languages[$i]['id'], $main_page, null, $associated_db_id);
					
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
						if ($insert_mapping) {
							$success_message = sprintf(TEXT_EZ_PAGE_MAPPING_ADDED, ucwords($languages[$i]['name']),
								'<a href="' . HTTP_SERVER . $uri . '" target="_blank">' . $uri . '</a>');
						} else {
							$success_message = sprintf(TEXT_EZ_PAGE_MAPPING_UPDATED,
								ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>');
						}
						
						// Previous mapping is now this mapping!
						$this->_prev_mappings[$page_id][$languages[$i]['id']] =
							$this->_mappings[$page_id][$languages[$i]['id']];
						
						$messageStack->add($success_message, 'success');
						
					} else {
						if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS, ucwords($languages[$i]['name']));
							
						} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DATA, ucwords($languages[$i]['name']));
						} else {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DB, ucwords($languages[$i]['name']));
						}
						
						// Build a link to the EZ-Page's edit page so the user can fix the problem immediately
						$edit_link = zen_href_link(FILENAME_EZPAGES, 'ezID=' . $page_id . '&action=new', 'NONSSL',
							true, true, false, false);
						
						$edit_link_title = TEXT_EDIT_EZ_PAGE . ' ' . $ez_page_title;
						
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!is_array($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
							array(
								'message' => $failure_message,
								'link' => HTTP_SERVER . $mapping,
								'link_title' => $mapping,
								'edit_link' => $edit_link,
								'edit_link_title' => $edit_link_title
								);
					}
				} else if ($prev_mapping != '' && $mapping == '') {
					// No URI mapping, consign existing mapping to the history, so old URI mapping isn't broken
					$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					
					$success_message = sprintf(TEXT_EZ_PAGE_MAPPING_MADE_HISTORICAL,
						ucwords($languages[$i]['name']), $ez_page_title);
					
					// Previous mapping is now this (lack of) a mapping!
					$this->_prev_mappings[$page_id][$languages[$i]['id']] =
						$this->_mappings[$page_id][$languages[$i]['id']];
					
					$messageStack->add($success_message, 'caution');
				}
			}
		}
	}
	
	// }}}
	
	
	// {{{ _setPrevURIMappingsForEZPages()
	
	/**
	 * Gets the current URI mappings for EZ-Pages and stores them in the "previous mappings"
	 * variable.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _setPrevURIMappingsForEZPages()
	{
		global $db;
		
		foreach ($this->_pages as $page_id => $page_titles) {
			$this->_prev_mappings[$page_id] = array();
			
			$columns_to_retrieve = array(
				'uri',
				'language_id'
				);
			
			$selections = array(
				'current_uri' => 1,
				'main_page' => FILENAME_EZPAGES,
				'associated_db_id' => $page_id
				);
			
			$page_mappings_result =
				$this->getURIMappingsResultset($columns_to_retrieve, $selections);
			
			while (!$page_mappings_result->EOF) {
				$this->_prev_mappings[$page_id][$page_mappings_result->fields['language_id']] =
					$page_mappings_result->fields['uri'];
				
				$page_mappings_result->MoveNext();
			}
		}
	}
	
	// }}}
}

// }}}
