<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate URIs Index Panel Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageIndex.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Action class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMAction.php');


// {{{ CeonUMMCreateManageIndex

/**
 * Handles index panel for Create/Manage URIs section.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageIndex extends CeonUMMAction
{
	// {{{ properties
	
	/**
	 * The default category mapping templates being used for this instance at runtime. Indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_default_categories_mapping_templates = null;
	
	/**
	 * The default product mapping templates being used for this instance at runtime. Indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_default_products_mapping_templates = null;
	
	/**
	 * The default "other" pages mapping templates being used for this instance at runtime. Indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_other_pages_mapping_templates = null;
	
	/**
	 * The default "EZ-Pages mapping templates being used for this instance at runtime. Indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_ez_pages_mapping_templates = null;
	
	/**
	 * The default manufacturer mapping templates being used for this instance at runtime. Indexed by language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_default_manufacturers_mapping_templates = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' . 'ceon_umm_create_manage_index.php');
		
		if (!defined('TEXT_CATEGORIES_PRODUCTS_URI_MAPPINGS') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' . 'ceon_umm_create_manage_index.php');
		}
		
		parent::__construct();
		
		// Run the actual functionality and build the output
		$this->_buildIndexPanel();
	}
	
	// }}}
	
	
	// {{{ _buildIndexPanel()
	
	/**
	 * Builds an introductory panel which lets the user see and set the default mapping templates for the store,
	 * and to access the various create/manage subsections.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildIndexPanel()
	{
		global $db, $languages, $num_languages;
		
		$content = '';
		
		
		$content .= '<ul class="SubPanelMenu">' . "\n";
		
		$content .= '<li><a href="#categories-products-uri-mappings">' . TEXT_CATEGORIES_PRODUCTS_URI_MAPPINGS .
			'</a></li>' . "\n";
		
		$content .= '<li><a href="#other-pages-uri-mappings">' . TEXT_OTHER_PAGES_URI_MAPPINGS . '</a></li>' .
			"\n";
		
		$content .= '<li><a href="#ez-pages-uri-mappings">' . TEXT_EZ_PAGES_URI_MAPPINGS . '</a></li>' . "\n";
		
		$content .= '<li><a href="#manufacturers-uri-mappings">' .
			TEXT_MANUFACTURERS_URI_MAPPINGS . '</a></li>' . "\n";
		
		$content .= '</ul>' . "\n";
		
		
		$content .= '<div class="SubPanelMenuSpacer"></div>' . "\n";
		
		
		$content .= $this->_buildIntroCategoriesProductsSubPanel();
		
		$content .= $this->_buildIntroOtherPagesSubPanel();
		
		$content .= $this->_buildIntroEZPagesSubPanel();
		
		$content .= $this->_buildIntroManufacturersSubPanel();
		
		
		// Build the back to top link
		$content .= '<p class="BackToTop"><a href="#top">' . TEXT_BACK_TO_TOP .
			zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to-top.png') . '</a></p>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _buildIntroCategoriesProductsSubPanel()
	
	/**
	 * Builds a subpanel which lets the user see and set the default mapping templates for categories/products, and
	 * to access the create/manage category/product URIs subsection.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildIntroCategoriesProductsSubPanel()
	{
		global $db, $languages, $num_languages;
		
		// This method uses the shared Admin language definitions.. make sure they're available
		$this->_loadLanguageDefinitions();
		
		$content = '';
		
		if (isset($this->_posted_data['default-categories-mapping-templates'])) {
			// Form with default categories/products mapping templates has been submitted
			$this->_processCategoriesProductsIntroPanelFormSubmission();
		} else {
			// Must load the default mappings
			$columns_to_retrieve = array(
				'mapping_template',
				'language_id'
				);
			
			$selections = array(
				'category_id' => 0
				);
			
			$default_mapping_templates_result =
				$this->getURIMappingTemplatesResultset('category', $columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$default_mapping_templates_result->EOF) {
				$language_id = $default_mapping_templates_result->fields['language_id'];
				
				$this->_default_categories_mapping_templates[$language_id] =
					$default_mapping_templates_result->fields['mapping_template'];
				
				$default_mapping_templates_result->MoveNext();
			}
			
			$default_mapping_templates_result =
				$this->getURIMappingTemplatesResultset('product', $columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$default_mapping_templates_result->EOF) {
				$language_id = $default_mapping_templates_result->fields['language_id'];
				
				$this->_default_products_mapping_templates[$language_id] =
					$default_mapping_templates_result->fields['mapping_template'];
				
				$default_mapping_templates_result->MoveNext();
			}
		}
		
		$content .= '<fieldset id="categories-products-uri-mappings" class="ContainingSubPanel">' . '<legend>' .
			TEXT_CATEGORIES_PRODUCTS_URI_MAPPINGS . '</legend>' . "\n";
		
		
		// Build subpanel with link to manage category/product URIs section //////
		$content .= '<fieldset id="create-manage-category-products"><legend>' .
			TEXT_MANAGE_CATEGORY_PRODUCT_URIS_TITLE . '</legend>' . "\n"; 
		
		$params = zen_get_all_get_params(array('cat-id'));
		
		$params .= 'listing=categories&cat-id=0';
		
		$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p>' . TEXT_ACCESS_CATEGORIES_LISTING . '</p>' . "\n";
		
		$content .= '<p class="ButtonLink"><a href="' . $link . '">' . TEXT_CREATE_MANAGE_CATEGORY_PRODUCT_URIS .
			'</a></p>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		// Build subpanel for managing categories/products mapping templates //////
		$content .= '<fieldset id="default-categories-products-mapping-templates"><legend>' .
			TEXT_DEFAULT_CATEGORIES_PRODUCTS_MAPPING_TEMPLATES_TITLE . '</legend>' . "\n"; 
		
		$content .= '<p>' . TEXT_DEFAULT_CATEGORIES_PRODUCTS_MAPPING_TEMPLATES_INTRO . '</p>' . "\n";
		
		$content .= '<p>' . TEXT_MAPPING_TEMPLATES_CHANGE . '</p>' . "\n";
		
		$content .= '<p>' . TEXT_DEFAULT_CATEGORIES_PRODUCTS_MAPPING_TEMPLATES_OVERRIDE_INFO . '</p>' . "\n";
		
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action . '" method="post"' .
			' name="default-categories-products-mapping-templates-form"' .
			' id="default-categories-products-mapping-templates-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Build the field(s) for the default categories mapping template(s)
		$content .= '<p>' . $this->_addTooltipDescriptionsOfPlacementTags('category',
			TEXT_VALID_PLACEMENT_TAGS_FOR_CATEGORY_MAPPING_TEMPLATE) . '</p>' . "\n";
		
		$content .= '<p><label class="DefaultMappingTemplates">' . "\n";
		
		if ($num_languages == 1) {
			$content .= TEXT_LABEL_DEFAULT_CATEGORIES_MAPPING_TEMPLATE;
		} else {
			$content .= TEXT_LABEL_DEFAULT_CATEGORIES_MAPPING_TEMPLATES;
		}
		
		$content .= '</label></p>' . "\n";
		
		$content .= '<ul class="DefaultMappingTemplatesFields">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li><span class="MappingLanguageIcon">' .
				zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
				$languages[$i]['image'], $languages[$i]['name']) . '</span>';
			
			$content .= ' <input type="text" name="default-categories-mapping-templates[' . $languages[$i]['id'] .
				']" value="' . $this->_default_categories_mapping_templates[$languages[$i]['id']] .
				'" maxlength="255" class="Textfield" />';
			
			if (isset($this->_error_messages['categories_mapping_templates']) &&
					isset($this->_error_messages['categories_mapping_templates'][$languages[$i]['id']])) {
				foreach ($this->_error_messages['categories_mapping_templates'][$languages[$i]['id']] as
						$error_message) {
					$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
				}
			}
			
			$content .= '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		$content .= '<div class="SpacerSmall"></div>' . "\n";
		
		
		// Build the field(s) for the default products mapping template(s)
		$content .= '<p id="default-product-mapping-templates-intro">' .
			$this->_addTooltipDescriptionsOfPlacementTags('product',
			TEXT_VALID_PLACEMENT_TAGS_FOR_PRODUCT_MAPPING_TEMPLATE) . '</p>' . "\n";
		
		$content .= '<p><label class="DefaultMappingTemplates">' . "\n";
		
		if ($num_languages == 1) {
			$content .= TEXT_LABEL_DEFAULT_PRODUCTS_MAPPING_TEMPLATE;
		} else {
			$content .= TEXT_LABEL_DEFAULT_PRODUCTS_MAPPING_TEMPLATES;
		}
		
		$content .= '</label></p>' . "\n";
		
		$content .= '<ul class="DefaultMappingTemplatesFields">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li><span class="MappingLanguageIcon">' .
				zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
				$languages[$i]['image'], $languages[$i]['name']) . '</span>';
			
			$content .= ' <input type="text" name="default-products-mapping-templates[' . $languages[$i]['id'] .
				']" value="' . $this->_default_products_mapping_templates[$languages[$i]['id']] .
				'" maxlength="255" class="Textfield" />';
			
			if (isset($this->_error_messages['products_mapping_templates']) &&
					isset($this->_error_messages['products_mapping_templates'][$languages[$i]['id']])) {
				foreach ($this->_error_messages['products_mapping_templates'][$languages[$i]['id']] as
						$error_message) {
					$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
				}
			}
			
			$content .= '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		$content .= '<input type="submit" class="DefaultMappingTemplatesSubmit" name="submit"' .
			' id="default-categories-products-mapping-templates-submit" value="' . TEXT_UPDATE_TEMPLATES . '" />' .
			"\n"; 
		
		$content .= '</form>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		$content .= '</fieldset>' . "\n";
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _processCategoriesProductsIntroPanelFormSubmission()
	
	/**
	 * Handle the submission of updated default template(s) for categories/products.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processCategoriesProductsIntroPanelFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		for ($i = 0; $i < $num_languages; $i++) {
			$default_category_mapping_template =
				$this->_posted_data['default-categories-mapping-templates'][$languages[$i]['id']];
			
			if (strlen($default_category_mapping_template) == 0) {
				if (!isset($this->_error_messages['categories_mapping_templates'])) {
					$this->_error_messages['categories_mapping_templates'] = array();
				}
				
				$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']] =
					array(
						TEXT_ERROR_MISSING_MAPPING_TEMPLATE
						);
				
			} else {
				$valid_template =
					$this->_validateURIMappingTemplate('category',$default_category_mapping_template);
				
				if (is_array($valid_template)) {
					if (!isset($this->_error_messages['categories_mapping_templates'])) {
						$this->_error_messages['categories_mapping_templates'] = array();
					}
					
					if (!is_array($this->_error_messages['categories_mapping_templates'][$languages[$i]['id']])) {
						$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']] = array();
					}
					
					$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']][] =
						TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
					
					foreach ($valid_template as $template_error) {
						$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']][] =
							$template_error;
					}
				} else {
					// Make sure the template complies with standard mapping rules of slash at start but none at
					// end
					$default_category_mapping_template =
						$this->_cleanUpURIMappingTemplate($default_category_mapping_template);
				}
				
				$this->_default_categories_mapping_templates[$languages[$i]['id']] =
					$default_category_mapping_template;
			}
			
			
			$default_product_mapping_template =
				$this->_posted_data['default-products-mapping-templates'][$languages[$i]['id']];
			
			if (strlen($default_product_mapping_template) == 0) {
				if (!isset($this->_error_messages['products_mapping_templates'])) {
					$this->_error_messages['products_mapping_templates'] = array();
				}
				
				$this->_error_messages['products_mapping_templates'][$languages[$i]['id']] =
					array(
						TEXT_ERROR_MISSING_MAPPING_TEMPLATE
						);
				
			} else {
				$valid_template = $this->_validateURIMappingTemplate('product', $default_product_mapping_template);
				
				if (is_array($valid_template)) {
					if (!isset($this->_error_messages['products_mapping_templates'])) {
						$this->_error_messages['products_mapping_templates'] = array();
					}
					
					if (!is_array($this->_error_messages['products_mapping_templates'][$languages[$i]['id']])) {
						$this->_error_messages['products_mapping_templates'][$languages[$i]['id']] = array();
					}
					
					$this->_error_messages['products_mapping_templates'][$languages[$i]['id']][] =
						TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
					
					foreach ($valid_template as $template_error) {
						$this->_error_messages['products_mapping_templates'][$languages[$i]['id']][] =
							$template_error;
					}
				} else {
					// Make sure the template complies with standard mapping rules of slash at start but none at
					// end
					$default_product_mapping_template =
						$this->_cleanUpURIMappingTemplate($default_product_mapping_template);
				}
				
				$this->_default_products_mapping_templates[$languages[$i]['id']] =
					$default_product_mapping_template;
			}
		}
		
		if (!isset($this->_error_messages['categories_mapping_templates']) &&
				!isset($this->_error_messages['products_mapping_templates'])) {
			// All templates valid!
			
			// Save all the mapping templates
			for ($i = 0; $i < $num_languages; $i++) {
				$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
					$this->_default_categories_mapping_templates[$languages[$i]['id']], 'category', 0,
					$languages[$i]['id']);
				
				if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
					$messageStack->add(
						sprintf(TEXT_DEFAULT_CATEGORIES_MAPPING_TEMPLATE_ADDED, ucwords($languages[$i]['name'])),
						'success');
				} else if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
					$messageStack->add(
						sprintf(TEXT_DEFAULT_CATEGORIES_MAPPING_TEMPLATE_UPDATED, ucwords($languages[$i]['name'])),
						'success');
				}
				
				$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
					$this->_default_products_mapping_templates[$languages[$i]['id']], 'product', 0,
					$languages[$i]['id']);
				
				if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
					$messageStack->add(
						sprintf(TEXT_DEFAULT_PRODUCTS_MAPPING_TEMPLATE_ADDED, ucwords($languages[$i]['name'])),
						'success');
				} else if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
					$messageStack->add(
						sprintf(TEXT_DEFAULT_PRODUCTS_MAPPING_TEMPLATE_UPDATED, ucwords($languages[$i]['name'])),
						'success');
				}
			}
		}
	}
	
	// }}}
	
	
	// {{{ _buildIntroOtherPagesSubPanel()
	
	/**
	 * Builds a subpanel which lets the user see and set the default mapping templates for other pages, and to
	 * access the create/manage other pages URIs subsection.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildIntroOtherPagesSubPanel()
	{
		global $db, $languages, $num_languages;
		
		$content = '';
		
		if (isset($this->_posted_data['other-pages-mapping-templates'])) {
			// Form with default other pages mapping templates has been submitted
			$this->_processOtherPagesIntroPanelFormSubmission();
		} else {
			// Must load the default mappings
			$columns_to_retrieve = array(
				'mapping_template',
				'language_id'
				);
			
			$selections = array(
				'page_type' => 'other-page'
				);
			
			$mapping_templates_result =
				$this->getURIMappingTemplatesResultset('other-page', $columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$mapping_templates_result->EOF) {
				$language_id = $mapping_templates_result->fields['language_id'];
				
				$this->_other_pages_mapping_templates[$language_id] =
					$mapping_templates_result->fields['mapping_template'];
				
				$mapping_templates_result->MoveNext();
			}
		}
		
		$content .= '<fieldset id="other-pages-uri-mappings" class="ContainingSubPanel">' .
			'<legend>' . TEXT_OTHER_PAGES_URI_MAPPINGS . '</legend>' . "\n";
		
		
		// Build subpanel with link to manage other page URIs section //////
		$content .= '<fieldset id="create-manage-other-pages"><legend>' . TEXT_MANAGE_OTHER_PAGE_URIS_TITLE .
			'</legend>' . "\n"; 
		
		$params = zen_get_all_get_params();
		
		$params .= 'listing=other-pages';
		
		$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p>' . TEXT_ACCESS_OTHER_PAGES_LISTING . '</p>' . "\n";
		
		$content .= '<p class="ButtonLink"><a href="' . $link . '">' . TEXT_CREATE_MANAGE_OTHER_PAGE_URIS .
			'</a></p>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		// Build subpanel for managing other pages mapping templates //////
		$content .= '<fieldset id="other-pages-mapping-templates"><legend>';
		
		if ($num_languages == 1) {
			$content .= TEXT_OTHER_PAGES_MAPPING_TEMPLATE_TITLE;
		} else {
			$content .= TEXT_OTHER_PAGES_MAPPING_TEMPLATES_TITLE;
		}
		
		$content .= '</legend>' . "\n"; 
		
		
		$content .= '<p>' . TEXT_OTHER_PAGES_MAPPING_TEMPLATES_OTHER_PAGES_DESC . '</p>' . "\n";
		
		if ($num_languages == 1) {
			$content .= '<p>' . TEXT_OTHER_PAGES_MAPPING_TEMPLATE_INTRO . '</p>' . "\n";
		} else {
			$content .= '<p>' . TEXT_OTHER_PAGES_MAPPING_TEMPLATES_INTRO . '</p>' . "\n";
		}
		
		if ($num_languages == 1) {
			$content .= '<p>' . TEXT_MAPPING_TEMPLATE_CHANGE . '</p>' . "\n";
		} else {
			$content .= '<p>' . TEXT_MAPPING_TEMPLATES_CHANGE . '</p>' . "\n";
		}
		
		$content .= '<p>' . TEXT_OTHER_PAGES_MAPPING_TEMPLATES_INFO . '</p>' . "\n";
		
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action . '" method="post" name="other-pages-mapping-templates-form"' .
			' id="other-pages-mapping-templates-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Build the field(s) for the default other pages mapping template(s)
		$content .= '<p>' . $this->_addTooltipDescriptionsOfPlacementTags('other-page',
			TEXT_VALID_PLACEMENT_TAGS_FOR_OTHER_PAGE_MAPPING_TEMPLATE) . '</p>' . "\n";
		
		$content .= '<p><label class="DefaultMappingTemplates">' . "\n";
		
		if ($num_languages == 1) {
			$content .= TEXT_LABEL_OTHER_PAGES_MAPPING_TEMPLATE;
		} else {
			$content .= TEXT_LABEL_OTHER_PAGES_MAPPING_TEMPLATES;
		}
		
		$content .= '</label></p>' . "\n";
		
		$content .= '<ul class="DefaultMappingTemplatesFields">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li><span class="MappingLanguageIcon">' .
				zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
				$languages[$i]['image'], $languages[$i]['name']) . '</span>';
			
			$content .= ' <input type="text" name="other-pages-mapping-templates[' . $languages[$i]['id'] . ']"' .
				' value="' . $this->_other_pages_mapping_templates[$languages[$i]['id']] .
				'" maxlength="255" class="Textfield" />';
			
			if (isset($this->_error_messages['other_pages_mapping_templates']) &&
					isset($this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']])) {
				foreach ($this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']] as
						$error_message) {
					$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
				}
			}
			
			$content .= '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		$content .= '<div class="SpacerSmall"></div>' . "\n";
		
		if ($num_languages == 1) {
			$button_text = TEXT_UPDATE_TEMPLATE;
		} else {
			$button_text = TEXT_UPDATE_TEMPLATES;
		}
		
		$content .= '<input type="submit" class="DefaultMappingTemplatesSubmit"' .
			' name="submit" id="other-pages-mapping-templates-submit" value="' . $button_text . '" />' . "\n"; 
		
		$content .= '</form>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		$content .= '</fieldset>' . "\n";
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _processOtherPagesIntroPanelFormSubmission()
	
	/**
	 * Handle the submission of updated default template(s) for other pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processOtherPagesIntroPanelFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		for ($i = 0; $i < $num_languages; $i++) {
			$other_page_mapping_template = $this->_posted_data
				['other-pages-mapping-templates'][$languages[$i]['id']];
			
			if (strlen($other_page_mapping_template) == 0) {
				if (!isset($this->_error_messages['other_pages_mapping_templates'])) {
					$this->_error_messages['other_pages_mapping_templates'] = array();
				}
				
				$this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']] =
					array(
						TEXT_ERROR_MISSING_MAPPING_TEMPLATE
						);
				
			} else {
				$valid_template = $this->_validateURIMappingTemplate('other-page', $other_page_mapping_template);
				
				if (is_array($valid_template)) {
					if (!isset($this->_error_messages['other_pages_mapping_templates'])) {
						$this->_error_messages['other_pages_mapping_templates'] = array();
					}
					
					if (!is_array($this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']])) {
						$this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']] = array();
					}
					
					$this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']][] =
						TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
					
					foreach ($valid_template as $template_error) {
						$this->_error_messages['other_pages_mapping_templates'][$languages[$i]['id']][] =
							$template_error;
					}
				} else {
					// Make sure the template complies with standard mapping rules of slash at start but none at
					// end
					$other_page_mapping_template =
						$this->_cleanUpURIMappingTemplate($other_page_mapping_template);
				}
				
				$this->_other_pages_mapping_templates[$languages[$i]['id']] = $other_page_mapping_template;
			}
		}
		
		if (!isset($this->_error_messages['other_pages_mapping_templates'])) {
			// All templates valid!
			
			// Save all the mapping templates
			for ($i = 0; $i < $num_languages; $i++) {
				$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
					$this->_other_pages_mapping_templates[$languages[$i]['id']], 'other-page', 'other-page',
					$languages[$i]['id']);
				
				if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
					$messageStack->add(sprintf(TEXT_OTHER_PAGES_MAPPING_TEMPLATE_ADDED,
						ucwords($languages[$i]['name'])), 'success');
				} else if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
					$messageStack->add(sprintf(TEXT_OTHER_PAGES_MAPPING_TEMPLATE_UPDATED,
						ucwords($languages[$i]['name'])), 'success');
				}
			}
		}
	}
	
	// }}}
	
	
	// {{{ _buildIntroEZPagesSubPanel()
	
	/**
	 * Builds a subpanel which lets the user see and set the default mapping templates for EZ-Pages, and to access
	 * the create/manage EZ-Pages URIs subsection.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildIntroEZPagesSubPanel()
	{
		global $db, $languages, $num_languages;
		
		$content = '';
		
		if (isset($this->_posted_data['ez-pages-mapping-templates'])) {
			// Form with default EZ-Pages mapping templates has been submitted
			$this->_processEZPagesIntroPanelFormSubmission();
		} else {
			// Must load the default mappings
			$columns_to_retrieve = array(
				'mapping_template',
				'language_id'
				);
			
			$selections = array(
				'page_type' => 'ez-page'
				);
			
			$mapping_templates_result =
				$this->getURIMappingTemplatesResultset('ez-page', $columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$mapping_templates_result->EOF) {
				$language_id = $mapping_templates_result->fields['language_id'];
				
				$this->_ez_pages_mapping_templates[$language_id] =
					$mapping_templates_result->fields['mapping_template'];
				
				$mapping_templates_result->MoveNext();
			}
		}
		
		$content .= '<fieldset id="ez-pages-uri-mappings" class="ContainingSubPanel">' .
			'<legend>' . TEXT_EZ_PAGES_URI_MAPPINGS . '</legend>' . "\n";
		
		
		// Build subpanel with link to manage EZ-Page URIs section //////
		$content .= '<fieldset id="create-manage-ez-pages"><legend>' . TEXT_MANAGE_EZ_PAGE_URIS_TITLE .
			'</legend>' . "\n"; 
		
		$params = zen_get_all_get_params();
		
		$params .= 'listing=ez-pages';
		
		$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p>' . TEXT_ACCESS_EZ_PAGES_LISTING . '</p>' . "\n";
		
		$content .= '<p class="ButtonLink"><a href="' . $link . '">' . TEXT_CREATE_MANAGE_EZ_PAGE_URIS .
			'</a></p>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		// Build subpanel for managing EZ-Pages mapping templates //////
		$content .= '<fieldset id="ez-pages-mapping-templates"><legend>';
		
		if ($num_languages == 1) {
			$content .= TEXT_EZ_PAGES_MAPPING_TEMPLATE_TITLE;
		} else {
			$content .= TEXT_EZ_PAGES_MAPPING_TEMPLATES_TITLE;
		}
		
		$content .= '</legend>' . "\n"; 
		
		
		if ($num_languages == 1) {
			$content .= '<p>' . TEXT_EZ_PAGES_MAPPING_TEMPLATE_INTRO . '</p>' . "\n";
		} else {
			$content .= '<p>' . TEXT_EZ_PAGES_MAPPING_TEMPLATES_INTRO . '</p>' . "\n";
		}
		
		if ($num_languages == 1) {
			$content .= '<p>' . TEXT_MAPPING_TEMPLATE_CHANGE . '</p>' . "\n";
		} else {
			$content .= '<p>' . TEXT_MAPPING_TEMPLATES_CHANGE . '</p>' . "\n";
		}
		
		$content .= '<p>' . TEXT_EZ_PAGES_MAPPING_TEMPLATES_INFO . '</p>' . "\n";
		
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action . '" method="post" name="ez-pages-mapping-templates-form"' .
			' id="ez-pages-mapping-templates-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Build the field(s) for the EZ-Page mapping template(s)
		$content .= '<p>' . $this->_addTooltipDescriptionsOfPlacementTags('ez-page',
			TEXT_VALID_PLACEMENT_TAGS_FOR_EZ_PAGE_MAPPING_TEMPLATE) . '</p>' . "\n";
		
		$content .= '<p><label class="DefaultMappingTemplates">' . "\n";
		
		if ($num_languages == 1) {
			$content .= TEXT_LABEL_EZ_PAGES_MAPPING_TEMPLATE;
		} else {
			$content .= TEXT_LABEL_EZ_PAGES_MAPPING_TEMPLATES;
		}
		
		$content .= '</label></p>' . "\n";
		
		$content .= '<ul class="DefaultMappingTemplatesFields">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li><span class="MappingLanguageIcon">' .
				zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
				$languages[$i]['image'], $languages[$i]['name']) . '</span>';
			
			$content .= ' <input type="text" name="ez-pages-mapping-templates[' . $languages[$i]['id'] . ']"' .
				' value="' . $this->_ez_pages_mapping_templates[$languages[$i]['id']] . '" maxlength="255"' .
				' class="Textfield" />';
			
			if (isset($this->_error_messages['ez_pages_mapping_templates']) &&
					isset($this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']])) {
				foreach ($this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']] as
						$error_message) {
					$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
				}
			}
			
			$content .= '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		$content .= '<div class="SpacerSmall"></div>' . "\n";
		
		if ($num_languages == 1) {
			$button_text = TEXT_UPDATE_TEMPLATE;
		} else {
			$button_text = TEXT_UPDATE_TEMPLATES;
		}
		
		$content .= '<input type="submit" class="DefaultMappingTemplatesSubmit"' .
			' name="submit" id="ez-pages-mapping-templates-submit" value="' . $button_text . '" />' . "\n"; 
		
		$content .= '</form>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		$content .= '</fieldset>' . "\n";
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _processEZPagesIntroPanelFormSubmission()
	
	/**
	 * Handle the submission of updated default template(s) for EZ-Pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processEZPagesIntroPanelFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		for ($i = 0; $i < $num_languages; $i++) {
			$ez_page_mapping_template = $this->_posted_data['ez-pages-mapping-templates'][$languages[$i]['id']];
			
			if (strlen($ez_page_mapping_template) == 0) {
				if (!isset($this->_error_messages['ez_pages_mapping_templates'])) {
					$this->_error_messages['ez_pages_mapping_templates'] = array();
				}
				
				$this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']] =
					array(
						TEXT_ERROR_MISSING_MAPPING_TEMPLATE
						);
				
			} else {
				$valid_template = $this->_validateURIMappingTemplate('ez-page', $ez_page_mapping_template);
				
				if (is_array($valid_template)) {
					if (!isset($this->_error_messages['ez_pages_mapping_templates'])) {
						$this->_error_messages['ez_pages_mapping_templates'] = array();
					}
					
					if (!is_array($this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']])) {
						$this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']] = array();
					}
					
					$this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']][] =
						TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
					
					foreach ($valid_template as $template_error) {
						$this->_error_messages['ez_pages_mapping_templates'][$languages[$i]['id']][] =
							$template_error;
					}
				} else {
					// Make sure the template complies with standard mapping rules of slash at start but none at
					// end
					$ez_page_mapping_template = $this->_cleanUpURIMappingTemplate($ez_page_mapping_template);
				}
				
				$this->_ez_pages_mapping_templates[$languages[$i]['id']] = $ez_page_mapping_template;
			}
		}
		
		if (!isset($this->_error_messages['ez_pages_mapping_templates'])) {
			// All templates valid!
			
			// Save all the mapping templates
			for ($i = 0; $i < $num_languages; $i++) {
				$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
					$this->_ez_pages_mapping_templates[$languages[$i]['id']], 'ez-page', 'ez-page',
					$languages[$i]['id']);
				
				if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
					$messageStack->add(sprintf(TEXT_EZ_PAGES_MAPPING_TEMPLATE_ADDED,
						ucwords($languages[$i]['name'])), 'success');
				} else if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
					$messageStack->add(sprintf(TEXT_EZ_PAGES_MAPPING_TEMPLATE_UPDATED,
						ucwords($languages[$i]['name'])), 'success');
				}
			}
		}
	}
	
	// }}}
	
	
	// {{{ _buildIntroManufacturersSubPanel()
	
	/**
	 * Builds a subpanel which lets the user see and set the default mapping templates for manufacturers, and to
	 * access the create/manage manufacturer URIs subsection.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildIntroManufacturersSubPanel()
	{
		global $db, $languages, $num_languages;
		
		$content = '';
		
		if (isset($this->_posted_data['default-manufacturers-mapping-templates'])) {
			// Form with default manufacturers mapping templates has been submitted
			$this->_processManufacturersIntroPanelFormSubmission();
		} else {
			// Must load the default mappings
			$columns_to_retrieve = array(
				'mapping_template',
				'language_id'
				);
			
			$selections = array(
				'page_type' => 'manufacturer'
				);
			
			$default_mapping_templates_result =
				$this->getURIMappingTemplatesResultset('manufacturer', $columns_to_retrieve, $selections);
			
			// Store any templates for any language
			while (!$default_mapping_templates_result->EOF) {
				$language_id = $default_mapping_templates_result->fields['language_id'];
				
				$this->_default_manufacturers_mapping_templates[$language_id] =
					$default_mapping_templates_result->fields['mapping_template'];
				
				$default_mapping_templates_result->MoveNext();
			}
		}
		
		$content .= '<fieldset id="manufacturers-uri-mappings" class="ContainingSubPanel">' .
			'<legend>' . TEXT_MANUFACTURERS_URI_MAPPINGS . '</legend>' . "\n";
		
		
		// Build subpanel with link to manage manufacturer URIs section //////
		$content .= '<fieldset id="create-manage-manufacturers"><legend>' . TEXT_MANAGE_MANUFACTURER_URIS_TITLE .
			'</legend>' . "\n"; 
		
		$params = zen_get_all_get_params();
		
		$params .= 'listing=manufacturers';
		
		$link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p>' . TEXT_ACCESS_MANUFACTURERS_LISTING . '</p>' . "\n";
		
		$content .= '<p class="ButtonLink"><a href="' . $link . '">' . TEXT_CREATE_MANAGE_MANUFACTURER_URIS .
			'</a></p>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		// Build subpanel for managing manufacturers mapping templates //////
		$content .= '<fieldset id="default-manufacturers-mapping-templates"><legend>';
		
		if ($num_languages == 1) {
			$content .= TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATE_TITLE;
		} else {
			$content .= TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATES_TITLE;
		}
		
		$content .= '</legend>' . "\n"; 
		
		if ($num_languages == 1) {
			$content .= '<p>' . TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATE_INTRO . '</p>' . "\n";
		} else {
			$content .= '<p>' . TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATES_INTRO . '</p>' . "\n";
		}
		
		if ($num_languages == 1) {
			$content .= '<p>' . TEXT_MAPPING_TEMPLATE_CHANGE . '</p>' . "\n";
		} else {
			$content .= '<p>' . TEXT_MAPPING_TEMPLATES_CHANGE . '</p>' . "\n";
		}
		
		$content .= '<p>' . TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATES_INFO . '</p>' . "\n";
		
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action . '" method="post"' .
			' name="default-manufacturers-mapping-templates-form"' .
			' id="default-manufacturers-mapping-templates-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Build the field(s) for the default manufacturers mapping template(s)
		$content .= '<p>' . $this->_addTooltipDescriptionsOfPlacementTags('manufacturer',
			TEXT_VALID_PLACEMENT_TAGS_FOR_MANUFACTURER_MAPPING_TEMPLATE) . '</p>' . "\n";
		
		$content .= '<p><label class="DefaultMappingTemplates">' . "\n";
		
		if ($num_languages == 1) {
			$content .= TEXT_LABEL_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATE;
		} else {
			$content .= TEXT_LABEL_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATES;
		}
		
		$content .= '</label></p>' . "\n";
		
		$content .= '<ul class="DefaultMappingTemplatesFields">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li><span class="MappingLanguageIcon">' .
				zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
				$languages[$i]['image'], $languages[$i]['name']) . '</span>';
			
			$content .= ' <input type="text" name="default-manufacturers-mapping-templates[' .
				$languages[$i]['id'] . ']" value="' .
				$this->_default_manufacturers_mapping_templates[$languages[$i]['id']] .
				'" maxlength="255" class="Textfield" />';
			
			if (isset($this->_error_messages['manufacturers_mapping_templates']) &&
					isset($this->_error_messages['manufacturers_mapping_templates'][$languages[$i]['id']])) {
				foreach ($this->_error_messages['manufacturers_mapping_templates'][$languages[$i]['id']] as
						$error_message) {
					$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
				}
			}
			
			$content .= '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		$content .= '<div class="SpacerSmall"></div>' . "\n";
		
		if ($num_languages == 1) {
			$button_text = TEXT_UPDATE_TEMPLATE;
		} else {
			$button_text = TEXT_UPDATE_TEMPLATES;
		}
		
		$content .= '<input type="submit" class="DefaultMappingTemplatesSubmit"' .
			' name="submit" id="default-manufacturers-mapping-templates-submit"' .
			' value="' . $button_text . '" />' . "\n"; 
		
		$content .= '</form>' . "\n";
		
		$content .= '</fieldset>' . "\n";
		
		
		$content .= '</fieldset>' . "\n";
		
		return $content;
	}
	
	// }}}
	
	
	// {{{ _processManufacturersIntroPanelFormSubmission()
	
	/**
	 * Handle the submission of updated default template(s) for manufacturers.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processManufacturersIntroPanelFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		for ($i = 0; $i < $num_languages; $i++) {
			$default_manufacturer_mapping_template =
				$this->_posted_data['default-manufacturers-mapping-templates'][$languages[$i]['id']];
			
			if (strlen($default_manufacturer_mapping_template) == 0) {
				if (!isset($this->_error_messages['manufacturers_mapping_templates'])) {
					$this->_error_messages['manufacturers_mapping_templates'] = array();
				}
				
				$this->_error_messages['manufacturers_mapping_templates'][$languages[$i]['id']] =
					array(
						TEXT_ERROR_MISSING_MAPPING_TEMPLATE
						);
				
			} else {
				$valid_template =
					$this->_validateURIMappingTemplate('manufacturer', $default_manufacturer_mapping_template);
				
				if (is_array($valid_template)) {
					if (!isset($this->_error_messages['manufacturers_mapping_templates'])) {
						$this->_error_messages['manufacturers_mapping_templates'] = array();
					}
					
					if (!is_array($this->_error_messages['manufacturers_mapping_templates']
							[$languages[$i]['id']])) {
						$this->_error_messages['manufacturers_mapping_templates'][$languages[$i]['id']] = array();
					}
					
					$this->_error_messages['manufacturers_mapping_templates'][$languages[$i]['id']][] =
						TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
					
					foreach ($valid_template as $template_error) {
						$this->_error_messages['manufacturers_mapping_templates'][$languages[$i]['id']][] =
							$template_error;
					}
				} else {
					// Make sure the template complies with standard mapping rules of slash at start but none at
					// end
					$default_manufacturer_mapping_template =
						$this->_cleanUpURIMappingTemplate($default_manufacturer_mapping_template);
				}
				
				$this->_default_manufacturers_mapping_templates[$languages[$i]['id']] =
					$default_manufacturer_mapping_template;
			}
		}
		
		if (!isset($this->_error_messages['manufacturers_mapping_templates'])) {
			// All templates valid!
			
			// Save all the mapping templates
			for ($i = 0; $i < $num_languages; $i++) {
				$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
					$this->_default_manufacturers_mapping_templates[$languages[$i]['id']], 'manufacturer',
					'manufacturer', $languages[$i]['id']);
				
				if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
					$messageStack->add(sprintf(TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATE_ADDED,
						ucwords($languages[$i]['name'])), 'success');
				} else if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
					$messageStack->add(sprintf(TEXT_DEFAULT_MANUFACTURERS_MAPPING_TEMPLATE_UPDATED,
						ucwords($languages[$i]['name'])), 'success');
				}
			}
		}
	}
	
	// }}}
}

// }}}
