<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate Object Mapping Templates Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageObjMappingTemplates.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Action class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMAction.php');


// {{{ CeonUMMCreateManageObjMappingTemplates

/**
 * Allows the user to manually set/change/delete the URI mapping templates for the specified object (page).
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageObjMappingTemplates extends CeonUMMAction
{
	// {{{ properties
	
	/**
	 * The type of object being managed.
	 *
	 * @var     string
	 * @access  protected
	 */
	protected $_object_type = null;
	
	/**
	 * The ID of the category being managed.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_category_id = null;
	
	/**
	 * The ID of the parent category of the category being managed.
	 *
	 * @var     integer
	 * @access  protected
	 */
	protected $_parent_category_id = null;
	
	/**
	 * Whether or not the category being managed has subcategories (otherwise assumed to simply contain products).
	 *
	 * @var     boolean
	 * @access  protected
	 */
	protected $_category_has_subcategories = null;
	
	/**
	 * The specific categories mapping templates being used by a category for this instance at runtime. Indexed by
	 * language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_categories_mapping_templates = null;
	
	/**
	 * The specific products mapping templates being used by a category for this instance at untime. Indexed by
	 * language ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_products_mapping_templates = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 * @param   string    $object_type   The type of object the mapping templates are intended for.
	 * @param   integer   $object_id     The ID of the object the mapping templates are for.
	 */
	public function __construct($object_type, $object_id)
	{
		global $db;
		
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' .
			'ceon_umm_create_manage_obj_mapping_templates.php');
		
		if (!defined('TEXT_MANAGE_MAPPING_TEMPLATES_TITLE_CATEGORY') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' . 'ceon_umm_create_manage_obj_mapping_templates.php');
		}
		
		// Set up the main environment control variables
		$this->_object_type = $object_type;
		
		if ($this->_object_type == 'category') {
			// Build the form for a category's mappings
			$this->_category_id = $object_id;
			
			$this->_parent_category_id = (int) $_GET['cat-id'];
			
			// Does this category have subcategories or is it a product category
			$this->_category_has_subcategories = false;
			
			$subcategories_result = $db->Execute("
				SELECT
					c.categories_id
				FROM
					" . TABLE_CATEGORIES . " c
				WHERE
					c.parent_id = '" . $this->_category_id . "'
				LIMIT
					1");
			
			if (!$subcategories_result->EOF) {
				$this->_category_has_subcategories = true;
			}
		}
		
		parent::__construct();
		
		// Run the actual functionality and build the output
		$this->_buildManageObjectMappingTemplatesPanel();
	}
	
	// }}}
	
	
	// {{{ _buildManageObjectMappingTemplatesPanel()
	
	/**
	 * Builds a subpanel which lets the user see and set the mapping templates for the specified object.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildManageObjectMappingTemplatesPanel()
	{
		global $db, $languages, $num_languages;
		
		// This method uses the shared Admin language definitions.. make sure they're available
		$this->_loadLanguageDefinitions();
		
		if (isset($this->_posted_data['mapping-templates-submit'])) {
			// Form has been submitted
			$this->_processMappingTemplatesFormSubmission();
		} else {
			// Set up defaults
			if ($this->_object_type == 'category') {
				$columns_to_retrieve = array(
					'mapping_template',
					'language_id'
					);
				
				$selections = array(
					'category_id' => $this->_category_id
					);
				
				if ($this->_category_has_subcategories) {
					$mapping_templates_result =
						$this->getURIMappingTemplatesResultset('category', $columns_to_retrieve, $selections);
					
					// Store any templates for any language
					while (!$mapping_templates_result->EOF) {
						$language_id = $mapping_templates_result->fields['language_id'];
						
						$this->_categories_mapping_templates[$language_id] =
							$mapping_templates_result->fields['mapping_template'];
						
						$mapping_templates_result->MoveNext();
					}
				}
				
				$mapping_templates_result =
					$this->getURIMappingTemplatesResultset('product', $columns_to_retrieve, $selections);
				
				// Store any templates for any language
				while (!$mapping_templates_result->EOF) {
					$language_id = $mapping_templates_result->fields['language_id'];
					
					$this->_products_mapping_templates[$language_id] =
						$mapping_templates_result->fields['mapping_template'];
					
					$mapping_templates_result->MoveNext();
				}
			}
		}
		
		// Build the output for this method ////
		$content = '';
		
		if ($this->_object_type == 'category') {
			$category_name = zen_get_category_name($this->_category_id, $_SESSION['languages_id']);
			
			$title = sprintf(TEXT_MANAGE_MAPPING_TEMPLATES_TITLE_CATEGORY,
				htmlentities($category_name, ENT_COMPAT, CHARSET));
		}
		
		$content .= '<fieldset id="manage-object-mapping-templates" class="ContainingSubPanel">' . '<legend>' .
			$title . '</legend>' . "\n";
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action . '" method="post" name="mapping-templates-form"' .
			' id="mapping-templates-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		if ($this->_object_type == 'category') {
			if ($this->_category_has_subcategories) {
				// Build the intro for the category's subcategories mapping template(s)
				$subsection_title = TEXT_MAPPING_TEMPLATES_SUBSECTION_TITLE_SUBCATEGORIES;
				
				if ($num_languages == 1) {
					$intro = sprintf(TEXT_MANAGE_MAPPING_TEMPLATE_INTRO_CATEGORY_PRODUCTS,
						htmlentities($category_name, ENT_COMPAT, CHARSET));
				} else {
					$intro = sprintf(TEXT_MAPPING_TEMPLATES_INTRO_CATEGORY_SUBCATEGORIES,
						htmlentities($category_name, ENT_COMPAT, CHARSET));
				}
				
				$intro2 = TEXT_MAPPING_TEMPLATES_INTRO_CATEGORY_SUBCATEGORIES_WILL_USE_NEW_TEMPLATE;
				
				if ($num_languages == 1) {
					if ($this->_parent_category_id == 0) {
						$intro3 = TEXT_MAPPING_TEMPLATE_INTRO_CATEGORIES_OVERRIDE_DEFAULT;
					} else {
						$intro3 = TEXT_MAPPING_TEMPLATE_INTRO_CATEGORIES_OVERRIDE_PARENT;
					}
					
					$intro4 = TEXT_REMOVE_THE_TEMPLATE_BY_ENTERING_NOTHING;
					
				} else {
					if ($this->_parent_category_id == 0) {
						$intro3 = TEXT_MAPPING_TEMPLATES_INTRO_CATEGORIES_OVERRIDE_DEFAULT;
					} else {
						$intro3 = TEXT_MAPPING_TEMPLATES_INTRO_CATEGORIES_OVERRIDE_PARENT;
					}
					
					$intro4 = TEXT_REMOVE_A_TEMPLATE_BY_ENTERING_NOTHING;
				}
				
				$valid_tags = $this->_addTooltipDescriptionsOfPlacementTags('category',
					TEXT_VALID_PLACEMENT_TAGS_FOR_CATEGORY_MAPPING_TEMPLATE);
				
				$content .= '<h3 class="First">' . $subsection_title . '</h3>' . "\n";
				
				$content .= '<p>' . $intro . '</p>' . "\n";
				
				$content .= '<p>' . $intro2 . '</p>' . "\n";
				
				$content .= '<p>' . $intro3 . '</p>' . "\n";
				
				$content .= '<p>' . $intro4 . '</p>' . "\n";
				
				$content .= '<p>' . $valid_tags . '</p>' . "\n";
				
				// Build the field(s) for the category's categories mapping template(s)
				$content .= '<p><label class="MappingTemplates">' . "\n";
				
				if ($num_languages == 1) {
					$content .= TEXT_LABEL_CATEGORY_SUBCATEGORIES_MAPPING_TEMPLATE;
				} else {
					$content .= TEXT_LABEL_CATEGORY_SUBCATEGORIES_MAPPING_TEMPLATES;
				}
				
				$content .= '</p>' . "\n";
				
				$content .= '<ul class="MappingTemplatesFields">' . "\n";
				
				for ($i = 0; $i < $num_languages; $i++) {
					$content .= '<li><span class="MappingLanguageIcon">' .
						zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
						$languages[$i]['image'], $languages[$i]['name']) . '</span>';
					
					$content .= ' <input type="text" name="categories-mapping-templates[' .
						$languages[$i]['id'] . ']" value="' .
						$this->_categories_mapping_templates[$languages[$i]['id']] . '" maxlength="255"'.
						' class="Textfield" />';
					
					if (isset($this->_error_messages['categories_mapping_templates']) &&
							isset($this->_error_messages['categories_mapping_templates'][$languages[$i]['id']])) {
						foreach ($this->_error_messages['categories_mapping_templates']
								[$languages[$i]['id']] as $error_message) {
							$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
						}
					}
					
					$content .= '</li>' . "\n";
				}
				
				$content .= '</ul>' . "\n";
				
				$content .= '<div class="SpacerSmall"></div>' . "\n";
			}
			
			// Build the intro for the category's product mapping template(s)
			$subsection_title = TEXT_MAPPING_TEMPLATES_SUBSECTION_TITLE_PRODUCTS;
			
			if ($this->_category_has_subcategories) {
				if ($num_languages == 1) {
					$intro = sprintf(TEXT_MANAGE_MAPPING_TEMPLATE_INTRO_CATEGORY_PRODUCTS,
						htmlentities($category_name, ENT_COMPAT, CHARSET));
				} else {
					$intro = sprintf(TEXT_MAPPING_TEMPLATES_INTRO_PRODUCTS_WITHIN_SUBS,
						htmlentities($category_name, ENT_COMPAT, CHARSET));
				}
				
				$intro2 = TEXT_MAPPING_TEMPLATES_INTRO_PRODUCTS_WITHIN_SUBS_WILL_USE_NEW_TEMPLATE;
				
			} else {
				if ($num_languages == 1) {
					$intro = sprintf(TEXT_MANAGE_MAPPING_TEMPLATE_INTRO_CATEGORY_PRODUCTS,
						htmlentities($category_name, ENT_COMPAT, CHARSET));
				} else {
					$intro = sprintf(TEXT_MAPPING_TEMPLATES_INTRO_PRODUCTS,
						htmlentities($category_name, ENT_COMPAT, CHARSET));
				}
				
				$intro2 = TEXT_MAPPING_TEMPLATES_INTRO_PRODUCTS_WILL_USE_NEW_TEMPLATE;
			}
			
			if ($num_languages == 1) {
				if ($this->_parent_category_id == 0) {
					$intro3 = TEXT_MAPPING_TEMPLATE_INTRO_PRODUCTS_OVERRIDE_DEFAULT;
				} else {
					$intro3 = TEXT_MAPPING_TEMPLATE_INTRO_PRODUCTS_OVERRIDE_PARENT;
				}
				
				$intro4 = TEXT_REMOVE_THE_TEMPLATE_BY_ENTERING_NOTHING;
				
			} else {
				if ($this->_parent_category_id == 0) {
					$intro3 = TEXT_MAPPING_TEMPLATES_INTRO_PRODUCTS_OVERRIDE_DEFAULT;
				} else {
					$intro3 = TEXT_MAPPING_TEMPLATES_INTRO_PRODUCTS_OVERRIDE_PARENT;
				}
				
				$intro4 = TEXT_REMOVE_A_TEMPLATE_BY_ENTERING_NOTHING;
			}
			
			$valid_tags = $this->_addTooltipDescriptionsOfPlacementTags('product',
				TEXT_VALID_PLACEMENT_TAGS_FOR_PRODUCT_MAPPING_TEMPLATE);
			
			$content .= '<h3>' . $subsection_title . '</h3>' . "\n";
			
			$content .= '<p>' . $intro . '</p>' . "\n";
			
			$content .= '<p>' . $intro2 . '</p>' . "\n";
			
			$content .= '<p>' . $intro3 . '</p>' . "\n";
			
			$content .= '<p>' . $intro4 . '</p>' . "\n";
			
			$content .= '<p>' . $valid_tags . '</p>' . "\n";
			
			// Build the field(s) for the category's products mapping template(s)
			$content .= '<p><label class="MappingTemplates">' . "\n";
			
			if ($num_languages == 1) {
				$content .= TEXT_LABEL_CATEGORY_PRODUCTS_MAPPING_TEMPLATE;
			} else {
				$content .= TEXT_LABEL_CATEGORY_PRODUCTS_MAPPING_TEMPLATES;
			}
			
			$content .= '</p>' . "\n";
			
			$content .= '<ul class="MappingTemplatesFields">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li><span class="MappingLanguageIcon">' . zen_image(DIR_WS_CATALOG_LANGUAGES .
					$languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) .
					'</span>';
				
				$content .= ' <input type="text" name="products-mapping-templates[' . $languages[$i]['id'] . ']"' .
					' value="' . $this->_products_mapping_templates[$languages[$i]['id']] .
					'" maxlength="255" class="Textfield" />';
				
				if (isset($this->_error_messages['products_mapping_templates']) &&
						isset($this->_error_messages['products_mapping_templates'][$languages[$i]['id']])) {
					foreach ($this->_error_messages['products_mapping_templates'][$languages[$i]['id']] as
							$error_message) {
						$content .= '<p class="MappingTemplateError">' . $error_message . '</p>' . "\n";
					}
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
		}
		
		if ($num_languages > 1 || ($this->_object_type == 'category' && $this->_category_has_subcategories)) {
			$button_text = TEXT_UPDATE_TEMPLATES;
		} else {
			$button_text = TEXT_UPDATE_TEMPLATE;
		}
		
		$content .= '<input type="submit" class="MappingTemplatesSubmit"' .
			' name="mapping-templates-submit" id="mapping-templates-submit" value="' . $button_text . '" />' .
			"\n"; 
		
		$content .= '</form>' . "\n";
		
		
		$content .= '</fieldset>' . "\n";
		
		
		// Build the back link
		$params = zen_get_all_get_params(array('manage-mapping-templates'));
		
		$back_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		if ($this->_object_type == 'category') {
			// Get the name of the category to go back to
			if ($this->_parent_category_id == 0) {
				$category_name = TEXT_ROOT_CATEGORY;
			} else {
				$category_info_result = $db->Execute("
					SELECT
						cd.categories_name
					FROM
						" . TABLE_CATEGORIES_DESCRIPTION . " cd
					WHERE
						cd.categories_id = '" . $this->_parent_category_id . "'
					AND
						cd.language_id = '" . (int) $_SESSION['languages_id'] . "';");
				
				if (!$category_info_result->EOF) {
					$category_name = $category_info_result->fields['categories_name'];
				}
			}
			
			$back_to = $category_name;
		}
		
		$content .= '<p class="BackLink">' . zen_image(DIR_WS_IMAGES . 'icons/ceon-umm-back-to.png') .
			' <a href="' . $back_link . '">' .
			sprintf(TEXT_BACK_TO, htmlentities($back_to, ENT_COMPAT, CHARSET)) . '</a></p>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _processMappingTemplatesFormSubmission()
	
	/**
	 * Handle the submission of mapping template(s).
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processMappingTemplatesFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		if ($this->_object_type == 'category') {
			for ($i = 0; $i < $num_languages; $i++) {
				// Does this category have subcategories?
				if ($this->_category_has_subcategories) {
					$subcategories_mapping_template =
						$this->_posted_data['categories-mapping-templates'][$languages[$i]['id']];
					
					if (strlen($subcategories_mapping_template) > 0) {
						$valid_template =
							$this->_validateURIMappingTemplate('category', $subcategories_mapping_template);
						
						if (is_array($valid_template)) {
							if (!isset($this->_error_messages['categories_mapping_templates'])) {
								$this->_error_messages['categories_mapping_templates'] = array();
							}
							
							if (!is_array($this->_error_messages['categories_mapping_templates']
									[$languages[$i]['id']])) {
								$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']] =
									array();
							}
							
							$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']][] =
								TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
							
							foreach ($valid_template as $template_error) {
								$this->_error_messages['categories_mapping_templates'][$languages[$i]['id']][] =
									$template_error;
							}
						} else {
							// Make sure the template complies with standard mapping rules of slash at start but
							// none at end
							$subcategories_mapping_template =
								$this->_cleanUpURIMappingTemplate($subcategories_mapping_template);
						}
					}
					
					$this->_categories_mapping_templates[$languages[$i]['id']] =
						$subcategories_mapping_template;
				}
				
				// Process the products mapping templates for this category
				$products_mapping_template =
					$this->_posted_data['products-mapping-templates'][$languages[$i]['id']];
				
				if (strlen($products_mapping_template) > 0) {
					$valid_template = $this->_validateURIMappingTemplate('product', $products_mapping_template);
					
					if (is_array($valid_template)) {
						if (!isset($this->_error_messages['products_mapping_templates'])) {
							$this->_error_messages['products_mapping_templates'] = array();
						}
						
						if (!is_array($this->_error_messages['products_mapping_templates']
								[$languages[$i]['id']])) {
							$this->_error_messages['products_mapping_templates'][$languages[$i]['id']] = array();
						}
						
						$this->_error_messages['products_mapping_templates'][$languages[$i]['id']][] =
							TEXT_ERROR_INVALID_MAPPING_TEMPLATE;
						
						foreach ($valid_template as $template_error) {
							$this->_error_messages['products_mapping_templates'][$languages[$i]['id']][] =
								$template_error;
						}
					} else {
						// Make sure the template complies with standard mapping rules of slash at start but none
						// at end
						$products_mapping_template = $this->_cleanUpURIMappingTemplate($products_mapping_template);
					}
				}
				
				$this->_products_mapping_templates[$languages[$i]['id']] =
					$products_mapping_template;
			}
			
			if (!isset($this->_error_messages['categories_mapping_templates']) &&
					!isset($this->_error_messages['products_mapping_templates'])) {
				// All templates valid!
				
				// Save all the mapping templates
				for ($i = 0; $i < $num_languages; $i++) {
					if ($this->_category_has_subcategories) {
						if ($this->_categories_mapping_templates[$languages[$i]['id']] == '') {
							// No template set so make sure there is none in the database!
							$mapping_template_deleted = $this->deleteURIMappingTemplate('category',
								$this->_category_id, $languages[$i]['id']);
							
							if ($mapping_template_deleted ==CEON_URI_MAPPING_MAPPING_TEMPLATE_DELETED) {
								$messageStack->add(sprintf(TEXT_CATEGORY_SUBCATEGORIES_MAPPING_TEMPLATE_DELETED,
									ucwords($languages[$i]['name'])), 'success');
							}
						} else {
							$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
								$this->_categories_mapping_templates[$languages[$i]['id']], 'category',
								$this->_category_id, $languages[$i]['id']);
							
							if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
								$messageStack->add(sprintf(TEXT_CATEGORY_SUBCATEGORIES_MAPPING_TEMPLATE_ADDED,
									ucwords($languages[$i]['name'])), 'success');
							} else if ($mapping_template_added_updated ==
									CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
								$messageStack->add(sprintf(TEXT_CATEGORY_SUBCATEGORIES_MAPPING_TEMPLATE_UPDATED,
									ucwords($languages[$i]['name'])), 'success');
							}
						}
					}
					
					if ($this->_products_mapping_templates[$languages[$i]['id']] == '') {
						// No template set so make sure there is none in the database!
						$mapping_template_deleted =
							$this->deleteURIMappingTemplate('product', $this->_category_id, $languages[$i]['id']);
						
						if ($mapping_template_deleted ==CEON_URI_MAPPING_MAPPING_TEMPLATE_DELETED) {
							$messageStack->add(sprintf(TEXT_CATEGORY_PRODUCTS_MAPPING_TEMPLATE_DELETED,
								ucwords($languages[$i]['name'])), 'success');
						}
					} else {
						$mapping_template_added_updated = $this->addUpdateURIMappingTemplate(
							$this->_products_mapping_templates[$languages[$i]['id']], 'product',
							$this->_category_id, $languages[$i]['id']);
						
						if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_ADDED) {
							$messageStack->add(sprintf(TEXT_CATEGORY_PRODUCTS_MAPPING_TEMPLATE_ADDED,
								ucwords($languages[$i]['name'])), 'success');
							
						} else if ($mapping_template_added_updated == CEON_URI_MAPPING_MAPPING_TEMPLATE_UPDATED) {
							$messageStack->add(sprintf(TEXT_CATEGORY_PRODUCTS_MAPPING_TEMPLATE_UPDATED,
								ucwords($languages[$i]['name'])), 'success');
						}
					}
				}
			}
		}
	}
	
	// }}}
}

// }}}
