<?php

/**
 * Ceon URI Mappings Manager Manage/Auto-generate "Other" Page URIs Class.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 * @version     $Id: class.CeonUMMCreateManageOtherPages.php 1059 2012-10-01 16:43:34Z conor $
 */

/**
 * Load in the Ceon URI Mappings Manager Listing class so it can be extended
 */
require_once(DIR_WS_CLASSES . 'class.CeonUMMListing.php');


// {{{ CeonUMMCreateManageOtherPages

/**
 * Handles URI management/auto-generation functionality for "other" pages.
 *
 * @package     ceon_uri_mappings_manager
 * @author      Conor Kerr <zen-cart.uri-mappings-manager@ceon.net>
 * @copyright   Copyright 2011-2012 Ceon
 * @copyright   Copyright 2003-2007 Zen Cart Development Team
 * @copyright   Portions Copyright 2003 osCommerce
 * @link        http://ceon.net/software/business/zen-cart/uri-mappings-manager
 * @license     http://www.zen-cart.com/license/2_0.txt GNU Public License V2.0
 */
class CeonUMMCreateManageOtherPages extends CeonUMMListing
{
	// {{{ properties
	
	/**
	 * The mapping templates being used by "other" Zen Cart pages for this instance at runtime. Indexed by language
	 * ID.
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_page_mapping_templates = null;
	
	/**
	 * The information about the "other" pages.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_pages = null;
	
	/**
	 * The information found out about the defines for "other" pages.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_defines_info = null;
	
	/**
	 * The list of which defines have been selected for "other" pages.
	 *
	 * @var     array
	 * @access  protected
	 */
	protected $_selected_defines = null;
	
	/**
	 * Variable holds a list of the template directory names defined for specific language IDs, as well as any
	 * default template directory for languages (which is assigned to the key 0).
	 *
	 * @var     array(string)
	 * @access  protected
	 */
	protected $_language_template_dirs = null;
	
	// }}}
	
	
	// {{{ Class Constructor
	
	/**
	 * Creates a new instance of the class.
	 * 
	 * @access  public
	 */
	public function __construct()
	{
		// Load the language definition file for the current language
		@include_once(DIR_WS_LANGUAGES . $_SESSION['language'] . '/' . 'ceon_umm_create_manage_other_pages.php');
		
		if (!defined('TEXT_OTHER_PAGES_MAPPINGS_TITLE') && $_SESSION['language'] != 'english') {
			// Fall back to english language file
			@include_once(DIR_WS_LANGUAGES . 'english/' . 'ceon_umm_create_manage_other_pages.php');
		}
		
		parent::__construct();
		
		// Run the actual functionality and build the output
		$this->_buildOtherPagesListing();
	}
	
	// }}}
	
	
	// {{{ _buildOtherPagesListing()
	
	/**
	 * Builds a list of all the "other" pages for the store, textfields to enter/change any mappings (en masse),
	 * and a gadget to autogenerate URIs for all the selected pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _buildOtherPagesListing()
	{
		global $db, $languages, $num_languages, $ceon_uri_mapping_demo, $messageStack;
		
		$content = '';
		
		// Get the mapping template(s) for other pages
		$this->_page_mapping_templates = array();
		
		for ($i = 0; $i < $num_languages; $i++) {
			$this->_page_mapping_templates[$languages[$i]['id']] =
				$this->getObjectURIMappingTemplate('other-page', $languages[$i]['id']);
		}
		
		// Initialise the main environment/data variables
		$this->_selected_items = array();
		$this->_mappings = array();
		$this->_defines_info = array();
		$this->_selected_defines = array();
		
		// Get the list of "other" Zen Cart pages which are supported by the software as well
		// @TODO this could be paged
		$this->_pages = array();
		
		$pages_result = $db->Execute("
			SELECT
				id,
				main_page,
				query_string_parameters,
				use_define_type,
				box_heading_define_name,
				heading_link_title_define_name,
				breadcrumb_define_name
            FROM
				" . TABLE_CEON_UMM_OTHER_PAGES . "
            WHERE
				1 = 1
			ORDER BY
				sort_order");
		
		while (!$pages_result->EOF) {
			$this->_pages[$pages_result->fields['id']] = array(
				'id' => $pages_result->fields['id'],
				'main_page' => $pages_result->fields['main_page'],
				'query_string_parameters' => $pages_result->fields['query_string_parameters'],
				'use_define_type' => $pages_result->fields['use_define_type'],
				'box_heading_define_name' => $pages_result->fields['box_heading_define_name'],
				'heading_link_title_define_name' => $pages_result->fields['heading_link_title_define_name'],
				'breadcrumb_define_name' => $pages_result->fields['breadcrumb_define_name']
				);
			
			$pages_result->MoveNext();
		}
		
		// Look up the text for, and info about, the defines for each page. Built here every time, even though this
		// is resource intensive, as this allows the user to change the define's file while looking at the
		// create/manage URIs for other pages listing
		foreach ($this->_pages as $page) {
			$this->_defines_info[$page['id']] = $this->_getZenCartPageDefinesText($page['main_page'],
				$page['box_heading_define_name'], $page['heading_link_title_define_name'],
				$page['breadcrumb_define_name']);
			
			// Use the default value for the define to be selected
			$this->_selected_defines[$page['id']] = $page['use_define_type'];
		}
		
		
		// Has the page been submitted, either for auto-generation or to have the auto-generation settings set or
		// the URIs manually updated?
		if (isset($this->_posted_data['auto-generate-selected-pages'])) {
			$this->_autogenerateOtherPagesMappings();
		} else if (isset($this->_posted_data['save-uris-autogeneration-settings'])) {
			$this->_processOtherPagesListingFormSubmission();
		} else {
			// Page being loaded for first time, load information and set up defaults
			
			// Get the current URI mapping(s) for each page, if any
			foreach ($this->_pages as $page) {
				$this->_mappings[$page['id']] = array();
				
				$columns_to_retrieve = array(
					'uri',
					'language_id'
					);
				
				$selections = array(
					'current_uri' => 1,
					'main_page' => $page['main_page'],
					'query_string_parameters' => $page['query_string_parameters']
					);
				
				$page_mapping_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections);
				
				while (!$page_mapping_result->EOF) {
					$this->_mappings[$page['id']][$page_mapping_result->fields['language_id']] =
						$page_mapping_result->fields['uri'];
					
					$page_mapping_result->MoveNext();
				}
			}
			
			// Use the default value for the define to be selected for each page
			foreach ($this->_pages as $page) {
				$this->_selected_defines[$page['id']] = $page['use_define_type'];
			}
			
			// Default to all listed pages being selected
			$this->_select_all = true;
			
			foreach ($this->_pages as $page) {
				$this->_selected_items[] = $page['id'];
			}
		}
		
		// All data has been gathered, build the page //////
		$panel_title = TEXT_OTHER_PAGES_MAPPINGS_TITLE;
		
		$content .= '<h1>' . $panel_title . '</h1>' . "\n";
		
		$content .= '<p>' . TEXT_OTHER_PAGES_MAPPINGS_INTRO . '</p>' . "\n";
		
		$content .= '<ul>' . "\n";
		
		$content .= '<li>' . TEXT_OTHER_PAGES_MAPPINGS_AUTOGENERATION_INTRO . '</li>' . "\n";
		
		$content .= '<li>' . TEXT_OTHER_PAGES_MAPPINGS_MANUAL_EDITING_INTRO . '</li>' . "\n";
		
		$content .= '<li>' . TEXT_OTHER_PAGES_MAPPINGS_MANUAL_EDITING_DELETE . '</li>' . "\n";
		
		$content .= '<li>' . TEXT_OTHER_PAGES_MAPPINGS_DEFINE_SOURCE_INFO . '</li>' . "\n";
		
		$content .= '</ul>' . "\n";
		
		// Add the form tag
		$form_action = zen_href_link(FILENAME_CEON_UMM, zen_get_all_get_params(), 'NONSSL');
		
		$content .= '<form action="' . $form_action .
			'" method="post" name="page-listing-form" id="page-listing-form">' . "\n";
		
		$content .= zen_draw_hidden_field('securityToken', $_SESSION['securityToken']);
		
		// Add the default button for the form, but align it offscreen. This means using the return key (pressing
		// enter) will submit the entered URIs rather than run the auto-generation
		$save_button_title = TEXT_SAVE_URIS_AND_AUTOGENERATION_PREFRENCES_BUTTON;
		$save_button_desc = TEXT_SAVE_URIS_AND_AUTOGENERATION_PREFRENCES_BUTTON_HOVER_INFO;
		
		$content .= '<input type="submit" class="SaveURIMappingsButton"' .
			' name="save-uris-autogeneration-settings" id="save-uris-autogeneration-settings"' .
			' value="' . $save_button_title . '" style="position: absolute; left: -100%;" />' . "\n";
		
		
		// Add the autogenerate controls for the other pages //////
		$content .= '<div class="AutogenerateControlsWrapper OtherPages" id="autogenerate-controls-top">' . "\n";
		
		$content .= '<fieldset class="AutogenerateControls"><legend>' . TEXT_AUTOGENERATE_MAPPINGS . "</legend>\n";
		
		$content .= '<p class="AutogenerationOptions">' . "\n";
		
		$content .= '<input type="radio" name="autogenerate-page-mappings"' .
			' value="exc-products" id="autogenerate-page-mappings"' .
			(1 == 1 ? ' checked="checked"' : '') . ' />' . "\n"; 
		
		$content .= '<label for="autogenerate-page-mappings">' . TEXT_SELECTED_PAGES . '</label>' . '</p>' . "\n";
		
		// Main button to generate URI mappings for all selected pages
		$button_title = TEXT_AUTOGENERATE_URI_MAPPINGS_BUTTON;
		$button_desc = TEXT_AUTOGENERATE_PAGE_TYPES_URI_MAPPINGS_BUTTON_HOVER_INFO;
		
		$content .= '<input type="submit" class="AutogenerateURIMappingsButton"' .
			' name="auto-generate-selected-pages" id="auto-generate-selected-pages"' .
			' value="' . $button_title . '" title="' . $button_desc . '" />' . "\n"; 
		
		$content .= '</fieldset>' . "\n";
		
		$content .= '</div>' . "\n";
		
		
		// Build the info for the other pages //////
		
		// Build the list of mapping templates for the other pages
		if ($num_languages == 1) {
			$content .= '<p class="MainMappingTemplatesIntro OtherPages"><label>' .
				TEXT_MAPPING_TEMPLATE_FOR_OTHER_PAGES . '&nbsp;' . '</label>' . '</p>' . "\n";
		} else {
			$content .= '<p class="MainMappingTemplatesIntro OtherPages"><label>' .
				TEXT_MAPPING_TEMPLATES_FOR_OTHER_PAGES . '&nbsp;' . '</label>' . '</p>' . "\n";
		}
		
		$content .= '<ul class="MainMappingTemplates">' . "\n";
		
		for ($i = 0; $i < $num_languages; $i++) {
			$content .= '<li>';
			
			$content .= '<span class="MappingLanguageIcon">' . zen_image(DIR_WS_CATALOG_LANGUAGES .
				$languages[$i]['directory'] . '/images/' . $languages[$i]['image'], $languages[$i]['name']) .
				'</span>';
			
			$content .= $this->_addTooltipDescriptionsOfPlacementTags('other-page',
				$this->_page_mapping_templates[$languages[$i]['id']]) . '</li>' . "\n";
		}
		
		$content .= '</ul>' . "\n";
		
		
		// Build the output for the pages
		$params = zen_get_all_get_params(array('listing', 'cat-id'));
		
		$back_link = zen_href_link(FILENAME_CEON_UMM, $params, 'NONSSL');
		
		$content .= '<p class="BackToSection TopLink">' . zen_image(DIR_WS_IMAGES . 'icons/ceon-umm-back-to.png') .
			' <a href="' . $back_link . '">' . TEXT_BACK_TO_CREATE_MANAGE_URIS . '</a></p>' . "\n";
		
		
		$select_all_text = addslashes(TEXT_TICK_TO_SELECT_ALL_PAGES);
		$deselect_all_text = addslashes(TEXT_TICK_TO_DESELECT_ALL_PAGES);
		
		$content .= <<<TABLE_JS
<script type="text/javascript">
<!--
function SelectAllPages(toggle_el, el_id)
{
	select_or_deselect = toggle_el.checked
	
	form_el = document.getElementById('page-listing-form');
	
	for (var t, i = 0; t = form_el.elements[el_id][i++]; t.checked=select_or_deselect);
	
	if (select_or_deselect) {
		toggle_el.title = '$deselect_all_text';
	} else {
		toggle_el.title = '$select_all_text';
	}
}
// -->
</script>
TABLE_JS;
		
		$content .= '<table cellpadding="0" cellspacing="0" id="page-listing" class="Listing">' .
			"\n";
		
		// Build header row
		$content .= '<tr>' . "\n";
		
		$content .= '<th id="checkboxes-col">' . '<input type="checkbox" name="select-all" value="1"' .
			' onclick="javascript:SelectAllPages(this, \'page[]\');"' .
			($this->_select_all ? ' checked="checked"' : '') . ' title="' .
			TEXT_TICK_TO_DESELECT_ALL_PAGES . '" />' . '</th>' . "\n";
		
		$content .= '<th>' . TEXT_PAGE . '</th>' . "\n";
		
		$content .= '<th>' . TEXT_AUTOGENERATION_PREFERENCE . '</th>' . "\n";
		
		$content .= '</tr>' . "\n";
		
		
		// Build page rows
		$num_pages = sizeof($this->_pages);
		
		$page_i = 0;
		
		foreach ($this->_pages as $page_id => $page_info) {
			$content .= '<tr class="' . (($page_i % 2 == 0) ? 'EvenRow' : 'OddRow') . '">' . "\n";
			
			$content .= '<td>' . "\n";
			
			$content .= '<input type="checkbox" name="page[]" value="' . $page_id . '"' .
				(in_array($page_id, $this->_selected_items) ? ' checked="checked"' : '') .
				' title="' . TEXT_TICK_TO_SELECT_PAGE_TYPE . '" class="Textfield" />' . "\n";
			
			$content .= '</td>' . "\n";
			
			
			// Build page type name and current URI mapping(s) for this page
			$content .= '<td>' . "\n";
			
			$content .= '<h3>' . htmlentities($page_info['main_page'], ENT_COMPAT, CHARSET) . '</h3>' . "\n";
			
			
			// Build the textfield(s) to enter/update/delete the mapping(s) for this page
			$content .= '<ul class="PageMappings">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				$content .= '<input type="hidden" name="prev-mappings[' . $page_id. '][' .
					$languages[$i]['id'] . ']" value="' .
					(!empty($this->_prev_mappings[$page_id][$languages[$i]['id']]) ? $this->_prev_mappings[$page_id][$languages[$i]['id']] : '') . '" />' . "\n";
				
				$content .= '<input type="text" name="mappings[' . $page_id. '][' .
					$languages[$i]['id'] . ']" value="' .
					(!empty($this->_mappings[$page_id][$languages[$i]['id']]) ? $this->_mappings[$page_id][$languages[$i]['id']] : '') . '" class="Textfield" />' . "\n";
				
				if (isset($this->_error_messages['mappings']) &&
						isset($this->_error_messages['mappings'][$page_id]) &&
						isset($this->_error_messages['mappings'][$page_id][$languages[$i]['id']])) {
					$content .= '<p class="MappingError">' .
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] . '</p>' . "\n";
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			$content .= '</td>' . "\n";
			
			
			// Build the autogeneration defines preferences
			$content .= '<td class="PageAutogenerationOptions">' . "\n";
			
			$content .= '<input type="radio" name="autogenerate-define-type[' . $page_id . ']"' .
				' value="' . CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME .
				'" id="autogenerate-define-type-box-heading"' .
				($this->_selected_defines[$page_id] == CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME ?
				' checked="checked"' : '') . ' class="AutogenerationOptionRadioGadget" />' . "\n";
			
			$content .= '<div class="AutogenerationOptionDesc">' . "\n";
			
			$content .= '<label for="autogenerate-define-type-box-heading">' .
				TEXT_DEFINE_TYPE_BOX_HEADING . ':</label>' . "\n";
			
			$content .= '<ul class="AutogenerationDefineText">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				if (isset($this->_defines_info[$page_id]['box_heading_define']
						[$languages[$i]['id']])) {
					// Got the value for this define, it can be used with auto-generation
					
					// Build the info about what defines were used when getting this value, so it can be displayed
					// in the title of the define's value
					$define_value_source_info = '';
					
					$define_sources = $this->_defines_info[$page_id]['box_heading_define'][$languages[$i]['id']]
						['define_sources'];
					
					$num_define_sources = sizeof($define_sources);
					
					$define_value_source_info = sprintf(TEXT_USES_DEFINE_IN_FILE,
						$define_sources[0]['define_name'],
						htmlentities(zen_break_string($define_sources[0]['definitions_file_path'], 50, "\n"), ENT_COMPAT, CHARSET, true));
					
					for ($define_source_i = 1; $define_source_i < $num_define_sources; $define_source_i++) {
						$define_value_source_info .= sprintf(TEXT_ALSO_USES_DEFINE_IN_FILE,
							$define_sources[$define_source_i]['define_name'],
							htmlentities(zen_break_string($define_sources[$define_source_i]['definitions_file_path'], 50, "\n"), ENT_COMPAT, CHARSET, true));
					}
					
					$content .= '<span title="' . $define_value_source_info . '">';
					
					$content .= htmlentities($this->_defines_info[$page_id]['box_heading_define']
						[$languages[$i]['id']]['define_value'], ENT_COMPAT, CHARSET, false) . "\n";
					
					$content .= '</span>';
					
				} else {
					$content .=
						'<span class="AutogenerationOptionNoValueFound">' . TEXT_NO_DEFINE_VALUE_FOUND . '</span>';
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			$content .= '</div>' . "\n";
			
			$content .= '<div class="SpacerSmall"></div>' . "\n";
			
			
			$content .= '<input type="radio" name="autogenerate-define-type[' . $page_id . ']"' .
				' value="' . CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME .
				'" id="autogenerate-define-type-heading-link-title"' .
				($this->_selected_defines[$page_id] == CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME ?
				' checked="checked"' : '') . ' class="AutogenerationOptionRadioGadget" />' . "\n"; 
			
			$content .= '<div class="AutogenerationOptionDesc">' . "\n";
			
			$content .= '<label for="autogenerate-define-type-heading-link-title">' .
				TEXT_DEFINE_TYPE_HEADING_LINK_TITLE . ':</label>' . "\n";
			
			$content .= '<ul class="AutogenerationDefineText">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				if (isset($this->_defines_info[$page_id]['heading_link_title_define'][$languages[$i]['id']])) {
					// Got the value for this define, it can be used with auto-generation
					
					// Build the info about what defines were used when getting this value, so it can be displayed
					// in the title of the define's value
					$define_value_source_info = '';
					
					$define_sources = $this->_defines_info[$page_id]['heading_link_title_define']
						[$languages[$i]['id']]['define_sources'];
					
					$num_define_sources = sizeof($define_sources);
					
					$define_value_source_info = sprintf(TEXT_USES_DEFINE_IN_FILE,
						$define_sources[0]['define_name'],
						htmlentities(zen_break_string($define_sources[0]['definitions_file_path'], 50, "\n"), ENT_COMPAT, CHARSET, true));
					
					for ($define_source_i = 1; $define_source_i < $num_define_sources; $define_source_i++) {
						$define_value_source_info .= sprintf(TEXT_ALSO_USES_DEFINE_IN_FILE,
							$define_sources[$define_source_i]['define_name'],
							htmlentities(zen_break_string($define_sources[$define_source_i]['definitions_file_path'], 50, "\n"), ENT_COMPAT, CHARSET, true));
					}
					
					$content .= '<span title="' . $define_value_source_info . '">';
					
					$content .= htmlentities($this->_defines_info[$page_id]
						['heading_link_title_define'][$languages[$i]['id']]['define_value'],ENT_COMPAT, CHARSET, false) .
						"\n";
					
					$content .= '</span>';
					
				} else {
					$content .=
						'<span class="AutogenerationOptionNoValueFound">' . TEXT_NO_DEFINE_VALUE_FOUND . '</span>';
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			$content .= '</div>' . "\n";
			
			$content .= '<div class="SpacerSmall"></div>' . "\n";
			
			
			$content .= '<input type="radio" name="autogenerate-define-type[' . $page_id . ']"' .
				' value="' . CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME .
				'" id="autogenerate-define-type-breadcrumb"' .
				($this->_selected_defines[$page_id] == CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME ?
				' checked="checked"' : '') . ' class="AutogenerationOptionRadioGadget" />' . "\n"; 
			
			$content .= '<div class="AutogenerationOptionDesc">' . "\n";
			
			$content .= '<label for="autogenerate-define-type-breadcrumb">' . TEXT_DEFINE_TYPE_BREADCRUMB .
				':</label>' . "\n";
			
			$content .= '<ul class="AutogenerationDefineText">' . "\n";
			
			for ($i = 0; $i < $num_languages; $i++) {
				$content .= '<li>';
				
				$content .= '<span class="MappingLanguageIcon">' .
					zen_image(DIR_WS_CATALOG_LANGUAGES . $languages[$i]['directory'] . '/images/' .
					$languages[$i]['image'], $languages[$i]['name']) . '</span>';
				
				if (isset($this->_defines_info[$page_id]['breadcrumb_define']
						[$languages[$i]['id']])) {
					// Got the value for this define, it can be used with auto-generation
					
					// Build the info about what defines were used when getting this value, so it can be displayed
					// in the title of the define's value
					$define_value_source_info = '';
					
					$define_sources =
						$this->_defines_info[$page_id]['breadcrumb_define'][$languages[$i]['id']]['define_sources'];
					
					$num_define_sources = sizeof($define_sources);
					
					$define_value_source_info = sprintf(TEXT_USES_DEFINE_IN_FILE,
						$define_sources[0]['define_name'],
						htmlentities(zen_break_string($define_sources[0]['definitions_file_path'], 50, "\n"), ENT_COMPAT, CHARSET, true));
					
					for ($define_source_i = 1; $define_source_i < $num_define_sources; $define_source_i++) {
						$define_value_source_info .= sprintf(TEXT_ALSO_USES_DEFINE_IN_FILE,
							$define_sources[$define_source_i]['define_name'],
							htmlentities(zen_break_string($define_sources[$define_source_i]['definitions_file_path'], 50, "\n"), ENT_COMPAT, CHARSET, true));
					}
					
					$content .= '<span title="' . $define_value_source_info . '">';
					
					$content .= htmlentities($this->_defines_info[$page_id]['breadcrumb_define']
						[$languages[$i]['id']]['define_value'], ENT_COMPAT, CHARSET, false) . "\n";
					
					$content .= '</span>';
					
				} else {
					$content .=
						'<span class="AutogenerationOptionNoValueFound">' . TEXT_NO_DEFINE_VALUE_FOUND . '</span>';
				}
				
				$content .= '</li>' . "\n";
			}
			
			$content .= '</ul>' . "\n";
			
			$content .= '</div>' . "\n";
			
			
			if (isset($this->_error_messages['defines']) && isset($this->_error_messages['defines'][$page_id])) {
				$content .= '<p class="FormError">' . $this->_error_messages['defines'][$page_id] . '</p>' . "\n";
			}
			
			$content .= '</td>' . "\n";
			
			
			$content .= '</tr>' . "\n";
			
			$page_i++;
		}
		
		$content .= '</table>' . "\n";
		
		
		// Main button to save any entered/modified URIs and the autogeneration preferences
		$content .= '<p class="SaveURIMappingsButton">' . "\n";
		
		$content .= '<input type="submit" class="SaveURIMappingsButton"' .
			' name="save-uris-autogeneration-settings" id="save-uris-autogeneration-settings"' .
			' value="' . $save_button_title . '" title="' . $save_button_desc . '" />' . "\n"; 
		
		$content .= '</p>';
		
		$content .= '<div class="SpacerSmall"></div>' . "\n";
		
		
		// Build the back to top link
		$content .= '<p class="BackToTop"><a href="#top">' . TEXT_BACK_TO_TOP .
			zen_image(DIR_WS_IMAGES . 'icons/' . 'ceon-umm-back-to-top.png') . '</a></p>' . "\n";
		
		
		// Build the "back to create/manage URIs" link
		$content .= '<p class="BackToSection">' . zen_image(DIR_WS_IMAGES . 'icons/ceon-umm-back-to.png') .
			' <a href="' . $back_link . '">' . TEXT_BACK_TO_CREATE_MANAGE_URIS . '</a></p>' . "\n";
		
		
		$content .= '</form>' . "\n";
		
		
		$this->_output = $content;
	}
	
	// }}}
	
	
	// {{{ _autogenerateOtherPagesMappings()
	
	/**
	 * Auto-generate the mappings for all the selected "other" pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _autogenerateOtherPagesMappings()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		$this->_handleItemSelections('page');
		
		$this->_handleOtherPagesDefinesSubmission();
		
		// Get the current URI mapping(s) for each "other" page, if any
		// Time is taken to load the information here instead of using data stored in a form, mainly because a
		// "refresh" of a posted page can include out of date data/relationships
		$this->_setPrevURIMappingsForOtherPages();
		
		foreach ($this->_pages as $page_id => $page_info) {
			if (!in_array($page_id, $this->_selected_items)) {
				// Not selected for auto-generation, simply store values for display
				$this->_mappings[$page_id] = isset($this->_prev_mappings[$page_id]) ? $this->_prev_mappings[$page_id] : '';
				
				continue;
			}
			
			// URI mapping is to be auto-generated
			
			// Get the value for the define for this page
			$define_type_id = $this->_selected_defines[$page_id];
			
			// Warn the user if they've selected a define which has no value
			switch ($define_type_id) {
				case CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME:
					$define_type_name = 'box_heading_define';
					break;
				case CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME:
					$define_type_name = 'heading_link_title_define';
					break;
				case CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME:
					$define_type_name = 'breadcrumb_define';
					break;
			}
			
			for ($i = 0; $i < $num_languages; $i++) {
				$prev_mapping = isset($this->_prev_mappings[$page_id][$languages[$i]['id']]) ? $this->_prev_mappings[$page_id][$languages[$i]['id']] : '';
				
				if (!isset($this->_defines_info[$page_id][$define_type_name][$languages[$i]['id']])) {
					// No value found for the selected define
					
					// If the page type exists/is used by the store then the user should be alerted
					if (!$this->_pageTypeExists($page_id)) {
						// Page type folder not found so don't count this as an error
						
						// No new mapping generated so restore mapping's value
						$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
						
						continue;
					}
					
					if (!isset($this->_error_messages['mappings'])) {
						$this->_error_messages['mappings'] = array();
					}
					
					if (!isset($this->_error_messages['mappings'][$page_id])) {
						$this->_error_messages['mappings'][$page_id] = array();
					}
					
					$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
						sprintf(TEXT_UNABLE_TO_AUTOGENERATE_URI_FOR_EMPTY_DEFINE, ucwords($languages[$i]['name']));
					
					// No new mapping generated so restore mapping's value
					$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
					
					continue;
				}
				
				// Generate URI using define's value
				$define_text =
					$this->_defines_info[$page_id][$define_type_name][$languages[$i]['id']]['define_value'];
				
				// Apply the define's value to the template for "other" pages
				$mapping = str_replace('{page-define-text}',
					$this->_convertStringForURI($define_text, $languages[$i]['code'], false),
					$this->_page_mapping_templates[$languages[$i]['id']]);
				
				// Add the path to the store if it is part of the template
				$mapping = str_replace('{dir-ws-catalog}', DIR_WS_CATALOG, $mapping);
				
				$mapping = $this->_cleanUpURIMapping($mapping);
				
				if (strlen($mapping) > 0 && $mapping != $prev_mapping) {
					// New mapping has been generated
					
					// Check that the mapping just generated doesn't clash with any existing mappings, so the user
					// can be notified
					$columns_to_retrieve = array(
						'main_page',
						'associated_db_id',
						'query_string_parameters'
						);
					
					$selections = array(
						'uri' => zen_db_prepare_input($mapping),
						'current_uri' => 1,
						'language_id' => $languages[$i]['id']
						);
					
					$order_by = null;
					
					$existing_uri_mapping_result =
						$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
					
					// If the existing mapping is simply having some capitalisation changed then a case insensitive
					// comparison might result in a false positive for a mapping clash, so prevent that by checking
					// the mapping's settings don't match
					if (!$existing_uri_mapping_result->EOF &&
							!($existing_uri_mapping_result->fields['main_page'] == $page_info['main_page'] &&
							is_null($existing_uri_mapping_result->fields['associated_db_id']) &&
							$existing_uri_mapping_result->fields['query_string_parameters'] ==
							$page_info['query_string_parameters'])) {
						// This mapping clashes with an existing mapping 
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!isset($this->_error_messages['mappings'][$page_id]) || !is_array($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
							sprintf(TEXT_ERROR_AUTOGENERATED_URI_CLASHES, '<a href="' .
							HTTP_SERVER . $mapping . '" target="_blank">' . $mapping . '</a>');
						
						// No new mapping generated so restore mapping's value
						$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
						
						// Can't create/save mapping
						continue;
					}
				}
				
				$insert_mapping = false;
				$update_mapping = false;
				
				if ($mapping != '') {
					// Check if the URI mapping is being updated or does not yet exist
					if ($prev_mapping == '') {
						$insert_mapping = true;
					} else if ($prev_mapping != $mapping) {
						$update_mapping = true;
					}
				}
				
				if ($insert_mapping || $update_mapping) {
					if ($update_mapping) {
						// Consign previous mapping to the history, so old URI mapping isn't broken
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					}
					
					// Add the new URI mapping
					$uri = $mapping;
					
					$main_page = $page_info['main_page'];
					
					$query_string_parameters = $page_info['query_string_parameters'];
					
					$mapping_added = $this->addURIMapping($uri, $languages[$i]['id'], $main_page,
						$query_string_parameters, null);
					
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
						if ($insert_mapping) {
							$success_message = sprintf(TEXT_OTHER_PAGE_MAPPING_ADDED,
								ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>');
						} else {
							$success_message = sprintf(TEXT_OTHER_PAGE_MAPPING_UPDATED,
								ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>');
						}
						
						$messageStack->add($success_message, 'success');
						
					} else {
						if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS,
								ucwords($languages[$i]['name'])) . '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>';
							
						} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_DATA,
								ucwords($languages[$i]['name'])) . $uri;
						} else {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_DB,
								ucwords($languages[$i]['name'])) . $uri;
						}
						
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!is_array($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] = $failure_message;
						
						// No new mapping generated so restore mapping's value
						$this->_mappings[$page_id][$languages[$i]['id']] = $prev_mapping;
						
						continue;
					}
				} else if ($prev_mapping != '' && $mapping == '') {
					// No URI mapping, consign existing mapping to the history, so old URI mapping isn't broken
					$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					
					$success_message = sprintf(TEXT_OTHER_PAGE_MAPPING_MADE_HISTORICAL,
						ucwords($languages[$i]['name']), $page_info['main_page']);
					
					$messageStack->add($success_message, 'caution');
				}
				
				// Mapping's value has been updated
				$this->_mappings[$page_id][$languages[$i]['id']] = $mapping;
			}
		}
	}
	
	// }}}
	
	
	// {{{ _processOtherPagesListingFormSubmission()
	
	/**
	 * Handle the submission of the other pages listing. Creates/updates any mappings, makes any cleared mappings
	 * historical and saves the autogeneration define preferences for the pages.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _processOtherPagesListingFormSubmission()
	{
		global $db, $languages, $num_languages, $messageStack;
		
		$this->_handleItemSelections('page');
		
		$this->_handleOtherPagesDefinesSubmission();
		
		// Get the current URI mapping(s) for each "other" page, if any
		// Time is taken to load the information here instead of using data stored in a form, mainly because a
		// "refresh" of a posted page can include out of date data/relationships
		$this->_setPrevURIMappingsForOtherPages();
		
		// Store the values in the textfields for the page mappings, then check each one to see if a mapping should
		// be created, changed or made historical
		foreach ($this->_pages as $page_id => $page_info) {
			$this->_mappings[$page_id] = array();
			
			for ($i = 0; $i < $num_languages; $i++) {
				$this->_mappings[$page_id][$languages[$i]['id']] = $this->_cleanUpURIMapping(
					$this->_posted_data['mappings'][$page_id][$languages[$i]['id']]);
				
				$prev_mapping = isset($this->_prev_mappings[$page_id][$languages[$i]['id']]) ? $this->_prev_mappings[$page_id][$languages[$i]['id']] : '';
				
				$mapping = $this->_mappings[$page_id][$languages[$i]['id']];
				
				$main_page = $page_info['main_page'];
				
				$query_string_parameters = $page_info['query_string_parameters'];
				
				if (strlen($mapping) > 0 && $mapping != $prev_mapping) {
					// New/updated mapping has been specified
					
					// Check that the mapping(s) just entered doesn't/don't clash with any existing mapping(s), so
					// the user can be notified
					$mapping_clashed = false;
					
					$columns_to_retrieve = array(
						'main_page',
						'associated_db_id',
						'query_string_parameters'
						);
					
					$selections = array(
						'uri' => zen_db_prepare_input($mapping),
						'current_uri' => 1,
						'language_id' => $languages[$i]['id']
						);
					
					$order_by = null;
					
					$existing_uri_mapping_result =
						$this->getURIMappingsResultset($columns_to_retrieve, $selections, $order_by, 1);
					
					// If the existing mapping is simply having some capitalisation changed then a case insensitive
					// comparison might result in a false positive for a mapping clash, so prevent that by checking
					// the mapping's settings don't match
					if (!$existing_uri_mapping_result->EOF &&
							!($existing_uri_mapping_result->fields['main_page'] == $page_info['main_page'] &&
							is_null($existing_uri_mapping_result->fields['associated_db_id']) &&
							$existing_uri_mapping_result->fields['query_string_parameters'] ==
							$page_info['query_string_parameters'])) {
						// This mapping clashes with an existing mapping
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!isset($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] =
							sprintf(TEXT_ERROR_PAGE_URI_ENTERED_CLASHES, '<a href="' . HTTP_SERVER . $mapping .
							'" target="_blank">' . $mapping . '</a>');
						
						continue;
					}
				}
				
				$insert_mapping = false;
				$update_mapping = false;
				
				if ($mapping != '') {
					// Check if the URI mapping is being updated or does not yet exist
					if ($prev_mapping == '') {
						$insert_mapping = true;
					} else if ($prev_mapping != $mapping) {
						$update_mapping = true;
					}
				}
				
				if ($insert_mapping || $update_mapping) {
					if ($update_mapping) {
						// Consign previous mapping to the history, so old URI mapping isn't broken
						$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					}
					
					// Add the new URI mapping
					$uri = $mapping;
					
					$mapping_added = $this->addURIMapping($uri, $languages[$i]['id'], $main_page,
						$query_string_parameters, null);
					
					if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_SUCCESS) {
						if ($insert_mapping) {
							$success_message = sprintf(TEXT_OTHER_PAGE_MAPPING_ADDED,
								ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>');
						} else {
							$success_message = sprintf(TEXT_OTHER_PAGE_MAPPING_UPDATED,
								ucwords($languages[$i]['name']), '<a href="' . HTTP_SERVER . $uri .
								'" target="_blank">' . $uri . '</a>');
						}
						
						$messageStack->add($success_message, 'success');
						
					} else {
						if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_MAPPING_EXISTS) {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_EXISTS,
								ucwords($languages[$i]['name'])) . '<a href="' . HTTP_SERVER .
								$uri . '" target="_blank">' . $uri . '</a>';
							
						} else if ($mapping_added == CEON_URI_MAPPING_ADD_MAPPING_ERROR_DATA_ERROR) {
							$failure_message = sprintf(TEXT_ERROR_ADD_MAPPING_DATA,
								ucwords($languages[$i]['name'])) . $uri;
						} else {
							$failure_message =
								sprintf(TEXT_ERROR_ADD_MAPPING_DB, ucwords($languages[$i]['name'])) . $uri;
						}
						
						if (!isset($this->_error_messages['mappings'])) {
							$this->_error_messages['mappings'] = array();
						}
						
						if (!isset($this->_error_messages['mappings'][$page_id])) {
							$this->_error_messages['mappings'][$page_id] = array();
						}
						
						$this->_error_messages['mappings'][$page_id][$languages[$i]['id']] = $failure_message;
					}
				} else if ($prev_mapping != '' && $mapping == '') {
					// No URI mapping, consign existing mapping to the history, so old URI mapping isn't broken
					$this->makeURIMappingHistorical($prev_mapping, $languages[$i]['id']);
					
					$success_message = sprintf(TEXT_OTHER_PAGE_MAPPING_MADE_HISTORICAL,
						ucwords($languages[$i]['name']), $main_page);
					
					$messageStack->add($success_message, 'caution');
				}
			}
		}
	}
	
	// }}}
	
	
	// {{{ _setPrevURIMappingsForOtherPages()
	
	/**
	 * Gets the current URI mappings for "other" pages and stores them in the "previous mappings" variable.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _setPrevURIMappingsForOtherPages()
	{
		global $db;
		
		foreach ($this->_pages as $page_id => $page_info) {
			$this->_prev_mappings[$page_id] = array();
			
			$columns_to_retrieve = array(
				'uri',
				'language_id'
				);
			
			$selections = array(
				'current_uri' => 1,
				'main_page' => $page_info['main_page'],
				'query_string_parameters' => $page_info['query_string_parameters']
				);
			
			$page_mappings_result = $this->getURIMappingsResultset($columns_to_retrieve, $selections);
			
			while (!$page_mappings_result->EOF) {
				$this->_prev_mappings[$page_id][$page_mappings_result->fields['language_id']] =
					$page_mappings_result->fields['uri'];
				
				$page_mappings_result->MoveNext();
			}
		}
	}
	
	// }}}
	
	
	// {{{ _handleOtherPagesDefinesSubmission()
	
	/**
	 * Saves the current define type selections for the pages of the other pages listing.
	 *
	 * @access  protected
	 * @return  none
	 */
	protected function _handleOtherPagesDefinesSubmission()
	{
		global $db, $languages, $num_languages;
		
		foreach ($this->_posted_data['autogenerate-define-type'] as $page_id => $define_type_id) {
			$this->_selected_defines[$page_id] = $define_type_id;
			
			$db->Execute("
				UPDATE
					" . TABLE_CEON_UMM_OTHER_PAGES . "
				SET
					use_define_type = '" . (int) $define_type_id . "'
				WHERE
					id = '" . (int) $page_id . "';");
			
			// Warn the user if they've selected a define which has no value
			switch ($define_type_id) {
				case CEON_URI_MAPPING_USE_BOX_HEADING_DEFINE_NAME:
					$define_type_name = 'box_heading_define';
					break;
				case CEON_URI_MAPPING_USE_HEADING_LINK_DEFINE_NAME:
					$define_type_name = 'heading_link_title_define';
					break;
				case CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME:
					$define_type_name = 'breadcrumb_define';
					break;
			}
			
			for ($i = 0; $i < $num_languages; $i++) {
				if (!isset($this->_defines_info[$page_id][$define_type_name][$languages[$i]['id']])) {
					// No value found for the selected define
					
					// If the page type exists/is used by the store then the user should be alerted
					if (!$this->_pageTypeExists($page_id)) {
						// Page type folder not found so don't count this as an error
						continue;
					}
					
					if (!isset($this->_error_messages['defines'])) {
						$this->_error_messages['defines'] = array();
					}
					
					if (!isset($this->_error_messages['defines'][$page_id])) {
						$this->_error_messages['defines'][$page_id] = array();
					}
					
					if ($define_type_id != CEON_URI_MAPPING_USE_BREADCRUMB_DEFINE_NAME) {
						$this->_error_messages['defines'][$page_id] = sprintf(
							TEXT_DEFINE_SELECTED_HAS_NO_VALUE, ucwords($languages[$i]['name']));
					} else {
						// Must at least have a define for the breadcrumb, as a fall back
						$this->_error_messages['defines'][$page_id] = sprintf(
							TEXT_DEFINE_SELECTED_HAS_NO_VALUE_NORMALLY_PAGE_HAS_BREADCRUMB,
							ucwords($languages[$i]['name']));
					}
				}
			}
		}
		
	}
	
	// }}
	
	
	// {{{ _pageTypeExists()
	
	/**
	 * Checks if the specified page type exists for this store by checking if the folder for the page exists in the
	 * modules directory. Requires that the information about the list of the page types set up for this module has
	 * already been built.
	 *
	 * @access  protected
	 * @param   string    $page_id   The ID of the page in the list of page types.
	 * @return  boolean   True if the page type exists for the store, false if it doesn't exist.
	 */
	protected function _pageTypeExists($page_id)
	{
		if (!file_exists(DIR_FS_CATALOG . DIR_WS_MODULES . 'pages/' . $this->_pages[$page_id]['main_page'])) {
			return false;
		}
		
		return true;
	}
	
	// }}}
	
	
	// {{{ _getZenCartPageDefinesText()
	
	/**
	 * Looks up the text for the defines for the specified page type.
	 *
	 * @access  protected
	 * @param   string    $page_type   The type of page.
	 * @param   string    $box_heading_define_name     The name of the box heading define name associated with the
	 *                                                 page type.
	 * @param   string    $heading_link_title_define_name   The name of the heading link title  define name
	 *                                                      associated with the page type.
	 * @param   string    $box_heading_define_name   The name of the breadcrumb define name associated with the
	 *                                               page type.
	 * @return  array(array(string))   An array with the text found for the 3 define types, and the define name
	 *                                 tested against, indexed by the language ID.
	 */
	protected function _getZenCartPageDefinesText($page_type, $box_heading_define_name, $heading_link_title_define_name,
		$breadcrumb_define_name)
	{
		global $db, $languages, $num_languages;
		
		$defines_info = array(
			'box_heading_define' => array(),
			'heading_link_title_define' => array(),
			'breadcrumb_define' => array()
			);
		
		if (is_null($this->_language_template_dirs)) {
			$this->_language_template_dirs = array();
			
			// Get the list of templates for the various languages in use on the site
			$templates_query = "
				SELECT
					*
				FROM
					" . TABLE_TEMPLATE_SELECT;
			
			$templates = $db->Execute($templates_query);
			
			while (!$templates->EOF) {
				$this->_language_template_dirs[$templates->fields['template_language']] =
					$templates->fields['template_dir'];
				
				$templates->MoveNext();
			}
		}
		
		$define_names_to_try = array();
		
		if (!is_null($box_heading_define_name)) {
			$define_names_to_try[] = array(
				'define_type' => 'box_heading_define',
				'define_name' => $box_heading_define_name
				);
		}
		
		if (!is_null($heading_link_title_define_name)) {
			$define_names_to_try[] = array(
				'define_type' => 'heading_link_title_define',
				'define_name' => $heading_link_title_define_name
				);
		}
		
		if (!is_null($breadcrumb_define_name)) {
			$define_names_to_try[] = array(
				'define_type' => 'breadcrumb_define',
				'define_name' => $breadcrumb_define_name
				);
		}
		
		// Fall back to default breadcrumbs for a page
		$define_names_to_try[] = array(
			'define_type' => 'breadcrumb_define',
			'define_name' => 'NAVBAR_TITLE_1'
			);
		
		$define_names_to_try[] = array(
			'define_type' => 'breadcrumb_define',
			'define_name' => 'NAVBAR_TITLE'
			);
		
		for ($lang_i = 0; $lang_i < $num_languages; $lang_i++) {
			// Set up the variables which are used to build the base paths for loading the language definition
			// files
			$language_directory = $languages[$lang_i]['directory'];
			
			if (isset($this->_language_template_dirs[$languages[$lang_i]['id']])) {
				$template_directory = $this->_language_template_dirs[$languages[$lang_i]['id']];
			} else {
				// No specific template for this language, use the default template directory for the store
				$template_directory = $this->_language_template_dirs[0];
			}
			
			foreach ($define_names_to_try as $define_name_to_try) {
				// Only need one match per define type per language
				if (isset($defines_info[$define_name_to_try['define_type']][$languages[$lang_i]['id']])) {
					continue;
				}
				
				// Handle define names which are arrays (delimited by space(s))
				if (strpos($define_name_to_try['define_name'], ' ') !== false) {
					$define_name_parts = explode(' ', $define_name_to_try['define_name']);
				} else {
					$define_name_parts = array($define_name_to_try['define_name']);
				}
				
				$num_define_name_parts = sizeof($define_name_parts);
				
				$define_value_parts = array();
				
				foreach ($define_name_parts as $current_define_name_part) {
					$define_value_info = $this->_getDefineValue($current_define_name_part, $language_directory,
						$template_directory, $page_type, true);
					
					if (sizeof($define_value_info['value_parts']) > 0) {
						$define_value = implode(' ', $define_value_info['value_parts']);
						
						$define_value = html_entity_decode($define_value);
						
						$define_value_parts[] =
							array(
								'define_value' => $define_value,
								'define_sources' => $define_value_info['define_sources']
								);
					}
				}
				
				$num_define_value_parts = sizeof($define_value_parts);
				
				// Only use the define values if all parts were matched
				if ($num_define_value_parts == $num_define_name_parts) {
					// Convert mutliple parts into a heirarchy by joining each with a slash
					$define_value = '';
					$define_sources = array();
					
					for ($value_part_i = 0; $value_part_i < $num_define_value_parts;
							$value_part_i++) {
						$define_value .= $define_value_parts[$value_part_i]['define_value'];
						
						if ($value_part_i < $num_define_value_parts - 1) {
							$define_value .= '/';
						}
						
						$define_sources = array_merge($define_sources,
							$define_value_parts[$value_part_i]['define_sources']);
					}
					
					$defines_info[$define_name_to_try['define_type']][$languages[$lang_i]['id']] =
						array(
							'define_value' => $define_value,
							'define_sources' => $define_sources
							);
				}
			}
		}
		
		return $defines_info;
	}
	
	// }}}
	
	
	// {{{ _getDefineValue()
	
	/**
	 * Attempts to find the specified define in the most pertinent language file for the specified page type.
	 *
	 * If an override file is found for the specified file and the define is not present within it then an attempt
	 * is made to find it in the page's default language file.
	 *
	 * If the define isn't found in the default language file then a final attempt is made to find it in the main
	 * language file for the site for the specified language.
	 *
	 * @access  protected
	 * @param   string    $define_name          The name of the define for which the value should be parsed and
	 *                                          returned.
	 * @param   string    $language_directory   The name of the directory for the language the define uses.
	 * @param   string    $template_directory   The name of the directory for the template for the language the
	 *                                          define uses.
	 * @param   string    $page_type            The name of the Zen Cart page on which the define is being used.
	 * @param   boolean   $lookup_embedded_define_values   Whether or not to look up the value for any embedded
	 *                                                     values.
	 * @return  array|false   An array containing the parts of the define's value that were parsed or false if an
	 *                        error occurred when parsing (either because the define's value used incorrect
	 *                        formatting or, when looking up values for embedded defines, no value could be found
	 *                        for an embedded define.
	 */
	protected function _getDefineValue($define_name, $language_directory, $template_directory, $page_type,
		$lookup_embedded_define_values)
	{
		$define_value_info = array(
			'value_parts' => array(),
			'define_sources' => array(),
			'errors' => array()
			);
		
		$language_file_source = null;
		
		$language_files_to_try = array();
		
		// First try to find the define in the override language file for the specified page
		$language_files_to_try[] = DIR_FS_CATALOG . DIR_WS_LANGUAGES . $language_directory . '/' .
			$template_directory . '/' . $page_type . '.php';
		
		// Next, try to find the define in the any the standard language file for the specified page
		$language_files_to_try[] =
			DIR_FS_CATALOG . DIR_WS_LANGUAGES . $language_directory . '/' . $page_type . '.php';
		
		// Fall back to trying to find the define in the main language file for the site for the
		// specified language.
		$language_files_to_try[] = DIR_FS_CATALOG . DIR_WS_LANGUAGES . $template_directory . '/' .
			$language_directory . '.php';
		
		$language_files_to_try[] = DIR_FS_CATALOG . DIR_WS_LANGUAGES . $language_directory . '.php';
		
		// Finally try the header language file, it sometimes has some extra defines
		$language_files_to_try[] = DIR_FS_CATALOG . DIR_WS_LANGUAGES . $language_directory . '/' .
			$template_directory . '/header.php';
		
		$language_files_to_try[] = DIR_FS_CATALOG . DIR_WS_LANGUAGES . $language_directory . '/' . 'header.php';
		
		$file_found = false;
		
		foreach ($language_files_to_try as $language_file_to_try) {
			
			if (!file_exists($language_file_to_try)) {
				continue;
			}
			
			$file_found = true;
			
			$language_file_source = file_get_contents($language_file_to_try);
			
			if (preg_match_all('|[\r\n;]+[\s]*define[^\(]*\([^\'"]*[\'"]+' . $define_name .
					'[\'"]+[^\'"]*(.*)\)[\s]*;|Uis', $language_file_source, $matches, PREG_SET_ORDER)) {
				
				foreach ($matches as $current_define) {
					// Do a basic test to make sure this define isn't commented out
					if (preg_match('|//[\s]*define\(|i', $current_define[0]) ||
							preg_match('|#[\s]*define\(|i', $current_define[0])) {
						// Define is commented out
						
						continue;
					}
					
					// Parse the define's value
					$define_value = trim($current_define[1]);
					
					// Remove the comma
					$define_value = trim(substr($define_value, 1, strlen($define_value) - 1));
					
					$define_value_info = $this->_parseDefineValue($define_value, $language_directory,
						$template_directory, $page_type, $lookup_embedded_define_values);
					
					// Add the info about the name of the define used and the path to file it was
					// found in
					$define_value_info['define_sources'][] = array(
						'define_name' => $define_name,
						'definitions_file_path' => $language_file_to_try
						);
					
					$define_value_info['define_sources'] = array_reverse($define_value_info['define_sources']);
					
					break 2;
				}
			}
		}
		
		if (!$file_found) {
			$define_value_info['errors'][] = 'Couldn\'t find any language define file for ' . $page_type;
		}
		
		return $define_value_info;
	}
	
	// }}}
	
	
	// {{{ _parseDefineValue()
	
	/**
	 * Attempts to parse the value for a define. Extracts any quoted strings and embedded defines from the value.
	 * Can handle single or double quotes and can handle any escaped quotes inside a quoted string.
	 *
	 * If instructed as such, will try to extract the value for any embedded define from the language file for the
	 * specified page name, or the language file for the language the defines, are using paying full attention to
	 * any override files that exist for the specific template.
	 *
	 * E.g. Can fully parse the value of the following define:
	 *
	 * define('DEFINE_BEING_PARSED', EMBEDDED_DEFINE_1 . 'String\'s Value' . EMBEDDED_DEFINE_2);
	 *
	 * @access  protected
	 * @param   string    $define_value         The value to be parsed.
	 * @param   string    $language_directory   The name of the directory for the language the define uses.
	 * @param   string    $template_directory   The name of the directory for the template for the language the
	 *                                          define uses.
	 * @param   string    $page_type            The Zen Cart page type the define is part of.
	 * @param   boolean   $lookup_embedded_define_values   Whether or not to look up the value for any embedded
	 *                                                     values.
	 * @return  array     An array which includes an array of the parts of the define's value that were parsed. If
	 *                    any errors occurred when parsing (either because a define's value used incorrect
	 *                    formatting or, when looking up values for an embedded define, no value could be found for
	 *                    an embedded define, a list of errors will also be included in the array.
	 */
	protected function _parseDefineValue($define_value, $language_directory, $template_directory, $page_type,
		$lookup_embedded_define_values)
	{
		// Some defines are built up of other defines. Array also holds the "parts" of the define identified
		$define_value_info = array(
			'value_parts' => array(),
			'define_sources' => array(),
			'errors' => array()
			);
		
		$define_value = trim($define_value);
		
		// Is the first part of the define another define or a quoted string?
		
		// Attempt to find an opening quote
		$opening_quote_pos = null;
		
		$single_quote_pos = strpos($define_value, "'");
		$double_quote_pos = strpos($define_value, '"');
		
		// Attempt to find a concatenation operator
		$dot_pos = strpos($define_value, '.');
		
		if ($single_quote_pos === false && $double_quote_pos === false && $dot_pos == false) {
			// Define's value is simply another define
			$define_name = $define_value;
			
			if ($lookup_embedded_define_values) {
				$embedded_define_value_info = $this->_getDefineValue($define_name, $language_directory,
					$template_directory, $page_type, true);
				
				$define_value_info = array_merge($define_value_info, $embedded_define_value_info);
				
			} else {
				$define_value_info['value_parts'][] = $define_name;
			}
			
			return $define_value_info;
		} 
		
		if ($single_quote_pos === false && $double_quote_pos !== false) {
			$quote_char = '"';
			
			$opening_quote_pos = $double_quote_pos;
			
		} else if ($single_quote_pos !== false && $double_quote_pos === false) {
			$quote_char = "'";
			
			$opening_quote_pos = $single_quote_pos;
			
		} else if ($single_quote_pos !== false && $double_quote_pos !== false) {
			if ($single_quote_pos < $double_quote_pos) {
				$quote_char = "'";
				
				$opening_quote_pos = $single_quote_pos;
				
			} else {
				$quote_char = '"';
				
				$opening_quote_pos = $double_quote_pos;
			}
		}
		
		// Is the [first part of the] define another define or a quoted string?
		if (is_null($opening_quote_pos) || $dot_pos < $opening_quote_pos) {
			// The [first part of the] define must be another define
			$define_name = trim(substr($define_value, 0, $dot_pos));
			
			if ($lookup_embedded_define_values) {
				$embedded_define_value_info = $this->_getDefineValue($define_name, $language_directory,
					$template_directory, $page_type, true);
				
				$define_value_info = array_merge($define_value_info, $embedded_define_value_info);
			} else {
				$define_value_info['value_parts'][] = $define_name;
			}
			
			$remaining_value_to_parse = substr($define_value, $dot_pos + 1);
			
			$further_define_value_parts = $this->_parseDefineValue($remaining_value_to_parse, $language_directory,
				$template_directory, $page_type, $lookup_embedded_define_values);
			
			foreach ($further_define_value_parts['value_parts'] as $value_part) {
				$define_value_info['value_parts'][] = $value_part;
			}
			
			foreach ($further_define_value_parts['define_sources'] as $define_source) {
				$define_value_info['define_sources'][] = $define_source;
			}
			
			foreach ($further_define_value_parts['errors'] as $error) {
				$define_value_info['errors'][] = $error;
			}
			
			return $define_value_info;
		}
		
		
		// The [first part of the] define is a quoted string
		if ($opening_quote_pos > 0) {
			// Extract the first part of the value which isn't enclosed in quotes (must be another define or series
			// of defines)
			$define_value_info['value_parts'][] = substr($define_value, 0, $opening_quote_pos);
		}
		
		if ($opening_quote_pos == (strlen($define_value) - 1)) {
			// Broken format for define's value
			$define_value_info['errors'][] = "Opening quote is at end of string!";
			
			return $define_value_info;
			
		} else {
			// Skip past opening quote
			$define_value = substr($define_value, $opening_quote_pos + 1,
				strlen($define_value) - ($opening_quote_pos + 1));
		}
		
		// Get matching quote
		$closing_quote_found = false;
		
		$remaining_value_to_parse = $define_value;
		
		$define_value_part = '';
		
		while (!$closing_quote_found) {
			$closing_quote_pos = strpos($remaining_value_to_parse, $quote_char);
			
			if ($closing_quote_pos === false) {
				// Closing quote character not found! Broken format for define's value!
				$define_value_info['errors'][] = "Closing quote not found!";
				
				return $define_value_info;
				
			} else if ($closing_quote_pos > 0) {
				// Is this the closing quote character or a quoted quote character?
				if (substr($remaining_value_to_parse, $closing_quote_pos, 1) == '\\') {
					// This isn't a closing quote character, add content to the define value part being built and
					// move on
					$define_value_part .= substr($remaining_value_to_parse, 0, $closing_quote_pos + 1);
					
					if ($closing_quote_pos + 1 == strlen($remaining_value_to_parse)) {
						// No more characters, broken format for define's value!
						$define_value_info['errors'][] = "Escaped quote found at end of string.";
						
						return $define_value_info;
						
					} else {
						$remaining_value_to_parse = substr($remaining_value_to_parse, $closing_quote_pos + 1);
					}
				} else {
					// Have found the closing quote
					$closing_quote_found = true;
					
					$define_value_part .= substr($remaining_value_to_parse, 0, $closing_quote_pos);
					
					if ($closing_quote_pos + 1 == strlen($remaining_value_to_parse)) {
						// No more characters
						$remaining_value_to_parse = '';
						
					} else {
						// Remove value found and closing quote from the remaining value to be parsed
						$remaining_value_to_parse = substr($remaining_value_to_parse, $closing_quote_pos + 1);
					}
				}
			} else {
				// Opening and closing quotes enclose an empty string
				$closing_quote_found = true;
				
				// Remove closing quote from the the remaining value to be parsed
				$remaining_value_to_parse = '';
			}
		}
		
		if ($define_value_part != '') {
			$define_value_info['value_parts'][] = $define_value_part;
		}
		
		$remaining_value_to_parse = trim($remaining_value_to_parse);
		
		if (strlen($remaining_value_to_parse) > 0) {
			// Get rid of any joining dots and continue to parse the rest of the value!
			$dot_pos = strpos($remaining_value_to_parse, '.');
			
			if ($dot_pos === false) {
				// Joining dot not found, format invalid
				$define_value_info['errors'][] =
					"Value not fully parsed but dot not found after previous quoted part found";
				
				return $define_value_info;
			}
			
			$remaining_value_to_parse = substr($remaining_value_to_parse, $dot_pos + 1,
				strlen($remaining_value_to_parse) - ($dot_pos + 1));
			
			$further_define_value_parts = $this->_parseDefineValue($remaining_value_to_parse, $language_directory,
				$template_directory, $page_type, $lookup_embedded_define_values);
			
			foreach ($further_define_value_parts['value_parts'] as $value_part) {
				$define_value_info['value_parts'][] = $value_part;
			}
			
			foreach ($further_define_value_parts['define_sources'] as $define_source) {
				$define_value_info['define_sources'][] = $define_source;
			}
			
			foreach ($further_define_value_parts['errors'] as $error) {
				$define_value_info['errors'][] = $error;
			}
		}
		
		return $define_value_info;
	}
	
	// }}}
}

// }}}
